package io.streamlayer.common.data.voximplant;

import android.content.Context;
import android.util.Log;

import com.voximplant.sdk.Voximplant;
import com.voximplant.sdk.client.ILogListener;
import com.voximplant.sdk.client.LogLevel;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Date;

import io.streamlayer.sdk.BuildConfig;

public class VoximplantFileLogger implements ILogListener {

    private static volatile VoximplantFileLogger mInstance;
    private static final String FILE_NAME = "vox.txt";
    private FileOutputStream mOutputStream;
    private final DateFormat mDateFormat = new SimpleDateFormat("yyyy/MM/dd HH:mm:ss");

    // can return null
    public static synchronized VoximplantFileLogger getInstance() {
        // set voximplant file logger only for staging builds
        if (!isLogsEnabled()) return null;
        if (mInstance == null) {
            synchronized (VoximplantFileLogger.class) {
                if (mInstance == null) {
                    mInstance = new VoximplantFileLogger();
                    Voximplant.setLogListener(mInstance);
                }
            }
        }
        return mInstance;
    }

    private VoximplantFileLogger() {
    }

    // enabled logs only on staging builds
    private static Boolean isLogsEnabled() {
        return BuildConfig.FLAVOR.equals("staging");
    }

    public void start(Context context) {
        if (!isLogsEnabled()) return;
        synchronized (this) {
            try {
                // create new file each time when logger initialized
                File logs = new File(context.getExternalFilesDir(null), "streamlayer_logs");
                if (!logs.exists()) logs.mkdirs();
                File logFile = new File(logs, FILE_NAME);
                if (logFile.exists()) {
                    logFile.delete();
                } else logFile.createNewFile();
                if (mOutputStream != null) { mOutputStream.close(); }
                mOutputStream = new FileOutputStream(logFile);
            } catch (Throwable e) {
                Log.e("VoximplantFileLogger", "FileLogger: failed to open file " + e);
            }
        }
    }

    public void stop() {
        if (!isLogsEnabled()) return;
        synchronized (this) {
            try {
                if (mOutputStream != null) {
                    mOutputStream.close();
                }
            } catch (IOException e) {
                Log.e("VoximplantFileLogger", "FileLogger: failed close output stream");
            } finally {
                mOutputStream = null;
            }
        }
    }

    @Override
    public void onLogMessage(LogLevel level, String logMessage) {
        if (!isLogsEnabled()) return;
        if (mOutputStream != null) {
            try {
                switch (level) {
                    case ERROR:
                        logMessage = mDateFormat.format(new Date()) + " ERROR:   " + logMessage + "\n";
                        break;
                    case WARNING:
                        logMessage = mDateFormat.format(new Date()) + " WARNING: " + logMessage + "\n";
                        break;
                    case INFO:
                        logMessage = mDateFormat.format(new Date()) + " INFO:    " + logMessage + "\n";
                        break;
                    case DEBUG:
                        logMessage = mDateFormat.format(new Date()) + " DEBUG:   " + logMessage + "\n";
                        break;
                    case VERBOSE:
                    default:
                        logMessage = mDateFormat.format(new Date()) + " VERBOSE: " + logMessage + "\n";
                        break;
                }
                mOutputStream.write(logMessage.getBytes());
            } catch (IOException e) {
                Log.e("VoximplantFileLogger", "FileLogger: failed to write log message");
            }
        }
    }
}