package io.streamlayer.sdk.calls.call;

import android.content.Context;
import android.util.Log;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;

import io.streamlayer.sdk.BuildConfig;

public class WatchPartyFileLogger {

    private static volatile WatchPartyFileLogger mInstance;
    private static final String FILE_NAME = "wp.txt";
    private FileOutputStream mOutputStream;

    // can return null
    public static synchronized WatchPartyFileLogger getInstance() {
        // set voximplant file logger only for staging builds
        if (!isLogsEnabled()) return null;
        if (mInstance == null) {
            synchronized (WatchPartyFileLogger.class) {
                if (mInstance == null) {
                    mInstance = new WatchPartyFileLogger();
                }
            }
        }
        return mInstance;
    }

    private WatchPartyFileLogger() {
    }

    // enabled logs only on staging builds
    private static Boolean isLogsEnabled() {
        return BuildConfig.FLAVOR.equals("staging");
    }

    public void start(Context context) {
        if (!isLogsEnabled()) return;
        synchronized (this) {
            try {
                // create new file each time when logger initialized
                File logs = new File(context.getExternalFilesDir(null), "streamlayer_logs");
                if (!logs.exists()) logs.mkdirs();
                File logFile = new File(logs, FILE_NAME);
                if (logFile.exists()) {
                    logFile.delete();
                } else logFile.createNewFile();
                if (mOutputStream != null) { mOutputStream.close(); }
                mOutputStream = new FileOutputStream(logFile);
            } catch (Throwable e) {
                Log.e("WatchPartyFileLogger", "FileLogger: failed to open file " + e);
            }
        }
    }

    public void stop() {
        if (!isLogsEnabled()) return;
        synchronized (this) {
            try {
                if (mOutputStream != null) {
                    mOutputStream.close();
                }
            } catch (IOException e) {
                Log.e("WatchPartyFileLogger", "FileLogger: failed close output stream");
            } finally {
                mOutputStream = null;
            }
        }
    }

    public void onLogMessage(String logMessage) {
        if (!isLogsEnabled()) return;
        if (mOutputStream != null) {
            try {
                mOutputStream.write(logMessage.getBytes());
            } catch (IOException e) {
                Log.e("WatchPartyFileLogger", "FileLogger: failed to write log message");
            }
        }
    }
}