package io.taig.taigless.ws

import cats.effect.kernel.Deferred
import cats.effect.{Resource, Sync}
import cats.syntax.all._
import fs2.Stream
import fs2.concurrent.Topic
import org.java_websocket.client.{WebSocketClient => JWebSocketClient}

abstract class Channel[F[_], A] {

  /** Send a message via the WebSocket connection and observe the incoming messages right afterwards
    *
    * This method ensures that the subscription to the incoming WebSocket messages is ready before sending the
    * given `value`.
    */
  def send(value: String): Stream[F, A]

  def sendAndForget(value: String): F[Unit]

  def subscription: Stream[F, A]

  def awaitSubscription: Resource[F, Stream[F, A]]
}

object Channel {
  def fromWebSocketClient[F[_], A](
      topic: Topic[F, Either[Throwable, Message[A]]],
      interruption: Deferred[F, Throwable]
  )(client: JWebSocketClient, maxQueued: Int)(implicit F: Sync[F]): Channel[F, A] = new Channel[F, A] {
    override def send(value: String): Stream[F, A] =
      Stream
        .resource(topic.subscribeAwait(maxQueued))
        .flatMap(messages => Stream.exec(F.blocking(client.send(value)).void) ++ messages.rethrow)
        .collect { case Message.Data(value) => value }
        .interruptWhen(interruption.get.map(_.asLeft[Unit]))

    override def sendAndForget(value: String): F[Unit] = F.blocking(client.send(value)).void

    override val subscription: Stream[F, A] =
      topic
        .subscribe(maxQueued)
        .rethrow
        .collect { case Message.Data(value) => value }
        .interruptWhen(interruption.get.map(_.asLeft[Unit]))

    override val awaitSubscription: Resource[F, Stream[F, A]] =
      topic.subscribeAwait(maxQueued).map { subscription =>
        subscription.rethrow
          .collect { case Message.Data(value) => value }
          .interruptWhen(interruption.get.map(_.asLeft[Unit]))
      }
  }
}
