package io.taig.taigless.ws

import cats.effect.std.Dispatcher
import cats.effect.syntax.all._
import cats.effect.{Async, Deferred, Resource}
import cats.syntax.all._
import fs2.concurrent.Topic
import fs2.Stream

import java.net.URI
import scala.concurrent.duration._

object WebSocket {
  def apply[F[_]: Async, A](
      topic: Topic[F, Either[Throwable, Message[String]]],
      interruption: Deferred[F, Throwable],
      client: WebSocketMessageHandler[F, A]
  )(maxQueued: Int, connectionTimeout: FiniteDuration): Resource[F, Channel[F, String]] =
    topic
      .subscribeAwait(maxQueued = Int.MaxValue)
      .flatMap { subscription =>
        val channel = Channel.fromWebSocketClient(topic, interruption)(client, maxQueued)

        client.start *> Resource.eval {
          subscription.rethrow
            .flatMap {
              case Message.Open    => Stream.emit(channel)
              case Message.Data(_) => Stream.raiseError(new IllegalStateException("Received data before open message"))
              case Message.Close(code, reason) =>
                val message = s"Received socket close while waiting for open: $code $reason"
                Stream.raiseError(new IllegalStateException(message))
            }
            .interruptWhen(interruption.get.map(_.asLeft[Unit]))
            .head
            .compile
            .lastOrError
            .timeout(connectionTimeout)
        }
      }
      .onFinalize(topic.close.void)

  def default[F[_]: Async](
      uri: URI,
      maxQueued: Int,
      connectionTimeout: FiniteDuration = 10.seconds,
      enqueueTimeout: FiniteDuration = 1.second
  ): Resource[F, Channel[F, String]] =
    for {
      dispatcher <- Dispatcher[F]
      topic <- Resource.make(Topic[F, Either[Throwable, Message[String]]])(_.close.void)
      interruption <- Resource.eval(Deferred[F, Throwable])
      client = new StringWebSocketMessageHandler[F](topic, interruption, dispatcher)(uri, enqueueTimeout)
      websocket <- WebSocket(topic, interruption, client)(maxQueued, connectionTimeout)
    } yield websocket
}
