/*
 * Decompiled with CFR 0.152.
 */
package io.trino.memory.context;

import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.SettableFuture;
import io.airlift.units.DataSize;
import io.trino.memory.context.AggregatedMemoryContext;
import io.trino.memory.context.LocalMemoryContext;
import io.trino.memory.context.MemoryAllocationValidator;
import io.trino.memory.context.MemoryReservationHandler;
import io.trino.memory.context.ValidatingAggregateContext;
import org.assertj.core.api.Assertions;
import org.testng.Assert;
import org.testng.annotations.Test;

public class TestMemoryContexts {
    private static final ListenableFuture<Void> NOT_BLOCKED = Futures.immediateVoidFuture();
    private static final long GUARANTEED_MEMORY = DataSize.of((long)1L, (DataSize.Unit)DataSize.Unit.MEGABYTE).toBytes();

    @Test
    public void testLocalMemoryContextClose() {
        TestMemoryReservationHandler reservationHandler = new TestMemoryReservationHandler(1000L);
        AggregatedMemoryContext aggregateContext = AggregatedMemoryContext.newRootAggregatedMemoryContext((MemoryReservationHandler)reservationHandler, (long)GUARANTEED_MEMORY);
        LocalMemoryContext localContext = aggregateContext.newLocalMemoryContext("test");
        localContext.setBytes(100L);
        Assert.assertEquals((long)localContext.getBytes(), (long)100L);
        Assert.assertEquals((long)aggregateContext.getBytes(), (long)100L);
        Assert.assertEquals((long)reservationHandler.getReservation(), (long)100L);
        localContext.close();
        Assert.assertEquals((long)localContext.getBytes(), (long)0L);
        Assert.assertEquals((long)aggregateContext.getBytes(), (long)0L);
        Assert.assertEquals((long)reservationHandler.getReservation(), (long)0L);
    }

    @Test
    public void testMemoryContexts() {
        TestMemoryReservationHandler reservationHandler = new TestMemoryReservationHandler(1000L);
        AggregatedMemoryContext aggregateContext = AggregatedMemoryContext.newRootAggregatedMemoryContext((MemoryReservationHandler)reservationHandler, (long)GUARANTEED_MEMORY);
        LocalMemoryContext localContext = aggregateContext.newLocalMemoryContext("test");
        Assert.assertEquals((Object)localContext.setBytes(10L), NOT_BLOCKED);
        Assert.assertEquals((long)localContext.getBytes(), (long)10L);
        Assert.assertEquals((long)aggregateContext.getBytes(), (long)10L);
        Assert.assertEquals((long)reservationHandler.getReservation(), (long)aggregateContext.getBytes());
        LocalMemoryContext secondLocalContext = aggregateContext.newLocalMemoryContext("test");
        Assert.assertEquals((Object)secondLocalContext.setBytes(20L), NOT_BLOCKED);
        Assert.assertEquals((long)secondLocalContext.getBytes(), (long)20L);
        Assert.assertEquals((long)aggregateContext.getBytes(), (long)30L);
        Assert.assertEquals((long)localContext.getBytes(), (long)10L);
        Assert.assertEquals((long)reservationHandler.getReservation(), (long)aggregateContext.getBytes());
        aggregateContext.close();
        Assert.assertEquals((long)aggregateContext.getBytes(), (long)0L);
        Assert.assertEquals((long)reservationHandler.getReservation(), (long)0L);
    }

    @Test
    public void testTryReserve() {
        TestMemoryReservationHandler reservationHandler = new TestMemoryReservationHandler(1000L);
        AggregatedMemoryContext parentContext = AggregatedMemoryContext.newRootAggregatedMemoryContext((MemoryReservationHandler)reservationHandler, (long)GUARANTEED_MEMORY);
        AggregatedMemoryContext aggregateContext1 = parentContext.newAggregatedMemoryContext();
        AggregatedMemoryContext aggregateContext2 = parentContext.newAggregatedMemoryContext();
        LocalMemoryContext childContext1 = aggregateContext1.newLocalMemoryContext("test");
        Assert.assertTrue((boolean)childContext1.trySetBytes(500L));
        Assert.assertTrue((boolean)childContext1.trySetBytes(1000L));
        Assert.assertFalse((boolean)childContext1.trySetBytes(1001L));
        Assert.assertEquals((long)reservationHandler.getReservation(), (long)parentContext.getBytes());
        aggregateContext1.close();
        aggregateContext2.close();
        parentContext.close();
        Assert.assertEquals((long)aggregateContext1.getBytes(), (long)0L);
        Assert.assertEquals((long)aggregateContext2.getBytes(), (long)0L);
        Assert.assertEquals((long)parentContext.getBytes(), (long)0L);
        Assert.assertEquals((long)reservationHandler.getReservation(), (long)0L);
    }

    @Test
    public void testHierarchicalMemoryContexts() {
        TestMemoryReservationHandler reservationHandler = new TestMemoryReservationHandler(1000L);
        AggregatedMemoryContext parentContext = AggregatedMemoryContext.newRootAggregatedMemoryContext((MemoryReservationHandler)reservationHandler, (long)GUARANTEED_MEMORY);
        AggregatedMemoryContext aggregateContext1 = parentContext.newAggregatedMemoryContext();
        AggregatedMemoryContext aggregateContext2 = parentContext.newAggregatedMemoryContext();
        LocalMemoryContext childContext1 = aggregateContext1.newLocalMemoryContext("test");
        LocalMemoryContext childContext2 = aggregateContext2.newLocalMemoryContext("test");
        Assert.assertEquals((Object)childContext1.setBytes(1L), NOT_BLOCKED);
        Assert.assertEquals((Object)childContext2.setBytes(1L), NOT_BLOCKED);
        Assert.assertEquals((long)aggregateContext1.getBytes(), (long)1L);
        Assert.assertEquals((long)aggregateContext2.getBytes(), (long)1L);
        Assert.assertEquals((long)parentContext.getBytes(), (long)(aggregateContext1.getBytes() + aggregateContext2.getBytes()));
        Assert.assertEquals((long)reservationHandler.getReservation(), (long)parentContext.getBytes());
    }

    @Test
    public void testGuaranteedMemoryDoesntBlock() {
        long maxMemory = 2L * GUARANTEED_MEMORY;
        TestMemoryReservationHandler reservationHandler = new TestMemoryReservationHandler(maxMemory);
        AggregatedMemoryContext parentContext = AggregatedMemoryContext.newRootAggregatedMemoryContext((MemoryReservationHandler)reservationHandler, (long)GUARANTEED_MEMORY);
        LocalMemoryContext childContext = parentContext.newLocalMemoryContext("test");
        reservationHandler.exhaustMemory();
        Assert.assertEquals((long)reservationHandler.getReservation(), (long)maxMemory);
        Assert.assertEquals((Object)childContext.setBytes(1000L), NOT_BLOCKED);
        Assert.assertNotEquals((Object)childContext.setBytes(GUARANTEED_MEMORY + 1L), NOT_BLOCKED);
        childContext.close();
        parentContext.close();
        Assert.assertEquals((long)childContext.getBytes(), (long)0L);
        Assert.assertEquals((long)parentContext.getBytes(), (long)0L);
        Assert.assertEquals((long)reservationHandler.getReservation(), (long)maxMemory);
    }

    @Test(expectedExceptions={IllegalStateException.class}, expectedExceptionsMessageRegExp="SimpleLocalMemoryContext is already closed")
    public void testClosedLocalMemoryContext() {
        AggregatedMemoryContext aggregateContext = AggregatedMemoryContext.newSimpleAggregatedMemoryContext();
        LocalMemoryContext localContext = aggregateContext.newLocalMemoryContext("test");
        localContext.close();
        localContext.setBytes(100L);
    }

    @Test(expectedExceptions={IllegalStateException.class}, expectedExceptionsMessageRegExp="SimpleAggregatedMemoryContext is already closed")
    public void testClosedAggregateMemoryContext() {
        AggregatedMemoryContext aggregateContext = AggregatedMemoryContext.newSimpleAggregatedMemoryContext();
        LocalMemoryContext localContext = aggregateContext.newLocalMemoryContext("test");
        aggregateContext.close();
        localContext.setBytes(100L);
    }

    @Test
    public void testValidatingAggregateContext() {
        TestMemoryReservationHandler reservationHandler = new TestMemoryReservationHandler(1000L, true);
        AggregatedMemoryContext rootContext = AggregatedMemoryContext.newRootAggregatedMemoryContext((MemoryReservationHandler)reservationHandler, (long)GUARANTEED_MEMORY);
        ValidatingAggregateContext childContext = new ValidatingAggregateContext(rootContext, (MemoryAllocationValidator)new TestAllocationValidator(500L));
        LocalMemoryContext localContext = childContext.newLocalMemoryContext("test");
        Assert.assertEquals((Object)localContext.setBytes(500L), NOT_BLOCKED);
        Assert.assertEquals((long)localContext.getBytes(), (long)500L);
        Assert.assertEquals((long)rootContext.getBytes(), (long)500L);
        Assert.assertEquals((long)reservationHandler.getReservation(), (long)rootContext.getBytes());
        Assertions.assertThatThrownBy(() -> localContext.setBytes(501L)).hasMessage("limit exceeded");
        Assert.assertEquals((long)localContext.getBytes(), (long)500L);
        Assert.assertEquals((long)rootContext.getBytes(), (long)500L);
        Assert.assertEquals((long)reservationHandler.getReservation(), (long)rootContext.getBytes());
        Assert.assertFalse((boolean)localContext.trySetBytes(501L));
        Assert.assertEquals((long)localContext.getBytes(), (long)500L);
        Assert.assertEquals((long)rootContext.getBytes(), (long)500L);
        Assert.assertEquals((long)reservationHandler.getReservation(), (long)rootContext.getBytes());
        Assert.assertEquals((Object)localContext.setBytes(400L), NOT_BLOCKED);
        Assert.assertEquals((long)localContext.getBytes(), (long)400L);
        Assert.assertEquals((long)rootContext.getBytes(), (long)400L);
        Assert.assertEquals((long)reservationHandler.getReservation(), (long)rootContext.getBytes());
        Assert.assertTrue((boolean)localContext.trySetBytes(300L));
        Assert.assertEquals((long)localContext.getBytes(), (long)300L);
        Assert.assertEquals((long)rootContext.getBytes(), (long)300L);
        Assert.assertEquals((long)reservationHandler.getReservation(), (long)rootContext.getBytes());
        LocalMemoryContext anotherLocalContext = rootContext.newLocalMemoryContext("another");
        Assert.assertEquals((Object)anotherLocalContext.setBytes(650L), NOT_BLOCKED);
        Assert.assertEquals((long)localContext.getBytes(), (long)300L);
        Assert.assertEquals((long)anotherLocalContext.getBytes(), (long)650L);
        Assert.assertEquals((long)rootContext.getBytes(), (long)950L);
        Assert.assertEquals((long)reservationHandler.getReservation(), (long)rootContext.getBytes());
        Assertions.assertThatThrownBy(() -> localContext.setBytes(400L)).hasMessage("out of memory");
        Assert.assertEquals((long)localContext.getBytes(), (long)300L);
        Assert.assertEquals((long)anotherLocalContext.getBytes(), (long)650L);
        Assert.assertEquals((long)rootContext.getBytes(), (long)950L);
        Assert.assertEquals((long)reservationHandler.getReservation(), (long)rootContext.getBytes());
        Assert.assertFalse((boolean)localContext.trySetBytes(400L));
        Assert.assertEquals((long)localContext.getBytes(), (long)300L);
        Assert.assertEquals((long)anotherLocalContext.getBytes(), (long)650L);
        Assert.assertEquals((long)rootContext.getBytes(), (long)950L);
        Assert.assertEquals((long)reservationHandler.getReservation(), (long)rootContext.getBytes());
        Assert.assertEquals((Object)anotherLocalContext.setBytes(499L), NOT_BLOCKED);
        Assert.assertEquals((Object)localContext.setBytes(400L), NOT_BLOCKED);
        Assert.assertEquals((long)localContext.getBytes(), (long)400L);
        Assert.assertEquals((long)anotherLocalContext.getBytes(), (long)499L);
        Assert.assertEquals((long)rootContext.getBytes(), (long)899L);
        Assert.assertEquals((long)reservationHandler.getReservation(), (long)rootContext.getBytes());
        Assert.assertEquals((Object)localContext.setBytes(500L), NOT_BLOCKED);
        Assert.assertEquals((long)localContext.getBytes(), (long)500L);
        Assert.assertEquals((long)anotherLocalContext.getBytes(), (long)499L);
        Assert.assertEquals((long)rootContext.getBytes(), (long)999L);
        Assert.assertEquals((long)reservationHandler.getReservation(), (long)rootContext.getBytes());
    }

    private static class TestMemoryReservationHandler
    implements MemoryReservationHandler {
        private long reservation;
        private final long maxMemory;
        private final boolean throwWhenExceeded;
        private SettableFuture<Void> future;

        public TestMemoryReservationHandler(long maxMemory) {
            this(maxMemory, false);
        }

        public TestMemoryReservationHandler(long maxMemory, boolean throwWhenExceeded) {
            this.maxMemory = maxMemory;
            this.throwWhenExceeded = throwWhenExceeded;
        }

        public long getReservation() {
            return this.reservation;
        }

        public ListenableFuture<Void> reserveMemory(String allocationTag, long delta) {
            if (delta > 0L && this.reservation + delta > this.maxMemory && this.throwWhenExceeded) {
                throw new IllegalStateException("out of memory");
            }
            this.reservation += delta;
            if (delta >= 0L) {
                if (this.reservation >= this.maxMemory) {
                    this.future = SettableFuture.create();
                    return this.future;
                }
            } else if (this.reservation < this.maxMemory && this.future != null) {
                this.future.set(null);
            }
            return NOT_BLOCKED;
        }

        public boolean tryReserveMemory(String allocationTag, long delta) {
            if (this.reservation + delta > this.maxMemory) {
                return false;
            }
            this.reservation += delta;
            return true;
        }

        public void exhaustMemory() {
            this.reservation = this.maxMemory;
        }
    }

    private static class TestAllocationValidator
    implements MemoryAllocationValidator {
        private final long limit;
        private long reserved;

        public TestAllocationValidator(long limit) {
            this.limit = limit;
        }

        public void reserveMemory(String allocationTag, long delta) {
            if (this.reserved + delta > this.limit) {
                throw new IllegalArgumentException("limit exceeded");
            }
            this.reserved += delta;
        }

        public boolean tryReserveMemory(String allocationTag, long delta) {
            if (this.reserved + delta > this.limit) {
                return false;
            }
            this.reserved += delta;
            return true;
        }
    }
}

