/*
 * Decompiled with CFR 0.152.
 */
package io.trino.plugin.sqlserver;

import com.google.common.base.Throwables;
import io.airlift.log.Logger;
import io.airlift.testing.Closeables;
import io.trino.testing.containers.TestContainers;
import io.trino.testing.sql.SqlExecutor;
import java.io.Closeable;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.sql.Connection;
import java.sql.SQLException;
import java.sql.Statement;
import java.time.Duration;
import java.time.temporal.ChronoUnit;
import java.util.Locale;
import java.util.Objects;
import java.util.UUID;
import java.util.function.BiConsumer;
import net.jodah.failsafe.Failsafe;
import net.jodah.failsafe.Policy;
import net.jodah.failsafe.PolicyListeners;
import net.jodah.failsafe.RetryPolicy;
import net.jodah.failsafe.Timeout;
import org.testcontainers.containers.GenericContainer;
import org.testcontainers.containers.MSSQLServerContainer;
import org.testcontainers.utility.DockerImageName;

public final class TestingSqlServer
implements AutoCloseable {
    private static final Logger log = Logger.get(TestingSqlServer.class);
    public static final BiConsumer<SqlExecutor, String> DEFAULT_DATABASE_SETUP = (executor, databaseName) -> {
        executor.execute(String.format("ALTER DATABASE %s SET ALLOW_SNAPSHOT_ISOLATION ON", databaseName));
        executor.execute(String.format("ALTER DATABASE %s SET READ_COMMITTED_SNAPSHOT ON", databaseName));
    };
    private static final RetryPolicy<InitializedState> CONTAINER_RETRY_POLICY = ((RetryPolicy)new RetryPolicy().withBackoff(1L, 5L, ChronoUnit.SECONDS).withMaxRetries(5).handleIf(throwable -> Throwables.getCausalChain((Throwable)throwable).stream().anyMatch(SQLException.class::isInstance))).onRetry(event -> log.warn("Query failed on attempt %s, will retry. Exception: %s", new Object[]{event.getAttemptCount(), event.getLastFailure().getMessage()}));
    private static final DockerImageName DOCKER_IMAGE_NAME = DockerImageName.parse((String)"mcr.microsoft.com/mssql/server:2017-CU13");
    private final MSSQLServerContainer<?> container;
    private final String databaseName;
    private final Closeable cleanup;

    public TestingSqlServer() {
        this(DEFAULT_DATABASE_SETUP);
    }

    public TestingSqlServer(BiConsumer<SqlExecutor, String> databaseSetUp) {
        InitializedState initializedState = (InitializedState)Failsafe.with((Policy[])new PolicyListeners[]{CONTAINER_RETRY_POLICY, Timeout.of((Duration)Duration.ofMinutes(5L))}).get(() -> TestingSqlServer.createContainer(databaseSetUp));
        this.container = initializedState.container;
        this.databaseName = initializedState.databaseName;
        this.cleanup = initializedState.cleanup;
        this.container.withUrlParam("database", this.databaseName);
    }

    private static InitializedState createContainer(BiConsumer<SqlExecutor, String> databaseSetUp) {
        String databaseName = "database_" + UUID.randomUUID().toString().replace("-", "");
        MSSQLServerContainer container = new MSSQLServerContainer(DOCKER_IMAGE_NAME){

            public String getUsername() {
                return super.getUsername().toLowerCase(Locale.ENGLISH);
            }
        };
        container.addEnv("ACCEPT_EULA", "yes");
        container.addEnv("MSSQL_COLLATION", "Latin1_General_CS_AS");
        Closeable cleanup = TestContainers.startOrReuse((GenericContainer)container);
        try {
            TestingSqlServer.setUpDatabase(TestingSqlServer.sqlExecutorForContainer(container), databaseName, databaseSetUp);
        }
        catch (Exception e) {
            Closeables.closeAllSuppress((Throwable)e, (AutoCloseable[])new AutoCloseable[]{cleanup});
            throw e;
        }
        return new InitializedState(container, databaseName, cleanup);
    }

    private static void setUpDatabase(SqlExecutor executor, String databaseName, BiConsumer<SqlExecutor, String> databaseSetUp) {
        executor.execute("CREATE DATABASE " + databaseName);
        databaseSetUp.accept(executor, databaseName);
    }

    public String getDatabaseName() {
        return this.databaseName;
    }

    public void execute(String sql) {
        TestingSqlServer.sqlExecutorForContainer(this.container).execute(sql);
    }

    private static SqlExecutor sqlExecutorForContainer(MSSQLServerContainer<?> container) {
        Objects.requireNonNull(container, "container is null");
        return sql -> {
            try (Connection connection = container.createConnection("");
                 Statement statement = connection.createStatement();){
                statement.execute(sql);
            }
            catch (Exception e) {
                throw new RuntimeException("Failed to execute statement: " + sql, e);
            }
        };
    }

    public Connection createConnection() throws SQLException {
        return this.container.createConnection("");
    }

    public String getUsername() {
        return this.container.getUsername();
    }

    public String getPassword() {
        return this.container.getPassword();
    }

    public String getJdbcUrl() {
        return this.container.getJdbcUrl();
    }

    @Override
    public void close() {
        try {
            this.cleanup.close();
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    private static class InitializedState {
        private final MSSQLServerContainer<?> container;
        private final String databaseName;
        private final Closeable cleanup;

        public InitializedState(MSSQLServerContainer<?> container, String databaseName, Closeable cleanup) {
            this.container = container;
            this.databaseName = databaseName;
            this.cleanup = cleanup;
        }
    }
}

