package io.xiaper.restkefu.controller.v1;

import io.xiaper.jpa.constant.StatusConsts;
import io.xiaper.jpa.constant.TypeConsts;
import io.xiaper.jpa.model.LeaveMessage;
import io.xiaper.jpa.model.User;
import io.xiaper.jpa.model.WorkGroup;
import io.xiaper.jpa.repository.LeaveMessageRepository;
import io.xiaper.jpa.repository.UserRepository;
import io.xiaper.jpa.repository.WorkGroupRepository;
import io.xiaper.jpa.util.JpaUtil;
import io.xiaper.jpa.util.JsonResult;
import io.xiaper.rest.controller.v1.BaseController;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.web.bind.annotation.*;

import java.security.Principal;
import java.util.Map;
import java.util.Optional;

/**
 * 留言
 *
 * @author xiaper.io
 */
@RestController
@RequestMapping("/api/leavemsg")
public class LeaveMessageController extends BaseController {

    @Autowired
    UserRepository userRepository;

    @Autowired
    LeaveMessageRepository leaveMessageRepository;

    @Autowired
    WorkGroupRepository workGroupRepository;

    /**
     * 分页获取留言记录
     *
     * FIXME: 管理员账号下面的所有客服账号均可见
     *
     * @param principal principal
     * @param page page
     * @param size size
     * @param client client
     * @return json
     */
    @GetMapping("/get")
    public JsonResult get(Principal principal,
                          @RequestParam(value = "page") int page,
                          @RequestParam(value = "size") int size,
                          @RequestParam(value = "client") String client) {

        JsonResult jsonResult = new JsonResult();

        if (principal != null) {

            //
            Optional<User> adminOptional = userRepository.findByUsername(principal.getName());
            if (adminOptional.isPresent()) {

                // 分页查询
                Pageable pageable = PageRequest.of(page, size, Sort.Direction.DESC, "id");

                Page<LeaveMessage> leaveMessagePage;

                if (adminOptional.get().isAdmin()) {
                    // 管理员
                    leaveMessagePage = leaveMessageRepository.findBySubDomain(adminOptional.get().getSubDomain(), pageable);
                } else if (adminOptional.get().isWorkGroupAdmin()) {
                    // FIXME: 多个工作组
                    // 客服组长
                    WorkGroup workGroup = (WorkGroup) adminOptional.get().getWorkGroups().toArray()[0];
                    leaveMessagePage = leaveMessageRepository.findByWorkGroup(workGroup, pageable);
                } else {
                    // 客服账号
                    leaveMessagePage = leaveMessageRepository.findByAgent(adminOptional.get(), pageable);
                }

                //
                jsonResult.setMessage("获取留言成功");
                jsonResult.setStatus_code(200);
                jsonResult.setData(leaveMessagePage);

            } else {

                jsonResult.setMessage("管理员用户不存在");
                jsonResult.setStatus_code(-2);
                jsonResult.setData(false);
            }

        } else {

            jsonResult.setMessage("access token invalid");
            jsonResult.setStatus_code(-1);
            jsonResult.setData(false);
        }

        return jsonResult;
    }


    /**
     * 分页获取未被领取的留言
     *
     * @param principal principal
     * @param page page
     * @param size size
     * @param client client
     * @return json
     */
    @GetMapping("/unclaimed")
    public JsonResult unclaimed(Principal principal,
                                @RequestParam(value = "page") int page,
                                @RequestParam(value = "size") int size,
                                @RequestParam(value = "client") String client) {

        JsonResult jsonResult = new JsonResult();

        if (principal != null) {

            //
            Optional<User> agentOptional = userRepository.findByUsername(principal.getName());
            if (agentOptional.isPresent()) {

                // TODO: 是否根据工作组区分？

                // 分页查询
                Pageable pageable = PageRequest.of(page, size, Sort.Direction.DESC, "id");

                Page<LeaveMessage> leaveMessagePage = leaveMessageRepository.findBySubDomainAndStatus(agentOptional.get().getSubDomain(),
                        StatusConsts.LEAVE_MESSAGE_STATUS_UNCLAIMED, pageable);

                // 返回结果
                jsonResult.setMessage("获取未被领取留言成功");
                jsonResult.setStatus_code(200);
                jsonResult.setData(leaveMessagePage);

            } else {

                // 返回结果
                jsonResult.setMessage("获取未被领取留言失败-当前用户不存在");
                jsonResult.setStatus_code(-1);
                jsonResult.setData(false);
            }

        } else {

            jsonResult.setMessage("access token invalid");
            jsonResult.setStatus_code(-1);
            jsonResult.setData(false);
        }

        return jsonResult;
    }

    /**
     * 分页获取已经被领取的留言
     *
     * @param principal principal
     * @param page page
     * @param size size
     * @param client client
     * @return json
     */
    @GetMapping("/claimed")
    public JsonResult claimed(Principal principal,
                              @RequestParam(value = "page") int page,
                              @RequestParam(value = "size") int size,
                              @RequestParam(value = "client") String client) {

        JsonResult jsonResult = new JsonResult();

        if (principal != null) {

            //
            Optional<User> agentOptional = userRepository.findByUsername(principal.getName());
            if (agentOptional.isPresent()) {

                // 分页查询
                Pageable pageable = PageRequest.of(page, size, Sort.Direction.DESC, "id");

                Page<LeaveMessage> leaveMessagePage = leaveMessageRepository.findByClaimer(agentOptional.get(), pageable);

                // 返回结果
                jsonResult.setMessage("获取已被领取留言成功");
                jsonResult.setStatus_code(200);
                jsonResult.setData(leaveMessagePage);

            } else {

                // 返回结果
                jsonResult.setMessage("获取已被领取留言失败-当前用户不存在");
                jsonResult.setStatus_code(-1);
                jsonResult.setData(false);
            }

        } else {

            jsonResult.setMessage("access token invalid");
            jsonResult.setStatus_code(-1);
            jsonResult.setData(false);
        }

        return jsonResult;
    }


    /**
     * 领取留言
     *
     * @param principal principal
     * @param map map
     * @return json
     */
    @PostMapping("/claim")
    @ResponseBody
    public JsonResult claim(Principal principal, @RequestBody Map map) {

        String lid = (String) map.get("lid");

        JsonResult jsonResult = new JsonResult();

        if (principal != null) {

            //
            Optional<User> agentOptional = userRepository.findByUsername(principal.getName());
            if (agentOptional.isPresent()) {

                //
                Optional<LeaveMessage> leaveMessageOptional = leaveMessageRepository.findByLid(lid);
                if (leaveMessageOptional.isPresent()) {

                    //
                    leaveMessageOptional.get().setStatus(StatusConsts.LEAVE_MESSAGE_STATUS_CLAIMED);
                    leaveMessageOptional.get().setClaimer(agentOptional.get());
                    leaveMessageRepository.save(leaveMessageOptional.get());

                    // 返回结果
                    jsonResult.setMessage("领取成功");
                    jsonResult.setStatus_code(200);
                    jsonResult.setData(lid);

                } else {

                    // 返回结果
                    jsonResult.setMessage("领取留言失败-留言不存在");
                    jsonResult.setStatus_code(-2);
                    jsonResult.setData(false);
                }

            } else {

                // 返回结果
                jsonResult.setMessage("领取留言失败-当前用户不存在");
                jsonResult.setStatus_code(-1);
                jsonResult.setData(false);
            }

        } else {

            jsonResult.setMessage("access token invalid");
            jsonResult.setStatus_code(-1);
            jsonResult.setData(false);
        }

        return jsonResult;
    }


    /**
     * 小结
     *
     * @param principal principal
     * @param map map
     * @return json
     */
    @PostMapping("/summarize")
    @ResponseBody
    public JsonResult summarize(Principal principal, @RequestBody Map map) {

        String lid = (String) map.get("lid");
        String summary = (String) map.get("summary");

        JsonResult jsonResult = new JsonResult();

        if (principal != null) {

            //
            Optional<User> agentOptional = userRepository.findByUsername(principal.getName());
            if (agentOptional.isPresent()) {

                //
                Optional<LeaveMessage> leaveMessageOptional = leaveMessageRepository.findByLid(lid);
                if (leaveMessageOptional.isPresent()) {

                    //
                    leaveMessageOptional.get().setStatus(StatusConsts.LEAVE_MESSAGE_STATUS_SUMMARIZED);
                    leaveMessageOptional.get().setSummary(summary);
                    leaveMessageRepository.save(leaveMessageOptional.get());

                    // 返回结果
                    jsonResult.setMessage("小结成功");
                    jsonResult.setStatus_code(200);
                    jsonResult.setData(lid);

                } else {

                    // 返回结果
                    jsonResult.setMessage("领取留言失败-留言不存在");
                    jsonResult.setStatus_code(-2);
                    jsonResult.setData(false);
                }


            } else {

                // 返回结果
                jsonResult.setMessage("留言小结失败-当前用户不存在");
                jsonResult.setStatus_code(-1);
                jsonResult.setData(false);
            }


        } else {

            jsonResult.setMessage("access token invalid");
            jsonResult.setStatus_code(-1);
            jsonResult.setData(false);
        }

        return jsonResult;
    }


    /**
     * 保存留言
     *
     * @param principal principal
     * @param map map
     * @return json
     */
    @PostMapping("/save")
    @ResponseBody
    public JsonResult save(Principal principal, @RequestBody Map map) {

        String wid = (String) map.get("wid");
        String aid = (String) map.get("aid");
        String type = (String) map.get("type");
        //
        String mobile = (String) map.get("mobile");
        String email = (String) map.get("email");
        String content = (String) map.get("content");
        String client = (String) map.get("client");

        JsonResult jsonResult = new JsonResult();

        // 检查内容是否为空
        if (null == mobile || mobile.trim().length() == 0
            || null == content || content.trim().length() == 0) {

            jsonResult.setMessage("留言失败-手机号和留言内容不能为空");
            jsonResult.setStatus_code(-4);
            jsonResult.setData(false);

            return jsonResult;
        }

        //
        if (principal != null) {

            // 访客账号
            Optional<User> visitorOptional = userRepository.findByUsername(principal.getName());
            if (visitorOptional.isPresent()) {

                // 保存留言
                LeaveMessage leaveMessage = new LeaveMessage();
                leaveMessage.setLid(JpaUtil.randomId());
                leaveMessage.setMobile(mobile);
                leaveMessage.setEmail(email);
                leaveMessage.setContent(content);
                leaveMessage.setType(type);
                leaveMessage.setVisitor(visitorOptional.get());
                leaveMessage.setStatus(StatusConsts.LEAVE_MESSAGE_STATUS_UNCLAIMED);
//                    leaveMessage.setUser(adminOptional.get());
//                    leaveMessage.setSubDomain(adminOptional.get().getSubDomain());

                if (type.equals(TypeConsts.THREAD_REQUEST_TYPE_APPOINTED)) {
                    // 指定会话留言
                    Optional<User> agentOptional = userRepository.findByUid(aid);
                    if (agentOptional.isPresent()) {
                        leaveMessage.setAgent(agentOptional.get());
                        leaveMessage.setSubDomain(agentOptional.get().getAdmin().getSubDomain());
                    }
                } else {
                    // 工作组会话留言
                    Optional<WorkGroup> workGroupOptional = workGroupRepository.findByWid(wid);
                    if (workGroupOptional.isPresent()) {
                        leaveMessage.setWorkGroup(workGroupOptional.get());
                        leaveMessage.setSubDomain(workGroupOptional.get().getUser().getSubDomain());
                    }
                }
                leaveMessageRepository.save(leaveMessage);

                // 返回结果
                jsonResult.setMessage("留言成功");
                jsonResult.setStatus_code(200);
                jsonResult.setData(leaveMessage);

            } else {

                jsonResult.setMessage("留言失败-访客用户不存在");
                jsonResult.setStatus_code(-3);
                jsonResult.setData(false);
            }

        } else {

            jsonResult.setMessage("access token invalid");
            jsonResult.setStatus_code(-1);
            jsonResult.setData(false);
        }

        return jsonResult;

    }


    /**
     * 备注留言回复
     *
     * @param principal principal
     * @param map map
     * @return json
     */
    @PostMapping("/reply")
    @ResponseBody
    public JsonResult reply(Principal principal, @RequestBody Map map) {

        String lid = (String) map.get("lid");
        Boolean replied = (Boolean) map.get("replied");
        String reply = (String) map.get("reply");

        JsonResult jsonResult = new JsonResult();

        //
        if (principal != null) {

            Optional<User> adminOptional = userRepository.findByUsername(principal.getName());
            if (adminOptional.isPresent()) {

                Optional<LeaveMessage> leaveMessageOptional = leaveMessageRepository.findByLid(lid);
                if (leaveMessageOptional.isPresent()) {

                    // 持久化保存
                    leaveMessageOptional.get().setReplied(replied);
                    leaveMessageOptional.get().setReply(reply);
                    leaveMessageRepository.save(leaveMessageOptional.get());

                    // 返回结果
                    jsonResult.setMessage("备注留言回复成功");
                    jsonResult.setStatus_code(200);
                    jsonResult.setData(leaveMessageOptional.get());

                } else {

                    jsonResult.setMessage("备注留言回复失败-管理员用户不存在");
                    jsonResult.setStatus_code(-2);
                    jsonResult.setData(false);
                }

            } else {

                jsonResult.setMessage("备注留言回复失败-管理员用户不存在");
                jsonResult.setStatus_code(-2);
                jsonResult.setData(false);
            }

        } else {

            jsonResult.setMessage("备注留言回复失败-access token invalid");
            jsonResult.setStatus_code(-1);
            jsonResult.setData(false);
        }

        return jsonResult;
    }


    /**
     * 搜索过滤留言
     *
     * @param principal principal
     * @param page page
     * @param size size
     * @param nickname nickname
     * @param createdAtStart start
     * @param createdAtEnd end
     * @param workGroupNickname nickname
     * @param agentRealName realname
     * @param client client
     * @return json
     */
    @GetMapping("/filter")
    public JsonResult filter(Principal principal,
                             @RequestParam(value = "page") int page,
                             @RequestParam(value = "size") int size,
                             //
                             @RequestParam(value = "nickname") String nickname,
                             @RequestParam(value = "createdAtStart") String createdAtStart,
                             @RequestParam(value = "createdAtEnd") String createdAtEnd,
                             @RequestParam(value = "workGroupNickname") String workGroupNickname,
                             @RequestParam(value = "agentRealName") String agentRealName,
                             @RequestParam(value = "client") String client) {

        logger.info("page {}, size {}, nickname {}, createdAtStart {}, createdAtEnd {}, wgNickname {}, agentRealName {}, client {}",
                page, size, nickname, createdAtStart, createdAtEnd, workGroupNickname, agentRealName, client);

        JsonResult jsonResult = new JsonResult();

        if (principal != null) {

            //
            Optional<User> adminOptional = userRepository.findByUsername(principal.getName());
            if (adminOptional.isPresent()) {

                // 分页查询
                Pageable pageable = PageRequest.of(page, size, Sort.Direction.DESC, "id");

                Specification specification = getSpecification(adminOptional.get(), TypeConsts.SPECIFICATION_TYPE_LEAVE_MESSAGE, nickname, createdAtStart, createdAtEnd,
                        workGroupNickname, agentRealName, client);
                Page<LeaveMessage> leaveMessagePage = leaveMessageRepository.findAll(specification, pageable);

                // 返回结果
                jsonResult.setMessage("搜索留言成功");
                jsonResult.setStatus_code(200);
                jsonResult.setData(leaveMessagePage);

            }  else {

                jsonResult.setMessage("管理员用户不存在");
                jsonResult.setStatus_code(-2);
                jsonResult.setData(false);
            }

        } else {

            jsonResult.setMessage("access token invalid");
            jsonResult.setStatus_code(-1);
            jsonResult.setData(false);
        }

        return jsonResult;
    }



}
