package io.xiaper.mq.config;

import io.xiaper.jpa.constant.MqConsts;
import io.xiaper.jpa.constant.TypeConsts;
import io.xiaper.jpa.model.*;
import io.xiaper.jpa.model.Thread;
import io.xiaper.jpa.repository.MessageRepository;
import io.xiaper.jpa.repository.ThreadRepository;
import io.xiaper.jpa.repository.UserRepository;
import io.xiaper.mq.service.MessageService;
import io.xiaper.mq.service.StatusService;
import io.xiaper.mq.service.ThreadService;
import io.xiaper.mq.service.TransformService;
import io.xiaper.mq.service.redis.RedisConnectService;
import io.xiaper.mq.service.redis.RedisStatisticService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.jms.annotation.JmsListener;
import org.springframework.messaging.simp.SimpMessagingTemplate;
import org.springframework.stereotype.Component;

import java.util.Iterator;
import java.util.Map;
import java.util.Set;

/**
 *
 * @author xiaper.io
 */
@Component
public class ActiveMessageListener {

    private Logger logger = LoggerFactory.getLogger(this.getClass());

    @Autowired
    MessageService messageService;

    @Autowired
    SimpMessagingTemplate simpMessagingTemplate;

    @Autowired
    ThreadRepository threadRepository;

    @Autowired
    MessageRepository messageRepository;

    @Autowired
    UserRepository userRepository;

    @Autowired
    StatusService statusService;

    @Autowired
    RedisConnectService redisConnectService;

    @Autowired
    RedisStatisticService redisStatisticService;

    @Autowired
    TransformService transformService;

    @Autowired
    ThreadService threadService;

    /**
     * 监听来自集群广播的全平台的消息
     *
     * @param message msg
     */
    @JmsListener(destination = MqConsts.TOPIC_PLATFORM_MESSAGE)
    public void receivePlatformMessage(final Message message) {
        logger.info("receive from platformQueue message queue, content: {}", message.toString());

        // TODO: 精简消息体
    }


    /**
     * 监听来自集群广播的公司的消息
     * 包括：客服在线状态、公司公告
     *
     * @param message msg
     */
    @JmsListener(destination =  MqConsts.TOPIC_COMPANY_MESSAGE)
    public void receiveCompanyMessage(final Message message) {
        logger.info("receive from companyQueue message queue, content: {}", message.toString());

        if (null != message.getUser()) {
            String subDomain = message.getUser().getSubDomain();
            simpMessagingTemplate.convertAndSend(MqConsts.TOPIC_COMPANY_PREFIX + subDomain, message);
        }

        // TODO: 精简消息体
    }


    /**
     * 监听来自集群广播的工作组消息
     *
     * @param message msg
     */
    @JmsListener(destination =  MqConsts.TOPIC_WORK_GROUP_MESSAGE)
    public void receiveWorkGroupMessage(final Message message) {
        logger.info("receive from workGroupQueue message queue, content: {}", message.toString());

        // 直接广播给连接此服务器的订阅此工作组消息的stomp客户端
        // 精简消息体
        if (message.getType().equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_QUEUE)) {

            //
            Map<String, Object> messageMap = transformService.getWorkGroupQueueMessageMap(message);
            //
            String wid = message.getWid();
            simpMessagingTemplate.convertAndSend(MqConsts.TOPIC_WORK_GROUP_PREFIX + wid, messageMap);

        } else {

            String wid = message.getWid();
            simpMessagingTemplate.convertAndSend(MqConsts.TOPIC_WORK_GROUP_PREFIX + wid, message);
        }

    }

    /**
     * 群组消息队列
     *
     * @param message msg
     */
    @JmsListener(destination =  MqConsts.TOPIC_GROUP_MESSAGE)
    public void receiveGroupMessage(final Message message) {
        logger.info("receive from groupQueue message queue, content: {}", message.toString());

        // 直接广播给连接此服务器的订阅此工作组消息的stomp客户端
        //
        Map<String, Object> messageMap = transformService.getGroupMessageMap(message);

        //
        String gid = message.getGid();
        simpMessagingTemplate.convertAndSend(MqConsts.TOPIC_GROUP_PREFIX + gid, messageMap);
    }


    /**
     * 同事消息队列
     *
     * @param message msg
     */
    @JmsListener(destination =  MqConsts.TOPIC_CONTACT_MESSAGE)
    public void receiveContactMessage(final Message message) {
        logger.info("receive from contactQueue message queue, content: {}", message.toString());

        // 直接广播给连接此服务器的订阅此工作组消息的stomp客户端
        Map<String, Object> messageMap = transformService.getContactMessageMap(message);

        // 广播给接收方
        String cid = message.getCid();
        simpMessagingTemplate.convertAndSend(MqConsts.TOPIC_CONTACT_PREFIX + cid, messageMap);

        // 将接受者thread替换为发送者的thread，广播给发送方
        User user = message.getThread().getAgent();
        User contact = message.getThread().getContact();
        Thread thread = threadService.getContactThread(contact, user);
        message.setThread(thread);
        messageMap = transformService.getContactMessageMap(message);

        String uid = message.getUser().getUid();
        simpMessagingTemplate.convertAndSend(MqConsts.TOPIC_CONTACT_PREFIX + uid, messageMap);
    }


    /**
     * 监听Thread会话消息
     *
     * @param message msg
     */
    @JmsListener(destination = MqConsts.TOPIC_THREAD_MESSAGE)
    public void receiveThreadMessage(final Message message) {
        logger.info("receive from threadQueue message queue, content: {}", message.toString());

        // 直接发送给连接此服务器的订阅此会话消息的stomp客户端
        if (null != message.getThread()) {
            //
            String tid = message.getThread().getTid();

            // 精简消息体
            if (message.getType().equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_PREVIEW)) {
                //
                Map<String, Object> messageMap = transformService.getThreadPreviewMessageMap(message);
                //
                simpMessagingTemplate.convertAndSend(MqConsts.TOPIC_THREAD_PREFIX + tid, messageMap);

            } else if (message.getType().equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_RECEIPT)) {
                //
                Map<String, Object> messageMap = transformService.getThreadReceiptMessageMap(message);
                //
                simpMessagingTemplate.convertAndSend(MqConsts.TOPIC_THREAD_PREFIX + tid, messageMap);

            } else if (message.getType().equals(TypeConsts.MESSAGE_TYPE_TEXT)
                    || message.getType().equals(TypeConsts.MESSAGE_TYPE_IMAGE)
                    || message.getType().equals(TypeConsts.MESSAGE_TYPE_FILE)
                    || message.getType().equals(TypeConsts.MESSAGE_TYPE_VOICE)
                    || message.getType().equals(TypeConsts.MESSAGE_TYPE_VIDEO)
                    || message.getType().equals(TypeConsts.MESSAGE_TYPE_SHORT_VIDEO)
                    || message.getType().equals(TypeConsts.MESSAGE_TYPE_LOCATION)
                    || message.getType().equals(TypeConsts.MESSAGE_TYPE_LINK)

                    || message.getType().equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_AGENT_CLOSE)
                    || message.getType().equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_VISITOR_CLOSE)
                    || message.getType().equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_AUTO_CLOSE)

                    || message.getType().equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_INVITE_RATE)
                    || message.getType().equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_RATE_RESULT)

                    // webrtc 相关
                    || message.getType().equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_WEBRTC_INVITE)
                    || message.getType().equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_WEBRTC_CANCEL)
                    || message.getType().equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_WEBRTC_ACCEPT)
                    || message.getType().equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_WEBRTC_REJECT)
                    || message.getType().equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_WEBRTC_READY)
                    || message.getType().equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_WEBRTC_BUSY)
                    || message.getType().equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_WEBRTC_CLOSE)) {

                logger.info("处理文本、图片、连接、webrtc 消息 {}", message.getType());
                //
                Map<String, Object> messageMap = transformService.getThreadContentMessageMap(message);
                //
                simpMessagingTemplate.convertAndSend(MqConsts.TOPIC_THREAD_PREFIX + tid, messageMap);

            } else if (message.getType().equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_CONNECT)
                    || message.getType().equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_DISCONNECT)) {

                //
                Map<String, Object> messageMap = transformService.getThreadConnectionMessageMap(message);
                //
                simpMessagingTemplate.convertAndSend(MqConsts.TOPIC_THREAD_PREFIX + tid, messageMap);

            } else {
                logger.warn("处理其他消息 {}", message.getType());

                simpMessagingTemplate.convertAndSend(MqConsts.TOPIC_THREAD_PREFIX + tid, message);
            }

        }

    }

    /**
     * 监听来自集群广播的发送给某用户的消息
     *
     * @param message msg
     */
    @JmsListener(destination = MqConsts.TOPIC_USER_MESSAGE)
    public void receiveUserMessage(final Message message) {
        logger.info("receive from userQueue message queue, content: {}", message.toString());

        // 直接发送给连接此服务器的订阅此用户消息的stomp客户端，注：一个用户可以同时多个客户端在线
        if (message.getType().equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_TRANSFER)) {
            //
            Map<String, Object> messageMap = transformService.getUserTransferMessageMap(message);
            //
            Transfer threadTransfer = message.getTransfer();
            if (null != threadTransfer) {
                simpMessagingTemplate.convertAndSend(MqConsts.TOPIC_USER_PREFIX + threadTransfer.getToUser().getUid(), messageMap);
            }
        } else if (message.getType().equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_TRANSFER_ACCEPT)
                || message.getType().equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_TRANSFER_REJECT)) {
            //
            Map<String, Object> messageMap = transformService.getUserTransferAcceptRejectMessageMap(message);
            //
            Transfer threadTransfer = message.getTransfer();
            if (null != threadTransfer) {
                simpMessagingTemplate.convertAndSend(MqConsts.TOPIC_USER_PREFIX + threadTransfer.getFromUser().getUid(), messageMap);
            }

        } else if (message.getType().equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_INVITE)) {
            //
            Map<String, Object> messageMap = transformService.getUserInviteMessageMap(message);
            //
            Invite threadInvite = message.getInvite();
            if (null != threadInvite) {
                simpMessagingTemplate.convertAndSend(MqConsts.TOPIC_USER_PREFIX + threadInvite.getToUser().getUid(), messageMap);
            }

        } else if (message.getType().equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_INVITE_ACCEPT)
                || message.getType().equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_INVITE_REJECT)) {
            //
            Map<String, Object> messageMap = transformService.getUserInviteAcceptRejectMessageMap(message);
            //
            Invite threadInvite = message.getInvite();
            if (null != threadInvite) {
                simpMessagingTemplate.convertAndSend(MqConsts.TOPIC_USER_PREFIX + threadInvite.getFromUser().getUid(), messageMap);
            }

        } else if (message.getType().equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_KICKOFF)) {
            //
            Map<String, Object> messageMap = transformService.getUserKickoffMessageMap(message);
            //
            simpMessagingTemplate.convertAndSend( MqConsts.TOPIC_USER_PREFIX + message.getUser().getUid(), messageMap);

        } else if (message.getType().equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_THREAD)) {
            //
            Map<String, Object> messageMap = transformService.getUserThreadMessageMap(message);
            //
            User agent = message.getThread().getAgent();
            simpMessagingTemplate.convertAndSend(MqConsts.TOPIC_USER_PREFIX + agent.getUid(), messageMap);

        } else if (message.getType().equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_GROUP_CREATE)) {
            //
            Map<String, Object> messageMap = transformService.getUserGroupCreateMessageMap(message);
            // 创建群
            Group group = message.getGroup();
            Set<User> userSet = group.getMembers();
            Iterator iterator = userSet.iterator();
            while (iterator.hasNext()) {
                User user = (User) iterator.next();
                simpMessagingTemplate.convertAndSend(MqConsts.TOPIC_USER_PREFIX + user.getUid(), messageMap);
            }

        } else if (message.getType().equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_GROUP_INVITE)) {
            //
            Map<String, Object> messageMap = transformService.getUserGroupInviteMessageMap(message);
            //
            Invite invite = message.getInvite();
            String uid = invite.getToUser().getUid();
            simpMessagingTemplate.convertAndSend(MqConsts.TOPIC_USER_PREFIX + uid, messageMap);

        } else if (message.getType().equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_GROUP_APPLY)) {
            //
            Map<String, Object> messageMap = transformService.getUserGroupApplyMessageMap(message);
            //
            Notice notice = message.getNotice();
            Set<User> admins = notice.getUsers();
            Iterator iterator = admins.iterator();
            while (iterator.hasNext()) {
                User admin = (User) iterator.next();
                simpMessagingTemplate.convertAndSend(MqConsts.TOPIC_USER_PREFIX + admin.getUid(), messageMap);
            }

        } else if (message.getType().equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_GROUP_APPLY_APPROVE)) {
            //
            Map<String, Object> messageMap = transformService.getUserGroupApplyApproveMessageMap(message);
            //
            Notice notice = message.getNotice();
            String uid = notice.getUser().getUid();
            simpMessagingTemplate.convertAndSend(MqConsts.TOPIC_USER_PREFIX + uid, messageMap);

        } else if (message.getType().equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_GROUP_APPLY_DENY)) {
            //
            Map<String, Object> messageMap = transformService.getUserGroupApplyDenyMessageMap(message);
            //
            Notice notice = message.getNotice();
            String uid = notice.getUser().getUid();
            simpMessagingTemplate.convertAndSend(MqConsts.TOPIC_USER_PREFIX + uid, messageMap);

        } else if (message.getType().equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_GROUP_TRANSFER)) {
            //
            Map<String, Object> messageMap = transformService.getUserGroupTransferMessageMap(message);
            //
            Notice notice = message.getNotice();
            Set<User> admins = notice.getUsers();
            Iterator iterator = admins.iterator();
            while (iterator.hasNext()) {
                User admin = (User) iterator.next();
                simpMessagingTemplate.convertAndSend(MqConsts.TOPIC_USER_PREFIX + admin.getUid(), messageMap);
            }

        } else if (message.getType().equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_GROUP_TRANSFER_ACCEPT)) {
            //
            Map<String, Object> messageMap = transformService.getUserGroupTransferAcceptMessageMap(message);
            //
            Notice notice = message.getNotice();
            String uid = notice.getUser().getUid();
            simpMessagingTemplate.convertAndSend(MqConsts.TOPIC_USER_PREFIX + uid, messageMap);

        } else if (message.getType().equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_GROUP_TRANSFER_REJECT)) {
            //
            Map<String, Object> messageMap = transformService.getUserGroupTransferRejectMessageMap(message);
            //
            Notice notice = message.getNotice();
            String uid = notice.getUser().getUid();
            simpMessagingTemplate.convertAndSend(MqConsts.TOPIC_USER_PREFIX + uid, messageMap);

        } else if (message.getType().equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_GROUP_KICK)) {


        } else if (message.getType().equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_GROUP_MUTE)) {


        } else if (message.getType().equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_GROUP_WITHDRAW)) {


        } else if (message.getType().equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_GROUP_DISMISS)) {


        } else {

            if (null != message.getThread() && null != message.getThread().getAgent()) {
                User agent = message.getThread().getAgent();
                simpMessagingTemplate.convertAndSend(MqConsts.TOPIC_USER_PREFIX + agent.getUid(), message);
            }
        }
    }




}




