package io.xiaper.mq.stomp.listener;

import io.xiaper.jpa.constant.UserConsts;
import io.xiaper.jpa.model.Thread;
import io.xiaper.jpa.model.User;
import io.xiaper.jpa.repository.MessageRepository;
import io.xiaper.jpa.repository.SubscribeRepository;
import io.xiaper.jpa.repository.ThreadRepository;
import io.xiaper.jpa.repository.UserRepository;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationListener;
import org.springframework.messaging.MessageHeaders;
import org.springframework.messaging.simp.SimpMessageHeaderAccessor;
import org.springframework.messaging.simp.SimpMessagingTemplate;
import org.springframework.stereotype.Component;
import org.springframework.web.socket.messaging.SessionUnsubscribeEvent;

import java.security.Principal;
import java.util.Optional;

/**
 * published when a new STOMP UNSUBSCRIBE is received.
 *
 * https://docs.spring.io/spring/docs/current/spring-framework-reference/web.html#websocket-stomp-appplication-context-events
 * @author xiaper.io
 */
@Component
public class StompUnsubscribeListener implements ApplicationListener<SessionUnsubscribeEvent> {

    private static final Logger logger = LoggerFactory.getLogger(StompUnsubscribeListener.class);

    /**
     * 通过simpMessagingTemplate向浏览器发送消息
     */
    @Autowired
    private SimpMessagingTemplate simpMessagingTemplate;

    @Autowired
    UserRepository userRepository;

    @Autowired
    ThreadRepository threadRepository;

    @Autowired
    SubscribeRepository subscribeRepository;

    @Autowired
    MessageRepository messageRepository;

    /**
     * 监听来自stomp，也即web端的取消订阅事件
     *
     * @param sessionUnsubscribeEvent
     */
    @Override
    public void onApplicationEvent(SessionUnsubscribeEvent sessionUnsubscribeEvent) {
        // logger.info(sessionUnsubscribeEvent.toString());

        //
        MessageHeaders headers = sessionUnsubscribeEvent.getMessage().getHeaders();
        Principal principal = SimpMessageHeaderAccessor.getUser(headers);
        if (principal == null) {
            return;
        }
        Optional<User> userOptional = userRepository.findByUsername(principal.getName());
        if (!userOptional.isPresent()) {
            return;
        }

        String sessionId = SimpMessageHeaderAccessor.getSessionId(headers);
        String destination = SimpMessageHeaderAccessor.getDestination(headers);

        /**
         * 处理topic/thread会话取消订阅事件
         * destination格式：/topic/thread.201807172109261
         */
        String topicThreadPrefix = "/topic/thread";
        if (destination.startsWith(topicThreadPrefix)) {
            //
            String[] splitSlash = destination.split("/");
            String[] splitDot = splitSlash[splitSlash.length - 1].split("[.]");
            String threadTid = splitDot[splitDot.length - 1];

            logger.info("tid: {}", threadTid);
            //
            Optional<Thread> threadOptional = threadRepository.findByTid(threadTid);
            if (threadOptional.isPresent()) {

                // TODO: 1. 持久化，用户离开会话
                subscribeRepository.deleteByThread(threadOptional.get());

                // TODO: 2. 通知相关用户
                Optional<User> userNotification = userRepository.findByUsername(UserConsts.USERNAME_NOTIFICATION);
                if (userNotification.isPresent()) {
                    //
                }
            }

        } else {
            // 其他topic取消订阅事件
        }

    }
}
