package io.xiaper.mq.config;

import io.xiaper.jpa.constant.MqConsts;
import org.springframework.amqp.core.Binding;
import org.springframework.amqp.core.BindingBuilder;
import org.springframework.amqp.core.Queue;
import org.springframework.amqp.core.TopicExchange;
import org.springframework.amqp.rabbit.annotation.EnableRabbit;
import org.springframework.amqp.rabbit.annotation.RabbitListenerConfigurer;
import org.springframework.amqp.rabbit.connection.ConnectionFactory;
import org.springframework.amqp.rabbit.core.RabbitTemplate;
import org.springframework.amqp.rabbit.listener.RabbitListenerEndpointRegistrar;
import org.springframework.amqp.support.converter.Jackson2JsonMessageConverter;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.messaging.converter.MappingJackson2MessageConverter;
import org.springframework.messaging.handler.annotation.support.DefaultMessageHandlerMethodFactory;

/**
 * Part 1: RabbitMQ Best Practice
 * https://www.cloudamqp.com/blog/2017-12-29-part1-rabbitmq-best-practice.html
 *
 * 教程：
 * https://spring.io/guides/gs/messaging-rabbitmq/
 * http://www.rabbitmq.com/tutorials/tutorial-five-spring-amqp.html
 *
 * 发送 转换 JSON 消息
 * Exchange JSON messages with Spring Boot AMQP and RabbitMQ:
 * https://thepracticaldeveloper.com/2016/10/23/produce-and-consume-json-messages-with-spring-boot-amqp/
 * https://github.com/mechero/spring-boot-amqp-messaging
 *
 * TODO: 如果发送消息对方不在线，则将消息存储到MySQL之后，RabbitMQ直接丢弃掉消息，
 * TODO: 也即：清理掉RabbitMQ后台Overview页面Queued Messages, 使其总数为0
 *
 * @author xiaper.io
 */
@EnableRabbit
@Configuration
public class RabbitConfig implements RabbitListenerConfigurer {

    /**
     * exchange types:
     * 1. fanout: broadcast
     * 2. direct: exact key
     * 3. topic: wildcard key
     * 4. headers
     *
     * @return queue
     */
    @Bean
    public TopicExchange topicExchange() {
        return new TopicExchange(MqConsts.EXCHANGE_DEFAULT_TOPIC_NAME);
    }

    /**
     * 监听平台公告
     *
     * @return queue
     */
    @Bean
    public Queue platformQueue() {
        return new Queue(MqConsts.QUEUE_PLATFORM_MESSAGE, true, false, false);
    }

    @Bean
    public Binding bindingTopicPlatformQueue(TopicExchange topicExchange, Queue platformQueue) {
        return BindingBuilder.bind(platformQueue).to(topicExchange).with(MqConsts.TOPIC_PLATFORM_MESSAGE);
    }

    /**
     * 监听公司消息
     *
     * @return queue
     */
    @Bean
    public Queue companyQueue() {
        return new Queue(MqConsts.QUEUE_COMPANY_MESSAGE, true, false, false);
    }

    @Bean
    public Binding bindingTopicCompanyQueue(TopicExchange topicExchange, Queue companyQueue) {
        return BindingBuilder.bind(companyQueue).to(topicExchange).with(MqConsts.TOPIC_COMPANY_MESSAGE);
    }

    /**
     * 监听所有工作组消息
     *
     * @return queue
     */
    @Bean
    public Queue workGroupQueue() {
        return new Queue(MqConsts.QUEUE_WORK_GROUP_MESSAGE,true,false,false);
    }

    @Bean
    public Binding bindingTopicWorkGroupQueue(TopicExchange topicExchange, Queue workGroupQueue) {
        return BindingBuilder.bind(workGroupQueue).to(topicExchange).with(MqConsts.TOPIC_WORK_GROUP_MESSAGE);
    }

    /**
     * 监听群组消息
     *
     * @return queue
     */
    @Bean
    public Queue groupQueue() {
        return new Queue(MqConsts.QUEUE_GROUP_MESSAGE,true,false,false);
    }

    @Bean
    public Binding bindingTopicGroupQueue(TopicExchange topicExchange, Queue groupQueue) {
        return BindingBuilder.bind(groupQueue).to(topicExchange).with(MqConsts.TOPIC_GROUP_MESSAGE);
    }

    /**
     * 监听同事消息
     *
     * @return queue
     */
    @Bean
    public Queue contactQueue() {
        return new Queue(MqConsts.QUEUE_CONTACT_MESSAGE, true, false, false);
    }

    @Bean
    public Binding bindingTopicContactQueue(TopicExchange topicExchange, Queue contactQueue) {
        return BindingBuilder.bind(contactQueue).to(topicExchange).with(MqConsts.TOPIC_CONTACT_MESSAGE);
    }

    /**
     * 监听Thread会话消息
     *
     * @return queue
     */
    @Bean
    public Queue threadQueue() {
        return new Queue(MqConsts.QUEUE_THREAD_MESSAGE, true, false, false);
    }

    @Bean
    public Binding bindingTopicThreadQueue(TopicExchange topicExchange, Queue threadQueue) {
        return BindingBuilder.bind(threadQueue).to(topicExchange).with(MqConsts.TOPIC_THREAD_MESSAGE);
    }

    /**
     * 监听所有用户消息
     *
     * @return queue
     */
    @Bean
    public Queue userQueue() {
        return new Queue(MqConsts.QUEUE_USER_MESSAGE, true, false, false);
    }

    @Bean
    public Binding bindingTopicUserQueue(TopicExchange topicExchange, Queue userQueue) {
        return BindingBuilder.bind(userQueue).to(topicExchange).with(MqConsts.TOPIC_USER_MESSAGE);
    }

    /**
     * 监听来自mqtt客户端的消息
     *
     * @return queue
     */
    @Bean
    public Queue mqttMessageQueue() {
        return new Queue(MqConsts.QUEUE_MQTT_MESSAGE, true, false, false);
    }

    @Bean
    public Binding bindingTopicMqttMessageQueue(TopicExchange topicExchange, Queue mqttMessageQueue) {
        return BindingBuilder.bind(mqttMessageQueue).to(topicExchange).with(MqConsts.TOPIC_MQTT_MESSAGE);
    }


    /**
     * 监听来自mqtt客户端的消息
     *
     * @return queue
     */
    @Bean
    public Queue mqttReceiptQueue() {
        return new Queue(MqConsts.QUEUE_MQTT_RECEIPT, true, false, false);
    }

    @Bean
    public Binding bindingTopicMqttReceiptQueue(TopicExchange topicExchange, Queue mqttReceiptQueue) {
        return BindingBuilder.bind(mqttReceiptQueue).to(topicExchange).with(MqConsts.TOPIC_MQTT_RECEIPT);
    }

    /**
     * 监听mqtt WebRTC消息
     * @return queue
     */
    @Bean
    public Queue mqttWebRTCQueue() {
        return new Queue(MqConsts.QUEUE_MQTT_WEBRTC, true, false, false);
    }

    @Bean
    public Binding bindingTopicMqttWebRTCQueue(TopicExchange topicExchange, Queue mqttWebRTCQueue) {
        return BindingBuilder.bind(mqttWebRTCQueue).to(topicExchange).with(MqConsts.TOPIC_MQTT_WEBRTC);
    }

    /**
     * 监听mqtt客户端的状态
     *
     * @return queue
     */
    @Bean
    public Queue mqttStatusQueue() {
        return new Queue(MqConsts.QUEUE_MQTT_STATUS, false, false, true);
    }

    @Bean
    public Binding bindingTopicMqttStatusQueue(TopicExchange topicExchange, Queue mqttStatusQueue) {
        return BindingBuilder.bind(mqttStatusQueue).to(topicExchange).with(MqConsts.TOPIC_MQTT_STATUS);
    }

    /**
     * 监听mqtt客户端的last will遗嘱
     *
     * @return queue
     */
    @Bean
    public Queue mqttLastWillQueue() {
        return new Queue(MqConsts.QUEUE_MQTT_LAST_WILL,false,false,true);
    }

    @Bean
    public Binding bindingTopicMqttLastWillQueue(TopicExchange topicExchange, Queue mqttLastWillQueue) {
        return BindingBuilder.bind(mqttLastWillQueue).to(topicExchange).with(MqConsts.TOPIC_MQTT_LAST_WILL);
    }


    /**
     * 监听来自mqtt客户端的消息: protobuf
     *
     * @return queue
     */
    @Bean
    public Queue mqttMessageProtobufQueue() {
        return new Queue(MqConsts.QUEUE_MQTT_MESSAGE_PROTOBUF, true, false, false);
    }

    @Bean
    public Binding bindingTopicMqttProtobufMessageQueue(TopicExchange topicExchange, Queue mqttMessageProtobufQueue) {
        return BindingBuilder.bind(mqttMessageProtobufQueue).to(topicExchange).with(MqConsts.TOPIC_MQTT_MESSAGE_PROTOBUF);
    }

    /**
     * 监听来自mqtt客户端的消息: protobuf
     *
     * @return queue
     */
    @Bean
    public Queue mqttReceiptProtobufQueue() {
        return new Queue(MqConsts.QUEUE_MQTT_RECEIPT_PROTOBUF, true, false, false);
    }

    @Bean
    public Binding bindingTopicMqttProtobufReceiptQueue(TopicExchange topicExchange, Queue mqttReceiptProtobufQueue) {
        return BindingBuilder.bind(mqttReceiptProtobufQueue).to(topicExchange).with(MqConsts.TOPIC_MQTT_RECEIPT_PROTOBUF);
    }


    /**
     * 监听mqtt客户端的状态: protobuf
     *
     * @return queue
     */
    @Bean
    public Queue mqttStatusProtobufQueue() {
        return new Queue(MqConsts.QUEUE_MQTT_STATUS_PROTOBUF, false, false, true);
    }

    @Bean
    public Binding bindingTopicMqttStatusProtobufQueue(TopicExchange topicExchange, Queue mqttStatusProtobufQueue) {
        return BindingBuilder.bind(mqttStatusProtobufQueue).to(topicExchange).with(MqConsts.TOPIC_MQTT_STATUS_PROTOBUF);
    }

    /**
     * 监听mqtt客户端的last will遗嘱: protobuf
     *
     * @return queue
     */
    @Bean
    public Queue mqttLastWillProtobufQueue() {
        return new Queue(MqConsts.QUEUE_MQTT_LAST_WILL_PROTOBUF,false,false,true);
    }

    @Bean
    public Binding bindingTopicMqttLastWillProtobufQueue(TopicExchange topicExchange, Queue mqttLastWillProtobufQueue) {
        return BindingBuilder.bind(mqttLastWillProtobufQueue).to(topicExchange).with(MqConsts.TOPIC_MQTT_LAST_WILL_PROTOBUF);
    }

    @Bean
    public RabbitTemplate rabbitTemplate(final ConnectionFactory connectionFactory) {
        final RabbitTemplate rabbitTemplate = new RabbitTemplate(connectionFactory);
        rabbitTemplate.setMessageConverter(producerJackson2MessageConverter());
        return rabbitTemplate;
    }

    @Bean
    public Jackson2JsonMessageConverter producerJackson2MessageConverter() {
        return new Jackson2JsonMessageConverter();
    }

    @Bean
    public MappingJackson2MessageConverter consumerJackson2MessageConverter() {
        return new MappingJackson2MessageConverter();
    }

    @Bean
    public DefaultMessageHandlerMethodFactory messageHandlerMethodFactory() {
        DefaultMessageHandlerMethodFactory factory = new DefaultMessageHandlerMethodFactory();
        factory.setMessageConverter(consumerJackson2MessageConverter());
        return factory;
    }

    @Override
    public void configureRabbitListeners(final RabbitListenerEndpointRegistrar registrar) {
        registrar.setMessageHandlerMethodFactory(messageHandlerMethodFactory());
    }


}

