package io.xiaper.mq.config;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import io.xiaper.jpa.constant.MqConsts;
import io.xiaper.jpa.constant.StatusConsts;
import io.xiaper.jpa.constant.TypeConsts;
import io.xiaper.jpa.model.*;
import io.xiaper.jpa.model.Thread;
import io.xiaper.jpa.repository.MessageRepository;
import io.xiaper.jpa.repository.ThreadRepository;
import io.xiaper.jpa.repository.UserRepository;
import io.xiaper.mq.service.MessageService;
import io.xiaper.mq.service.StatusService;
import io.xiaper.mq.service.ThreadService;
import io.xiaper.mq.service.TransformService;
import io.xiaper.mq.service.redis.RedisConnectService;
import io.xiaper.mq.service.redis.RedisStatisticService;
import io.xiaper.mq.service.wechat.WeChatMiniService;
import io.xiaper.mq.service.wechat.WeChatMpService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.amqp.rabbit.annotation.RabbitListener;
import org.springframework.amqp.rabbit.core.RabbitTemplate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.messaging.simp.SimpMessagingTemplate;
import org.springframework.stereotype.Service;

import java.util.Iterator;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

/**
 *
 *
 * @author xiaper.io
 */
@Service
public class RabbitMessageListener {

    private final Logger logger = LoggerFactory.getLogger(this.getClass());

    @Autowired
    RabbitTemplate rabbitTemplate;

    @Autowired
    MessageService messageService;

    @Autowired
    SimpMessagingTemplate simpMessagingTemplate;

    @Autowired
    ThreadRepository threadRepository;

    @Autowired
    MessageRepository messageRepository;

    @Autowired
    UserRepository userRepository;

    @Autowired
    StatusService statusService;

    @Autowired
    RedisConnectService redisConnectService;

    @Autowired
    RedisStatisticService redisStatisticService;

    @Autowired
    WeChatMpService weChatMpService;

    @Autowired
    WeChatMiniService weChatMiniService;

    @Autowired
    TransformService transformService;

    @Autowired
    ThreadService threadService;

    /**
     * 监听来自集群广播的全平台的消息
     *
     * @param message msg
     */
    @RabbitListener(queues =  "#{platformQueue.name}")
    public void receivePlatformMessage(final Message message) {
        logger.info("receive from platformQueue message queue, content: {}", message.toString());

        // TODO: 精简消息体

    }

    /**
     * 监听来自集群广播的公司的消息
     * 包括：客服在线状态、公司公告
     *
     * @param message msg
     */
    @RabbitListener(queues =  "#{companyQueue.name}")
    public void receiveCompanyMessage(final Message message) {
        logger.info("receive from companyQueue message queue, content: {}", message.toString());

        if (null != message.getUser()) {
            String subDomain = message.getUser().getSubDomain();
            simpMessagingTemplate.convertAndSend(MqConsts.TOPIC_COMPANY_PREFIX + subDomain, message);
        }

        // TODO: 精简消息体

    }

    /**
     * 监听来自集群广播的工作组消息
     *
     * @param message msg
     */
    @RabbitListener(queues =  "#{workGroupQueue.name}")
    public void receiveWorkGroupMessage(final Message message) {
        logger.info("receive from workGroupQueue message queue, content: {}", message.toString());

        // 直接广播给连接此服务器的订阅此工作组消息的stomp客户端
        // 精简消息体
        if (message.getType().equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_QUEUE)) {

            //
            Map<String, Object> messageMap = transformService.getWorkGroupQueueMessageMap(message);
            //
            String wid = message.getWid();
            simpMessagingTemplate.convertAndSend(MqConsts.TOPIC_WORK_GROUP_PREFIX + wid, messageMap);

        } else if (message.getType().equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_BROWSE_START)
                || message.getType().equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_BROWSE_END)) {

            //
            Map<String, Object> messageMap = transformService.getWorkGroupBrowseMessageMap(message);
            //
            String wid = message.getWid();
            simpMessagingTemplate.convertAndSend(MqConsts.TOPIC_WORK_GROUP_PREFIX + wid, messageMap);

        } else {

            String wid = message.getWid();
            simpMessagingTemplate.convertAndSend(MqConsts.TOPIC_WORK_GROUP_PREFIX + wid, message);
        }
    }


    /**
     * 群组消息队列
     *
     * @param message msg
     */
    @RabbitListener(queues =  "#{groupQueue.name}")
    public void receiveGroupMessage(final Message message) {
        logger.info("receive from groupQueue message queue, content: {}", message.toString());

        // 直接广播给连接此服务器的订阅此工作组消息的stomp客户端
        Map<String, Object> messageMap = transformService.getGroupMessageMap(message);
        //
        String gid = message.getGid();
        simpMessagingTemplate.convertAndSend(MqConsts.TOPIC_GROUP_PREFIX + gid, messageMap);
    }


    /**
     * 单聊消息队列
     *
     * @param message msg
     */
    @RabbitListener(queues =  "#{contactQueue.name}")
    public void receiveContactMessage(final Message message) {
        logger.info("receive from contactQueue message queue, content: {}", message.toString());

        String cid = message.getCid();

        // 精简消息体
        if (message.getType().equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_PREVIEW)) {
            //
            Map<String, Object> messageMap = transformService.getContactPreviewMessageMap(message);
            //
            simpMessagingTemplate.convertAndSend(MqConsts.TOPIC_CONTACT_PREFIX + cid, messageMap);

        } else if (message.getType().equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_RECEIPT)) {
            //
            Map<String, Object> messageMap = transformService.getContactReceiptMessageMap(message);
            //
            simpMessagingTemplate.convertAndSend(MqConsts.TOPIC_CONTACT_PREFIX + cid, messageMap);

        } else if (message.getType().equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_WEBRTC_INVITE) ||
                message.getType().equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_WEBRTC_OFFER_AUDIO) ||
                message.getType().equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_WEBRTC_OFFER_VIDEO) ||
                message.getType().equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_WEBRTC_ANSWER) ||
                message.getType().equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_WEBRTC_CANDIDATE) ||
                //
                message.getType().equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_WEBRTC_CANCEL) ||
                message.getType().equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_WEBRTC_REJECT) ||
                message.getType().equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_WEBRTC_ACCEPT) ||
                message.getType().equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_WEBRTC_CLOSE)) {
            //
            Map<String, Object> messageMap = transformService.getContactMessageMap(message);
            //
            simpMessagingTemplate.convertAndSend(MqConsts.TOPIC_CONTACT_PREFIX + cid, messageMap);

        } else {

            // 直接广播给连接此服务器的订阅此工作组消息的stomp客户端
            // FIXME: 下面4行多余的代码主要用于解决thread.isTemp函数中agent.getFollows等函数size为0的情况
            User user = message.getThread().getAgent();
            User contact = message.getThread().getContact();
            Thread thread = threadService.getContactThread(user, contact);
            message.setThread(thread);
            Map<String, Object> messageMap = transformService.getContactMessageMap(message);

            // 广播给接收方
            simpMessagingTemplate.convertAndSend(MqConsts.TOPIC_CONTACT_PREFIX + cid, messageMap);

            // 将接受者thread替换为发送者的thread，广播给发送方
            user = message.getThread().getAgent();
            contact = message.getThread().getContact();
            thread = threadService.getContactThread(contact, user);
            message.setThread(thread);
            messageMap = transformService.getContactMessageMap(message);

            String uid = message.getUser().getUid();
            simpMessagingTemplate.convertAndSend(MqConsts.TOPIC_CONTACT_PREFIX + uid, messageMap);
        }

    }


    /**
     * 监听Thread会话消息
     *
     * @param message msg
     */
    @RabbitListener(queues = "#{threadQueue.name}")
    public void receiveThreadMessage(final Message message) {
        logger.info("receive from threadQueue message queue, content: {}", message.toString());

        // 直接发送给连接此服务器的订阅此会话消息的stomp客户端
        if (null != message.getThread()) {
            //
            String tid = message.getThread().getTid();

            // 精简消息体
            if (message.getType().equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_PREVIEW)) {
                //
                Map<String, Object> messageMap = transformService.getThreadPreviewMessageMap(message);
                //
                simpMessagingTemplate.convertAndSend(MqConsts.TOPIC_THREAD_PREFIX + tid, messageMap);

            } else if (message.getType().equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_RECEIPT)) {
                //
                Map<String, Object> messageMap = transformService.getThreadReceiptMessageMap(message);
                //
                simpMessagingTemplate.convertAndSend(MqConsts.TOPIC_THREAD_PREFIX + tid, messageMap);

            } else if (message.getType().equals(TypeConsts.MESSAGE_TYPE_TEXT)
                    || message.getType().equals(TypeConsts.MESSAGE_TYPE_IMAGE)
                    || message.getType().equals(TypeConsts.MESSAGE_TYPE_FILE)
                    || message.getType().equals(TypeConsts.MESSAGE_TYPE_VOICE)
                    || message.getType().equals(TypeConsts.MESSAGE_TYPE_VIDEO)
                    || message.getType().equals(TypeConsts.MESSAGE_TYPE_SHORT_VIDEO)
                    || message.getType().equals(TypeConsts.MESSAGE_TYPE_LOCATION)
                    || message.getType().equals(TypeConsts.MESSAGE_TYPE_LINK)
                    || message.getType().equals(TypeConsts.MESSAGE_TYPE_COMMODITY)
                    || message.getType().equals(TypeConsts.MESSAGE_TYPE_RED_PACKET)


                    || message.getType().equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_AGENT_CLOSE)
                    || message.getType().equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_VISITOR_CLOSE)
                    || message.getType().equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_AUTO_CLOSE)

                    || message.getType().equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_INVITE_RATE)
                    || message.getType().equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_RATE_RESULT)

                    // webrtc 相关
                    || message.getType().equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_WEBRTC_INVITE)
                    || message.getType().equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_WEBRTC_CANCEL)
                    || message.getType().equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_WEBRTC_ACCEPT)
                    || message.getType().equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_WEBRTC_REJECT)
                    || message.getType().equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_WEBRTC_READY)
                    || message.getType().equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_WEBRTC_BUSY)
                    || message.getType().equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_WEBRTC_CLOSE)) {

                //
                Map<String, Object> messageMap = transformService.getThreadContentMessageMap(message);
                //
                simpMessagingTemplate.convertAndSend(MqConsts.TOPIC_THREAD_PREFIX + tid, messageMap);

            } else if (message.getType().equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_CONNECT)
                    || message.getType().equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_DISCONNECT)) {

                //
                Map<String, Object> messageMap = transformService.getThreadConnectionMessageMap(message);
                //
                simpMessagingTemplate.convertAndSend(MqConsts.TOPIC_THREAD_PREFIX + tid, messageMap);

            } else if (message.getType().equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_WEBRTC_CANDIDATE)) {
                //
                Map<String, Object> messageMap = transformService.getThreadWebRtcCandidateMessageMap(message);
                //
                simpMessagingTemplate.convertAndSend(MqConsts.TOPIC_THREAD_PREFIX + tid, messageMap);

            } else if (message.getType().equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_WEBRTC_OFFER_VIDEO)
                    || message.getType().equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_WEBRTC_OFFER_AUDIO)
                    || message.getType().equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_WEBRTC_ANSWER)) {
                //
                Map<String, Object> messageMap = transformService.getThreadWebRtcOfferAnswerMessageMap(message);
                //
                simpMessagingTemplate.convertAndSend(MqConsts.TOPIC_THREAD_PREFIX + tid, messageMap);

            } else {
                logger.warn("其他消息 {}", message.getType());

                simpMessagingTemplate.convertAndSend(MqConsts.TOPIC_THREAD_PREFIX + tid, message);
            }

            // 推送到微信公众号、小程序
            pushToWeChat(message);
        }

    }


    /**
     * 推送给微信公众号、小程序
     *
     * @param message msg
     */
    private void pushToWeChat(final Message message) {

        if (message.getType().equals(TypeConsts.MESSAGE_TYPE_TEXT)
            || message.getType().equals(TypeConsts.MESSAGE_TYPE_IMAGE)
            || message.getType().equals(TypeConsts.MESSAGE_TYPE_FILE)
            || message.getType().equals(TypeConsts.MESSAGE_TYPE_VOICE)
            || message.getType().equals(TypeConsts.MESSAGE_TYPE_VIDEO)
            || message.getType().equals(TypeConsts.MESSAGE_TYPE_SHORT_VIDEO)
            || message.getType().equals(TypeConsts.MESSAGE_TYPE_LOCATION)
            || message.getType().equals(TypeConsts.MESSAGE_TYPE_LINK)) {

            // FIXME: 微信相关消息应该通过queue方式处理，否则部署多个服务器实例就会收到多条重复消息推送
            // FIXME: 停止向访客端推送客服上线、离线消息
            if (message.getThread().getVisitor().isWeChatMp() && !message.getUser().isWeChatMp()) {
                // 访客来自公众号 且 消息不是来自公众号(client标示)

                weChatMpService.pushToWeChat(message);

            } else if (message.getThread().getVisitor().isWeChatMini() && !message.getUser().isWeChatMini()) {
                // 访客来自小程序 且 消息不是来自小程序(client标示)

                weChatMiniService.pushToWeChat(message);
            }
        }
    }

    /**
     * 监听来自集群广播的发送给某用户的消息
     *
     * @param message msg
     */
    @RabbitListener(queues = "#{userQueue.name}")
    public void receiveUserMessage(final Message message) {
        logger.info("receive from userQueue message queue, content: {}", message.toString());

        if (message.getType().equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_TRANSFER)) {
            //
            Map<String, Object> messageMap = transformService.getUserTransferMessageMap(message);
            //
            Transfer threadTransfer = message.getTransfer();
            if (null != threadTransfer) {
                simpMessagingTemplate.convertAndSend(MqConsts.TOPIC_USER_PREFIX + threadTransfer.getToUser().getUid(), messageMap);
            }
        } else if (message.getType().equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_TRANSFER_ACCEPT)
                || message.getType().equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_TRANSFER_REJECT)) {
            //
            Map<String, Object> messageMap = transformService.getUserTransferAcceptRejectMessageMap(message);
            //
            Transfer threadTransfer = message.getTransfer();
            if (null != threadTransfer) {
                simpMessagingTemplate.convertAndSend(MqConsts.TOPIC_USER_PREFIX + threadTransfer.getFromUser().getUid(), messageMap);
            }

        } else if (message.getType().equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_INVITE)) {
            //
            Map<String, Object> messageMap = transformService.getUserInviteMessageMap(message);
            //
            Invite threadInvite = message.getInvite();
            if (null != threadInvite) {
                simpMessagingTemplate.convertAndSend(MqConsts.TOPIC_USER_PREFIX + threadInvite.getToUser().getUid(), messageMap);
            }

        } else if (message.getType().equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_INVITE_ACCEPT)
                || message.getType().equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_INVITE_REJECT)) {
            //
            Map<String, Object> messageMap = transformService.getUserInviteAcceptRejectMessageMap(message);
            //
            Invite threadInvite = message.getInvite();
            if (null != threadInvite) {
                simpMessagingTemplate.convertAndSend(MqConsts.TOPIC_USER_PREFIX + threadInvite.getFromUser().getUid(), messageMap);
            }

        } else if (message.getType().equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_BROWSE_INVITE)) {

            //
            Map<String, Object> messageMap = transformService.getUserBrowseInviteMessageMap(message);

            BrowseInvite browseInvite = message.getBrowseInvite();
            if (null != browseInvite) {
                simpMessagingTemplate.convertAndSend(MqConsts.TOPIC_USER_PREFIX + browseInvite.getToUser().getUid(), messageMap);
            }

        } else if (message.getType().equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_RECEIPT)) {

            // 发送消息回执：通知发送者，消息已经成功发送
            Map<String, Object> messageMap = transformService.getUserReceiptMessageMap(message);
            //
            simpMessagingTemplate.convertAndSend( MqConsts.TOPIC_USER_PREFIX + message.getUser().getUid(), messageMap);

        } else if (message.getType().equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_KICKOFF)) {
            //
            Map<String, Object> messageMap = transformService.getUserKickoffMessageMap(message);
            //
            simpMessagingTemplate.convertAndSend( MqConsts.TOPIC_USER_PREFIX + message.getUser().getUid(), messageMap);

        } else if (message.getType().equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_THREAD) ||
                message.getType().equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_THREAD_REENTRY)) {
            //
            Map<String, Object> messageMap = transformService.getUserThreadMessageMap(message);
            //
            User agent = message.getThread().getAgent();
            simpMessagingTemplate.convertAndSend(MqConsts.TOPIC_USER_PREFIX + agent.getUid(), messageMap);

        } else if (message.getType().equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_GROUP_CREATE)) {
            //
            Map<String, Object> messageMap = transformService.getUserGroupCreateMessageMap(message);
            // 创建群
            Group group = message.getGroup();
            Set<User> userSet = group.getMembers();
            Iterator iterator = userSet.iterator();
            while (iterator.hasNext()) {
                User user = (User) iterator.next();
                simpMessagingTemplate.convertAndSend(MqConsts.TOPIC_USER_PREFIX + user.getUid(), messageMap);
            }

        } else if (message.getType().equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_GROUP_INVITE)) {
            //
            Map<String, Object> messageMap = transformService.getUserGroupInviteMessageMap(message);
            //
//            Invite invite = message.getInvite();
//            String uid = invite.getToUser().getUid();
            String uid = message.getCid();
            simpMessagingTemplate.convertAndSend(MqConsts.TOPIC_USER_PREFIX + uid, messageMap);

        } else if (message.getType().equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_GROUP_APPLY)) {
            //
            Map<String, Object> messageMap = transformService.getUserGroupApplyMessageMap(message);
            //
            Notice notice = message.getNotice();
            Set<User> admins = notice.getUsers();
            Iterator iterator = admins.iterator();
            while (iterator.hasNext()) {
                User admin = (User) iterator.next();
                simpMessagingTemplate.convertAndSend(MqConsts.TOPIC_USER_PREFIX + admin.getUid(), messageMap);
            }

        } else if (message.getType().equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_GROUP_APPLY_APPROVE)) {
            //
            Map<String, Object> messageMap = transformService.getUserGroupApplyApproveMessageMap(message);
            //
            Notice notice = message.getNotice();
            String uid = notice.getUser().getUid();
            simpMessagingTemplate.convertAndSend(MqConsts.TOPIC_USER_PREFIX + uid, messageMap);

        } else if (message.getType().equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_GROUP_APPLY_DENY)) {
            //
            Map<String, Object> messageMap = transformService.getUserGroupApplyDenyMessageMap(message);
            //
            Notice notice = message.getNotice();
            String uid = notice.getUser().getUid();
            simpMessagingTemplate.convertAndSend(MqConsts.TOPIC_USER_PREFIX + uid, messageMap);

        } else if (message.getType().equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_GROUP_TRANSFER)) {
            //
            Map<String, Object> messageMap = transformService.getUserGroupTransferMessageMap(message);
            //
            Notice notice = message.getNotice();
            Set<User> admins = notice.getUsers();
            Iterator iterator = admins.iterator();
            while (iterator.hasNext()) {
                User admin = (User) iterator.next();
                simpMessagingTemplate.convertAndSend(MqConsts.TOPIC_USER_PREFIX + admin.getUid(), messageMap);
            }

        } else if (message.getType().equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_GROUP_TRANSFER_ACCEPT)) {
            //
            Map<String, Object> messageMap = transformService.getUserGroupTransferAcceptMessageMap(message);
            //
            Notice notice = message.getNotice();
            if (notice != null) {
                //
                if (notice.getUser() != null) {
                    //
                    String uid = notice.getUser().getUid();
                    simpMessagingTemplate.convertAndSend(MqConsts.TOPIC_USER_PREFIX + uid, messageMap);

                } else {
                    logger.info("notice.getUser is null");
                }

            } else {
                logger.info("notice is null");
            }

        } else if (message.getType().equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_GROUP_TRANSFER_REJECT)) {
            //
            Map<String, Object> messageMap = transformService.getUserGroupTransferRejectMessageMap(message);
            //
            Notice notice = message.getNotice();
            String uid = notice.getUser().getUid();
            simpMessagingTemplate.convertAndSend(MqConsts.TOPIC_USER_PREFIX + uid, messageMap);

        } else if (message.getType().equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_GROUP_KICK)) {

            Map<String, String> messageMap = transformService.getUserGroupKickMessageMap(message);
            String uid = message.getCid();
            simpMessagingTemplate.convertAndSend(MqConsts.TOPIC_USER_PREFIX + uid, messageMap);

        } else if (message.getType().equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_GROUP_MUTE)) {

            Map<String, String> messageMap = transformService.getUserGroupMuteMessageMap(message);
            String uid = message.getCid();
            simpMessagingTemplate.convertAndSend(MqConsts.TOPIC_USER_PREFIX + uid, messageMap);

        } else if (message.getType().equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_GROUP_UNMUTE)) {

            Map<String, String> messageMap = transformService.getUserGroupUnMuteMessageMap(message);
            String uid = message.getCid();
            simpMessagingTemplate.convertAndSend(MqConsts.TOPIC_USER_PREFIX + uid, messageMap);

        } else if (message.getType().equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_GROUP_SET_ADMIN)) {

            Map<String, String> messageMap = transformService.getUserGroupSetAdminMessageMap(message);
            String uid = message.getCid();
            simpMessagingTemplate.convertAndSend(MqConsts.TOPIC_USER_PREFIX + uid, messageMap);

        } else if (message.getType().equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_GROUP_UNSET_ADMIN)) {

            Map<String, String> messageMap = transformService.getUserGroupUnSetAdminMessageMap(message);
            String uid = message.getCid();
            simpMessagingTemplate.convertAndSend(MqConsts.TOPIC_USER_PREFIX + uid, messageMap);

        } else if (message.getType().equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_GROUP_WITHDRAW)) {


        } else if (message.getType().equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_GROUP_DISMISS)) {


        } else if (message.getType().equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_WEBRTC_INVITE)) {


        } else if (message.getType().equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_WEBRTC_CANCEL)) {


        } else if (message.getType().equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_WEBRTC_OFFER_VIDEO)) {


        } else if (message.getType().equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_WEBRTC_OFFER_AUDIO)) {


        } else if (message.getType().equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_WEBRTC_ANSWER)) {


        } else if (message.getType().equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_WEBRTC_CANDIDATE)) {


        } else if (message.getType().equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_WEBRTC_ACCEPT)) {


        } else if (message.getType().equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_WEBRTC_REJECT)) {


        } else if (message.getType().equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_WEBRTC_READY)) {


        } else if (message.getType().equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_WEBRTC_BUSY)) {


        } else if (message.getType().equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_WEBRTC_CLOSE)) {


        } else {

            if (null != message.getThread() && null != message.getThread().getAgent()) {
                User agent = message.getThread().getAgent();
                simpMessagingTemplate.convertAndSend(MqConsts.TOPIC_USER_PREFIX + agent.getUid(), message);
            }
        }
    }


    /**
     * 监听来自rabbitmq广播的mqtt客户端发送的消息
     * FIXME: 对应来自mqtt客户端的消息Jackson2JsonMessageConverter, 报错：Could not convert incoming message with content-type [null], 'json' keyword missing
     *
     * {"client":"android","content":"3","localId":"8a81c9d6-becb-420e-a2b2-f68e1331c891",
     * "sessionType":"thread","status":"sending",
     * "tid":"201812071713051","type":"text","username":"201812051425471"}
     *
     * org.json.JSONObject messageObject = new org.json.JSONObject(mqttMessage);
     *
     * @param mqttMessage msg
     */
    @RabbitListener(queues = "#{mqttMessageQueue.name}")
    public void receiveMqttMessage(final String mqttMessage) {
        logger.info("receive from mqtt messageQueue: {}", mqttMessage);

        JSONObject jsonObject = JSON.parseObject(mqttMessage);

        String tid = jsonObject.getString("tid");
        String type = jsonObject.getString("type");
        String client = jsonObject.getString("client");
        String content  = jsonObject.getString("content");
        String username = jsonObject.getString("username");
        String status = jsonObject.getString("status");
        String localId = jsonObject.getString("localId");
        String sessionType = jsonObject.getString("sessionType");
        String version = jsonObject.getString("version");

        // type text, client android, content: 在？, username 201812302342591, tid 201901041851441, sessionType null, localId null, status null
        logger.info("type {}, client {}, content: {}, username {}, tid {}, sessionType {}, localId {}, status {}",
                type, client, content, username, tid, sessionType, localId, status);

        if (type == null) {
            return;
        }

        // 当前发消息用户
        Optional<User> userOptional = userRepository.findByUsername(username);
        //
        Message message = new Message();
        message.setType(type);
        message.setClient(client);
        message.setStatus(status);
        message.setLocalId(localId);
        message.setSessionType(sessionType);
        //
        if (type.equals(TypeConsts.MESSAGE_TYPE_TEXT)) {
            message.setContent(content);
        } else if (type.equals(TypeConsts.MESSAGE_TYPE_IMAGE)) {
            message.setImageUrl(content);
        }

        // 区分访客会话、联系人会话、群组会话
        messageService.routeMessage(sessionType, tid, message, userOptional.get());
    }

    /**
     * 消息回执：发送成功、送达、已读 等
     * @param mqttMessage msg
     */
    @RabbitListener(queues = "#{mqttReceiptQueue.name}")
    public void receiveMqttReceipt(final String mqttMessage) {
        logger.info("receive from mqtt receipt: {}", mqttMessage);

        JSONObject jsonObject = JSON.parseObject(mqttMessage);
        String clientId = jsonObject.getString("clientId");
        String mId = jsonObject.getString("mId");
        String status = jsonObject.getString("status");
        String version = jsonObject.getString("version");
        logger.info("clientId {}, mId {}, status  {}, version {}", clientId, mId, status, version);

        if (clientId == null) {
            return;
        }

        String [] clients = clientId.split("/");
        if (clients.length < 3) {
            logger.error("clientId {} 格式错误", clientId);
            return;
        }
        String username = clients[0];
        String client = clients[1];
        String uuid = clients[2];

        logger.info("username {}, client {}, uuid {}", username, client, uuid);

        //
        messageService.noticeReceipt(username, mId, status);

    }

    /**
     * WebRTC音视频
     * @param mqttMessage msg
     */
    @RabbitListener(queues = "#{mqttWebRTCQueue.name}")
    public void receiveMqttWebRTC(final String mqttMessage) {
        logger.info("receive from mqtt webrtc: {}", mqttMessage);

        JSONObject jsonObject = JSON.parseObject(mqttMessage);
        String clientId = jsonObject.getString("clientId");
        String localId = jsonObject.getString("localId");
        String sessionType = jsonObject.getString("sessionType");
        String type = jsonObject.getString("type");
        String uuid = jsonObject.getString("uuid");
        String content = jsonObject.getString("content");
        String version = jsonObject.getString("version");

        //
        if (clientId == null) {
            return;
        }

        String [] clients = clientId.split("/");
        if (clients.length < 3) {
            logger.error("clientId {} 格式错误", clientId);
            return;
        }
        String username = clients[0];
        String client = clients[1];
        String randomId = clients[2];

        logger.info("username {}, client {}, randomId {}", username, client, randomId);

        //
        if (sessionType.equals(TypeConsts.MESSAGE_SESSION_TYPE_THREAD)) {


        } else if (sessionType.equals(TypeConsts.MESSAGE_SESSION_TYPE_CONTACT)) {

            Optional<User> userOptional = userRepository.findByUsername(username);
            if (userOptional.isPresent()) {

                // TODO: 判断对方是否在线？

                // 插入聊天记录
                Message message = new Message();
                message.setClient(client);
                message.setType(type);
                message.setLocalId(localId);
                message.setSessionType(sessionType);

                if (type.equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_WEBRTC_INVITE)) {
                    message.setContent(userOptional.get().getNickname()+"邀请视频");
                } else if (type.equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_WEBRTC_OFFER_AUDIO) ||
                        type.equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_WEBRTC_OFFER_VIDEO) ||
                        type.equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_WEBRTC_ANSWER)) {
                    message.setSessionDescription(content);
                } else if (type.equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_WEBRTC_CANDIDATE)) {
                    message.setCandidate(content);
                } else if (type.equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_WEBRTC_CANCEL)) {
                    message.setContent(userOptional.get().getNickname()+"取消视频");
                } else if (type.equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_WEBRTC_REJECT)) {
                    message.setContent(userOptional.get().getNickname()+"拒绝视频");
                } else if (type.equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_WEBRTC_ACCEPT)) {
                    message.setContent(userOptional.get().getNickname()+"接受视频");
                } else if (type.equals(TypeConsts.MESSAGE_TYPE_NOTIFICATION_WEBRTC_CLOSE)) {
                    message.setContent(userOptional.get().getNickname()+"结束视频");
                }

                //
                messageService.routeContactMessage(uuid, message, userOptional.get());
            }

        } else if (sessionType.equals(TypeConsts.MESSAGE_SESSION_TYPE_GROUP)) {


        }
        
    }

    /**
     * 监听来自rabbitmq广播的mqtt客户端的在线状态
     * FIXME: 对应来自mqtt客户端的消息Jackson2JsonMessageConverter 报错：Could not convert incoming message with content-type [null], 'json' keyword missing
     *
     * @param mqttMessage msg
     */
    @RabbitListener(queues = "#{mqttStatusQueue.name}")
    public void receiveMqttStatus(final String mqttMessage) {
        logger.info("receive from mqtt statusQueue: {}", mqttMessage);

        JSONObject jsonObject = JSON.parseObject(mqttMessage);
        String clientId = jsonObject.getString("clientId");
        String status = jsonObject.getString("status");
        logger.info("clientId {}, status  {}", clientId, status);

        if (clientId == null) {
            return;
        }

        String [] clients = clientId.split("/");
        if (clients.length < 3) {
            logger.error("clientId {} 格式错误", clientId);
            return;
        }
        String username = clients[0];
        String client = clients[1];
        String uuid = clients[2];

        logger.info("username {}, client {}, uuid {}", username, client, uuid);

        Optional<User> userOptional = userRepository.findByUsername(username);
        if (userOptional.isPresent()) {

            statusService.saveStatus(status, client, userOptional.get());

            if (status.equals(StatusConsts.USER_STATUS_CONNECTED)) {

                // 更新redis内连线状态
                redisConnectService.updateConnectedStatus(userOptional.get());

                // 通知建立长连接
                messageService.notifyConnected(userOptional.get());

                // 通知其他客户端下线
                statusService.kickoff(userOptional.get(), clientId);

            } else if (status.equals(StatusConsts.USER_STATUS_DISCONNECTED)) {

                // 更新断开连线状态
                redisConnectService.updateDisconnectedStatus(userOptional.get());

                // 通知断开长连接
                messageService.notifyDisconnected(userOptional.get());

                // TODO: 通知其他客户端状态
            }

        } else {

            logger.warn("用户不存在，设置mqtt客户端状态");
        }
    }


    /**
     * mqtt客户端异常掉线监听
     * FIXME: 对应来自mqtt客户端的消息Jackson2JsonMessageConverter 报错：Could not convert incoming message with content-type [null], 'json' keyword missing
     *
     * { 'username': '201808231040571', 'client': 'ios', 'status': 'disconnect' }
     * @param mqttMessage msg
     */
    @RabbitListener(queues = "#{mqttLastWillQueue.name}")
    public void receiveMqttLastWill(final String mqttMessage) {
        logger.info("receive from mqtt lastWill: {}", mqttMessage);

        JSONObject jsonObject = JSON.parseObject(mqttMessage);
        String clientId = jsonObject.getString("clientId");
        String status = jsonObject.getString("status");
        logger.info("clientId {}, status  {}", clientId, status);

        if (clientId == null) {
            return;
        }

        String [] clients = clientId.split("/");
        if (clients.length < 3) {
            logger.error("clientId {} 格式错误", clientId);
            return;
        }
        String username = clients[0];
        String client = clients[1];
        String uuid = clients[2];

        logger.info("username {}, client {}, uuid {}", username, client, uuid);

        Optional<User> userOptional = userRepository.findByUsername(username);
        if (userOptional.isPresent()) {

            // 保存状态
            statusService.saveStatus(status, client, userOptional.get());

            // 更新断开连线状态
            redisConnectService.updateDisconnectedStatus(userOptional.get());

            // 通知断开长连接
            // FIXME: 强制断开（Android关闭模拟器），会循环收到离线消息通知
            messageService.notifyDisconnected(userOptional.get());

            // TODO: 通知其他客户端状态

            // TODO: 添加类型web browse功能通知客服前端

        } else {

            logger.info("用户不存在，设置mqtt客户端断开状态");
        }
    }


    /**
     * protobuf 消息
     * @param mqttMessage msg
     */
    @RabbitListener(queues = "#{mqttMessageProtobufQueue.name}")
    public void receiveMqttMessageProtobuf(final String mqttMessage) {
        logger.info("receive from mqtt messageProtobufQueue: {}", mqttMessage);
    }



}


