package io.xiaper.mq.stomp;

import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Configuration;
import org.springframework.messaging.simp.config.MessageBrokerRegistry;
import org.springframework.messaging.simp.stomp.StompReactorNettyCodec;
import org.springframework.messaging.tcp.reactor.ReactorNettyTcpClient;
import org.springframework.web.socket.config.annotation.EnableWebSocketMessageBroker;
import org.springframework.web.socket.config.annotation.StompEndpointRegistry;
import org.springframework.web.socket.config.annotation.WebSocketMessageBrokerConfigurer;
import org.springframework.web.socket.config.annotation.WebSocketTransportRegistration;

import java.net.InetSocketAddress;

/**
 * https://docs.spring.io/spring/docs/current/spring-framework-reference/web.html#websocket-stomp
 *
 * FIXME: Failed to write SockJsFrame content='a["\n"]'; nested exception is org.eclipse.jetty.io.EofException
 * FIXME: Caused by: org.eclipse.jetty.io.EofException: null / Caused by: java.io.IOException: Broken pipe
 *
 * @author xiaper.io
 */
@Configuration
@EnableWebSocketMessageBroker
public class StompConfig implements WebSocketMessageBrokerConfigurer {

//    private final Logger logger = LoggerFactory.getLogger(this.getClass());

    @Value("${spring.rabbitmq.host}")
    private String relayHost;

    @Value("${spring.rabbitmq.stomp.port}")
    private int relayPort;

    @Value("${spring.rabbitmq.username}")
    private String systemLogin;

    @Value("${spring.rabbitmq.password}")
    private String systemPasscode;

    @Value("${spring.rabbitmq.stomp.login}")
    private String clientLogin;

    @Value("${spring.rabbitmq.stomp.passcode}")
    private String clientPasscode;

//    @Value("${spring.rabbitmq.addresses}")
//    private String relayAddresses;

    /**
     * setAllowedOrigins 解决跨域问题
     * /stomp is the HTTP URL for the endpoint to which a WebSocket (or SockJS) client needs to connect for the WebSocket handshake.
     *
     * @param registry registry
     */
    @Override
    public void registerStompEndpoints(StompEndpointRegistry registry) {
        registry.addEndpoint("/stomp").setAllowedOrigins("*").withSockJS();
        registry.addEndpoint("/stomp/mini").addInterceptors(new StompHandshakeInterceptor()).setAllowedOrigins("*");
    }

    /**
     * SpringBoot官方说明 Using WebSocket to build an interactive web application:
     * https://spring.io/guides/gs/messaging-stomp-websocket/
     * https://www.sitepoint.com/implementing-spring-websocket-server-and-client/
     *
     * 图片说明：
     * https://docs.spring.io/spring/docs/current/spring-framework-reference/images/message-flow-broker-relay.png
     *
     * /topic/**: 广播
     * /queue/**: 点对点
     *
     * config.enableSimpleBroker("/topic","/queue");
     *
     * 说明：clientLogin/clientPasscode/systemLogin/systemPasscode 必须填写
     *
     * STOMP messages whose destination header begins with /app are routed to @MessageMapping methods in @Controller classes.
     *
     * 1. /app 开头的消息被路由到 @MessageMapping methods in annotated controllers,
     * 2. /topic 和 /queue 开头的消息被直接路由到 message broker（RabbitMQ/ActiveMQ)
     *
     * 集群部署方案：
     * 1. relayHost 指向 HAProxy
     * 2. 由 HAProxy 负责负载均衡分配 RabbitMQ
     *
     * 关于集群：
     * 1. https://github.com/rstoyanchev/spring-websocket-portfolio/issues/47
     * 2. https://github.com/spring-projects/spring-framework/issues/17057
     * 3. https://github.com/reactor/reactor/issues/412
     *
     * 静态连接一台broker：
     * .setRelayHost(relayHost).setRelayPort(relayPort)
     * .setClientLogin(clientLogin).setClientPasscode(clientPasscode)
     * .setSystemLogin(systemLogin).setSystemPasscode(systemPasscode);
     *
     * 动态配置多台broker：
     * .setTcpClient(createTcpClient());
     *
     * 支持ssl:
     * https://stackoverflow.com/questions/53105451/how-to-use-spring-messaging-5-1-to-connect-stompssl-broker
     *
     * @param config config
     */
    @Override
    public void configureMessageBroker(MessageBrokerRegistry config) {
        // 重要：不能用enableSimpleBroker，要使用Relay版本，方便集群
        config.enableStompBrokerRelay("/topic", "/queue")
            .setClientLogin(clientLogin).setClientPasscode(clientPasscode)
            .setSystemLogin(systemLogin).setSystemPasscode(systemPasscode)
            .setRelayHost(relayHost).setRelayPort(relayPort);
//            .setTcpClient(createTcpClient());
        //
        config.setApplicationDestinationPrefixes("/app");
        config.setUserDestinationPrefix("/user");
        // 保证消息有序性，启用有开销，必要时开启
        // config.setPreservePublishOrder(true);
    }

    /**
     * 用于优化 clientOutboundChannel 性能
     *
     * @param registration registration
     */
    @Override
    public void configureWebSocketTransport(WebSocketTransportRegistration registration) {
        registration.setSendTimeLimit(15 * 1000).setSendBufferSizeLimit(512 * 1024);
        registration.setMessageSizeLimit(128 * 1024);
    }


    /**
     * 断线重连到其他服务器
     * https://docs.spring.io/spring/docs/current/spring-framework-reference/web.html#websocket-stomp-handle-broker-relay-configure
     * 解决方案：
     * https://github.com/spring-projects/spring-framework/issues/21480
     *
     * TODO: 学习 java Supplier 用法
     *
     * @return client
     */
    private ReactorNettyTcpClient<byte[]> createTcpClient() {
        return new ReactorNettyTcpClient<>(client -> client.addressSupplier(() -> {

                    return new InetSocketAddress(relayHost, relayPort);

                }), new StompReactorNettyCodec());
    }

    /**
     * 拦截来自客户端的所有inbound消息
     * intercept inbound messages from clients:
     *
     * @param registration registration
     */
//    @Override
//    public void configureClientInboundChannel(ChannelRegistration registration) {
//        registration.interceptors(new StompChannelInterceptor());
//    }


}
