package io.xiaper.mq.stomp;

import io.xiaper.mq.service.MessageService;
import io.xiaper.mq.service.ThreadService;
import io.xiaper.jpa.model.Message;
import io.xiaper.jpa.model.User;
import io.xiaper.jpa.repository.MessageRepository;
import io.xiaper.jpa.repository.ThreadRepository;
import io.xiaper.jpa.repository.UserRepository;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.messaging.handler.annotation.*;
import org.springframework.messaging.simp.annotation.SendToUser;
import org.springframework.messaging.simp.annotation.SubscribeMapping;
import org.springframework.stereotype.Controller;

import java.security.Principal;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

/**
 * https://docs.spring.io/spring/docs/current/spring-framework-reference/web.html#websocket-stomp
 *
 * @author xiaper.io
 */
@Controller
public class StompController {

    private final static Logger logger = LoggerFactory.getLogger(StompController.class);

    @Autowired
    ThreadRepository threadRepository;

    @Autowired
    UserRepository userRepository;

    @Autowired
    MessageRepository messageRepository;

    @Autowired
    ThreadService threadService;

    @Autowired
    MessageService messageService;

    /**
     * 一对一聊天
     * https://docs.spring.io/spring/docs/current/spring-framework-reference/web.html#websocket-stomp-handle-annotations
     *
     * @param principal principal
     * @param message msg
     */
    @MessageMapping("/message")
    public void message(Principal principal, Message message) {
        logger.info("one message from:" + principal.getName() +" content:"+ message.toString());
        // 消息原路返回
        // FIXME: 通过RabbitMq发送
        // simpMessagingTemplate.convertAndSendToUser(principal.getName(), "/queue/message", message);
    }

    /**
     * 会话消息
     * stomp和mqtt客户端均能够收到消息:
     *
     * stomp端需要订阅,格式如：/topic/thread.123456 , 注意添加前缀 '/topic/'
     * stomp服务器发送消息：
     * String stompDestination = "/topic/"+ messageSessionType + "." + tid;
     * mqtt端需要订阅，格式如：thread/123456, 注意无前缀 '/topic/' ，同时 '.' 需要替换为 '/'
     * mqtt服务器发送消息：
     * String mqttDestination = "/topic/" + messageSessionType + "/" + tid;
     *
     * 其中chat为:
     * 工作组会话消息、同事消息、群组消息
     * thread/contact/group
     *
     * thread.tid
     * contact.tid
     * group.tid
     *
     * TODO: 收到消息之后优先转发消息，通过另外的Service服务监听MQ消息写库？
     *
     * @param principal principal
     * @param tid tid
     * @param message msg
     */
    @MessageMapping("/{sessionType}.{tid}")
    public void message(Principal principal,
                       @DestinationVariable(value = "sessionType") String sessionType,
                       @DestinationVariable(value = "tid") String tid,
                       Message message) {

        logger.info(sessionType + " message from:" + principal.getName() + " tid:" + tid + " content:" + message.toString());

        message.setSessionType(sessionType);

        Optional<User> userOptional = userRepository.findByUsername(principal.getName());

        messageService.routeMessage(sessionType, tid, message, userOptional.get());
    }

    /**
     * protobuf 消息体:
     * @see <a href="https://docs.spring.io/spring/docs/current/spring-framework-reference/web.html#websocket-stomp-handle-annotations"></a>
     *
     * @param principal principal
     * @param sessionType sessionType
     * @param tid tid
     */
    @MessageMapping("/protobuf.{sessionType}.{tid}")
    public void message(Principal principal,
                        @DestinationVariable(value = "sessionType") String sessionType,
                        @DestinationVariable(value = "tid") String tid) {

        // protobuf username: 270580156@qq.com sessionType: thread, tid 201902191652321, protobuf [49, 54, 44, 49]
        logger.info("protobuf username: {} sessionType: {}, tid {}", principal.getName(), sessionType, tid);

//        FIXME: 反序列化失败
//        try {
//            MessagesProtos.MyMessage myMessage = MessagesProtos.MyMessage.parseFrom((byte[]) message.getPayload());
//            logger.info("pageNumber {}", myMessage.getPageNumber());
//        } catch (InvalidProtocolBufferException e) {
//            e.printStackTrace();
//        }

    }


    /**
     * 订阅会话id，返回会话中的用户
     * 注意: 不能监听订阅 /topic/
     *
     * https://docs.spring.io/spring/docs/current/spring-framework-reference/web.html#websocket-stomp-subscribe-mapping
     *
     * @param principal principal
     * @param threadId tid
     * @return list
     */
    @SubscribeMapping("/thread.{threadId}")
    public List<User> subscribeThread(Principal principal, @DestinationVariable(value = "threadId") String threadId) {
        logger.info("subscribe from: " + principal.getName() + " threadId:" + threadId);
        // TODO: 从数据库中查询并返回
        return new ArrayList<>();
    }

    /**
     * 点对点推送
     *
     * @param text text
     * @param sessionId sessionId
     * @return string
     * @throws Exception
     */
    @MessageMapping(value = "/speak")
    @SendToUser(value = "/personal")
    public String speak(@Payload String text, @Header("simpSessionId") String sessionId) throws Exception {
        logger.info("收到私人消息:" + text);
        return text;
    }

    /**
     * 异常信息推送
     *
     * broadcast=false 说明：
     * If the user has more than one session, by default all of the sessions subscribed to the given destination are targeted.
     * However sometimes, it may be necessary to target only the session that sent the message being handled.
     * This can be done by setting the broadcast
     *
     * @param exception
     * @return
     */
    @MessageExceptionHandler
    @SendToUser(destinations="/queue/errors", broadcast=false)
    public String handleException(Throwable exception) {
        exception.printStackTrace();
        return exception.getMessage();
    }





}





