package io.xiaper.mq.stomp.listener;

import io.xiaper.jpa.constant.*;
import io.xiaper.jpa.model.*;
import io.xiaper.jpa.repository.*;
import io.xiaper.jpa.util.JpaUtil;
import io.xiaper.mq.service.BrowseService;
import io.xiaper.mq.service.MessageService;
import io.xiaper.mq.service.MessagingTemplate;
import io.xiaper.mq.service.StatusService;
import io.xiaper.mq.service.redis.RedisConnectService;
import io.xiaper.mq.service.redis.RedisStatisticService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationListener;
import org.springframework.messaging.MessageHeaders;
import org.springframework.messaging.simp.SimpMessageHeaderAccessor;
import org.springframework.stereotype.Component;
import org.springframework.web.socket.messaging.SessionDisconnectEvent;

import java.security.Principal;
import java.util.Iterator;
import java.util.List;
import java.util.Optional;

/**
 * published when a STOMP session ends.
 *
 * https://docs.spring.io/spring/docs/current/spring-framework-reference/web.html#websocket-stomp-appplication-context-events
 * @author xiaper.io
 */
@Component
public class StompDisconnectListener implements ApplicationListener<SessionDisconnectEvent> {

    private static final Logger logger = LoggerFactory.getLogger(StompDisconnectListener.class);

    @Autowired
    UserRepository userRepository;

    @Autowired
    StatusRepository statusRepository;

    @Autowired
    SubscribeRepository subscribeRepository;

    @Autowired
    MessageRepository messageRepository;

    @Autowired
    BrowseRepository browseRepository;

    @Autowired
    ThreadRepository threadRepository;

    @Autowired
    RoleRepository roleRepository;

    @Autowired
    QueueRepository queueRepository;

    @Autowired
    RedisConnectService redisConnectService;

    @Autowired
    RedisStatisticService redisStatisticService;

    @Autowired
    StatusService statusService;

    @Autowired
    MessageService messageService;

    @Autowired
    BrowseService browseService;

    @Autowired
    MessagingTemplate messagingTemplate;

    /**
     * 监听来自stomp，也即web端的连接断开信息
     * 注意：一个session可能会重复（多次）收到此断开消息：In some cases, this event is published more than once per session
     *
     * @param sessionDisconnectEvent
     */
    @Override
    public void onApplicationEvent(SessionDisconnectEvent sessionDisconnectEvent) {
        logger.info(sessionDisconnectEvent.toString());

        MessageHeaders headers = sessionDisconnectEvent.getMessage().getHeaders();
        Principal principal = SimpMessageHeaderAccessor.getUser(headers);
        if (principal == null) {
            return;
        }
        Optional<User> userOptional = userRepository.findByUsername(principal.getName());
        if (!userOptional.isPresent()) {
            return;
        }
        User user = userOptional.get();
        String sessionId = SimpMessageHeaderAccessor.getSessionId(headers);

        // 更新断开连线状态
        redisConnectService.updateDisconnectedStatus(user);

        // 通知离线
        messageService.notifyDisconnected(user, sessionId);

        // logger.info("sessionId: {}, uid: {}", sessionId, user.getUid());

        // 更新网页在线状态
        Optional<Browse> browseOptional = browseRepository.findFirstBySessionIdAndVisitor(sessionId, user);
        if (user.isVisitor() && browseOptional.isPresent()) {
            browseOptional.get().leave();
            browseRepository.save(browseOptional.get());

            // 通知客服端，访客已经关闭网页
            browseService.notifyBrowseEnd(user, browseOptional.get().getWorkGroup(), browseOptional.get());
        }

        // 如有排队，则标记排队离开，通知相关客服，并同步缓存
        if (user.isVisitor()) {
            List<Queue> queueList = queueRepository.findByVisitorAndStatus(user, StatusConsts.QUEUE_STATUS_QUEUING);
            Iterator iterator = queueList.iterator();
            while (iterator.hasNext()) {
                Queue queue = (Queue) iterator.next();
                queue.leave();
                queueRepository.save(queue);

                // 通知消息
                Message message = new Message();
                message.setMid(JpaUtil.randomId());
                message.setWid(queue.getWorkGroup().getWid());
                message.setThread(queue.getThread());
                message.setClient(ClientConsts.CLIENT_SYSTEM);
                message.setType(TypeConsts.MESSAGE_TYPE_NOTIFICATION_QUEUE_LEAVE);
                message.setContent(user.getNickname() + BdConstants.DEFAULT_QUEUE_LEAVE);

                Optional<User> userNotification = userRepository.findByUsername(UserConsts.USERNAME_NOTIFICATION);
                message.setUser(userNotification.get());
                messageRepository.save(message);

                //
//                rabbitTemplate.convertAndSend(MqConsts.EXCHANGE_DEFAULT_TOPIC_NAME, MqConsts.TOPIC_THREAD_MESSAGE, message);
                messagingTemplate.convertAndSend(MqConsts.TOPIC_THREAD_MESSAGE, message);

                // TODO: 有待优化
                Optional<Role> roleOptional = roleRepository.findByValue(RoleConsts.ROLE_ADMIN);
                if (roleOptional.isPresent()) {
                    Optional<User> adminOptional = userRepository.findBySubDomainAndRolesContains(user.getSubDomain(), roleOptional.get());
                    if (adminOptional.isPresent()) {
                        // 更新排队数
                        redisStatisticService.removeCurrentQueue(adminOptional.get().getSubDomain(), queue.getQid());
                        // 更新放弃排队的数量
                        redisStatisticService.increaseLeaveQueueCount(adminOptional.get().getSubDomain());
                    }
                }
            }
        }

//        // FIXME: 处理数据库中出现的大量同一时间的重复记录
//        Status status = new Status();
//        status.setStatus(StatusConsts.USER_STATUS_DISCONNECTED);
//        status.setSessionId(sessionId);
//        status.setUser(user);
//        // FIXME： 访客肯定是web端，对于客服有可能是web/mac/windows端，通过开放另外的接口解决
//        status.setClient(user.getClient());
//        statusRepository.save(status);

        // 在公司内部通知所有同事连接状态
        if (!user.isVisitor()) {
            statusService.notifyConnectionStatus(user, StatusConsts.USER_STATUS_DISCONNECTED);
        }

    }



    /**
     * 通知用户已经离开
     */
    private void notifyThreadDisconnect() {


    }


}

