package li.rudin.arduino.core.queue;

import java.util.HashMap;
import java.util.Map;

import li.rudin.arduino.api.Arduino;
import li.rudin.arduino.api.listener.ArduinoListener;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MessageQueue implements ArduinoListener
{
	/**
	 * Local logger
	 */
	private static final Logger logger = LoggerFactory.getLogger(MessageQueue.class);


	public MessageQueue(Arduino device)
	{
		logger.trace("Created new queue: {}", this);
		this.device = device;
	}

	private final Arduino device;

	@Override
	public void onMessageReceived(String key, String value)
	{
		ResultHolder result;
		synchronized(locks)
		{
			if (!locks.containsKey(key))
				return;

			result = locks.get(key);
		}
		synchronized(result)
		{
			logger.trace("Notifying: {}", result);
			result.setResult(value);
			result.notifyAll();
		}
	}

	/**
	 * Map of result holders
	 */
	private final Map<String, ResultHolder> locks = new HashMap<String, ResultHolder>();

	public String get(String key, String value, int timeout)
	{
		ResultHolder result;
		synchronized(locks)
		{
			result = locks.get(key);
			if (result == null)
			{
				result = new ResultHolder();
				locks.put(key, result);
			}
		}

		device.send(key, value);

		long timeoutExpires = System.currentTimeMillis() + timeout;

		synchronized(result)
		{
			while(true)
			{
				try
				{
					logger.trace("Waiting {} s for Result: {}", timeout, result);
					result.wait(timeout);
					//http://stackoverflow.com/questions/8178399/java-how-to-distinguish-between-spurious-wakeup-and-timeout-in-wait
					if(System.currentTimeMillis() >= timeoutExpires || result.getResult() != null)
						break;
				}
				catch (InterruptedException e)
				{
					return null;
				}
			}

			String ret = result.getResult();

			logger.trace("Got Result: {}", ret);

			return ret;
		}
	}

	private class ResultHolder
	{
		private String result;

		/**
		 * @return the result
		 */
		public String getResult()
		{
			return result;
		}

		/**
		 * @param result the result to set
		 */
		public void setResult(String result)
		{
			this.result = result;
		}
	}


	@Override
	public void onMessageTransmitted(String key, String value) {}



}
