package li.rudin.arduino.core.ethernet;

import java.io.IOException;
import java.io.OutputStream;
import java.net.InetSocketAddress;
import java.net.Socket;

import li.rudin.arduino.api.cache.KeyValueCache;
import li.rudin.arduino.api.ethernet.ArduinoEthernet;
import li.rudin.arduino.api.message.Message;
import li.rudin.arduino.api.state.ConnectionState;
import li.rudin.arduino.core.ObservableArduino;
import li.rudin.arduino.core.cache.CacheListener;
import li.rudin.arduino.core.cache.NoOpKeyValueCache;
import li.rudin.arduino.core.pool.ArduinoThreadPool;
import li.rudin.arduino.core.queue.MessageQueue;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ArduinoEthernetImpl extends ObservableArduino implements ArduinoEthernet
{
	/**
	 * Local logger
	 */
	private static final Logger logger = LoggerFactory.getLogger(ArduinoEthernetImpl.class);

	public ArduinoEthernetImpl(String host, int port)
	{
		this.host = host;
		this.port = port;
		addListener(queue);
		addListener(new CacheListener(this));
	}
	
	private final String host;
	private final int port;

	private Socket socket;
	private OutputStream output;

	private final MessageQueue queue = new MessageQueue(this);

	private ConnectionState current = ConnectionState.DISCONNECTED, target = ConnectionState.DISCONNECTED;

	@Override
	public String getHost()
	{
		return host;
	}

	@Override
	public int getPort()
	{
		return port;
	}

	@Override
	public ConnectionState getCurrentState()
	{
		return current;
	}

	@Override
	public ConnectionState getTargetState()
	{
		return target;
	}

	@Override
	public void setTargetState(final ConnectionState state)
	{
		final ConnectionState lastState = this.target;
		this.target = state;

		if (lastState == ConnectionState.DISCONNECTED && state == ConnectionState.CONNECTED)
		{
			//disconnected -> connected
			connect();
		}
		else if (lastState == ConnectionState.CONNECTED && state == ConnectionState.DISCONNECTED)
		{
			//connected -> disconnected
			disconnect();
		}
	}

	public void connect()
	{
		ArduinoThreadPool pool = ArduinoThreadPool.getInstance();
		
		logger.debug("Connecting to {}:{}", host, port);
		try
		{
			socket = new Socket();
			socket.setTcpNoDelay(true);
			socket.connect(new InetSocketAddress(host, port), 1000);

			current = ConnectionState.CONNECTED;
			output = socket.getOutputStream();
			pool.submit(new ReceiveListener(socket.getInputStream(), this));
		}
		catch (Exception e)
		{
			logger.debug("Connect error", e);
			disconnect();
			pool.submit(new ReconnectTimer(this));
		}
		
		fireStateChange();
	}

	public void disconnect()
	{
		logger.debug("Disconnecting from {}:{}", host, port);
		current = ConnectionState.DISCONNECTED;

		try
		{
			output = null;
			socket.close();
		}
		catch (Exception e)
		{
			logger.debug("disconnect", e);
		}
		
		fireStateChange();
	}
	
	/**
	 * Send lock
	 */
	private final Object sendLock = new Object();

	@Override
	public void send(String key, String value)
	{
		Message msg = new Message(key, value);
		
		logger.debug("Sending message: '{}'", msg);

		if (current == ConnectionState.CONNECTED && output != null)
		{
			try
			{
				synchronized(sendLock)
				{
					output.write( msg.getBytes() );
					output.flush();
				}
				
				this.fireTransmitted(key, value);

				logger.debug("Sent message: '{}'", msg);
			}
			catch (IOException e)
			{
				//Send failed
				logger.debug("send failed", e);
			}
		}
		else
		{
			logger.debug("Not Sending message (not connected): '{}'", msg);
		}
	}

	/**
	 * Returns the socket
	 * @return
	 */
	public Socket getSocket()
	{
		return socket;
	}

	@Override
	public String get(String key)
	{
		if (cache.isCached(key))
			return cache.get(key);
		
		return queue.get(key, "", 5000);
	}

	@Override
	public void setCache(KeyValueCache cache)
	{
		this.cache = cache;
	}
	
	/**
	 * The current cache
	 */
	private KeyValueCache cache = new NoOpKeyValueCache();

	@Override
	public KeyValueCache getCache()
	{
		return cache;
	}

}
