package li.rudin.arduino.core.ethernet;

import java.io.InputStream;

import li.rudin.arduino.api.message.Message;
import li.rudin.arduino.api.message.MessageParseResult;
import li.rudin.arduino.api.message.MessageParser;
import li.rudin.arduino.api.state.ConnectionState;
import li.rudin.arduino.core.pool.ArduinoThreadPool;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ReceiveListener implements Runnable
{
	/**
	 * Local logger
	 */
	private static final Logger logger = LoggerFactory.getLogger(ReceiveListener.class);
	
	
	public ReceiveListener(InputStream input, ArduinoEthernetImpl arduino)
	{
		this.input = input;
		this.arduino = arduino;
	}
	
	private final InputStream input;
	private final ArduinoEthernetImpl arduino;
	
	@Override
	public void run()
	{
		logger.debug("Entering rx-listener thread");
		try
		{
			final int BUFFER_SIZE = 4096;
			int bufferPos = 0;
			byte[] buffer = new byte[BUFFER_SIZE];
			
			while(arduino.getCurrentState() == ConnectionState.CONNECTED)
			{
				if (bufferPos >= BUFFER_SIZE)
				{
					//Reset buffer pos due to overflow
					logger.debug("Resetting bufferPos");
					bufferPos = 0;
				}
				
				int count = input.read(buffer, bufferPos, BUFFER_SIZE-bufferPos);
				
				if (count < 0)
					throw new IllegalArgumentException("No more data");
				
				logger.debug("Received {} bytes of data @ bufferPos {}", count, bufferPos);
				
				bufferPos += count;
				
				try
				{
					MessageParseResult result = MessageParser.parse(buffer, bufferPos);
					for (Message msg: result.getMessages())
					{
						logger.debug("Received message: {}", msg);
						arduino.fireReceived(msg.key, msg.value);
					}

					//Copy remaining message parts to beginning
					for (int i=0; i<bufferPos-result.getConsumedBytes(); i++)
						buffer[i] = buffer[i+result.getConsumedBytes()];
					
					bufferPos -= result.getConsumedBytes();
					
				}
				catch (Exception e)
				{
					logger.debug("Parser exception", e);
					bufferPos = 0;
				}
			}
		}
		catch (Exception e)
		{
			if (arduino.getTargetState() == ConnectionState.DISCONNECTED)
			{
				//Disconnecting
				logger.debug("Exiting rx-listener thread: manual disconnect");
				return;
			}
			
			logger.debug("Caught exception, reconnecting", e);
			arduino.disconnect();
			
			ArduinoThreadPool.getInstance().submit(new ReconnectTimer(arduino));
			
			logger.debug("Exiting rx-listener thread: connection error");
			return;
		}
		
	}

}
