/*
 * Decompiled with CFR 0.152.
 */
package org.springframework.cloud.sleuth.instrument.jdbc;

import com.zaxxer.hikari.HikariDataSource;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import javax.sql.DataSource;
import org.assertj.core.api.Assertions;
import org.junit.jupiter.api.Test;
import org.springframework.boot.test.context.assertj.AssertableApplicationContext;
import org.springframework.boot.test.context.runner.ApplicationContextRunner;
import org.springframework.cloud.sleuth.Tracer;
import org.springframework.cloud.sleuth.exporter.FinishedSpan;
import org.springframework.cloud.sleuth.instrument.jdbc.TraceJdbcEventListener;
import org.springframework.cloud.sleuth.instrument.jdbc.TraceQueryExecutionListener;
import org.springframework.cloud.sleuth.test.TestSpanHandler;
import org.springframework.context.annotation.Bean;

abstract class TracingListenerStrategyTests {
    public static final String SPAN_SQL_QUERY_TAG_NAME = "jdbc.query";
    public static final String SPAN_ROW_COUNT_TAG_NAME = "jdbc.row-count";

    TracingListenerStrategyTests() {
    }

    abstract ApplicationContextRunner parentContextRunner();

    protected abstract Class autoConfiguration();

    protected abstract Class testConfiguration();

    @Test
    void testShouldAddSpanForConnection() {
        this.parentContextRunner().run(context -> {
            DataSource dataSource = (DataSource)context.getBean(DataSource.class);
            TestSpanHandler spanReporter = (TestSpanHandler)context.getBean(TestSpanHandler.class);
            Connection connection = dataSource.getConnection();
            connection.commit();
            connection.rollback();
            connection.close();
            Assertions.assertThat(spanReporter.reportedSpans()).hasSize(1);
            FinishedSpan connectionSpan = spanReporter.reportedSpans().get(0);
            Assertions.assertThat((String)connectionSpan.getName()).isEqualTo("connection");
            Assertions.assertThat((String)connectionSpan.getRemoteServiceName()).isEqualTo("TESTDB-BAZ");
            Assertions.assertThat((Iterable)connectionSpan.getEvents()).extracting("value").contains(new Object[]{"jdbc.commit"});
            Assertions.assertThat((Iterable)connectionSpan.getEvents()).extracting("value").contains(new Object[]{"jdbc.rollback"});
            Assertions.assertThat((Object)((Tracer)context.getBean(Tracer.class)).currentSpan()).isNull();
        });
    }

    @Test
    void testShouldAddSpanForConnectionWithFixedRemoteServiceName() {
        ((ApplicationContextRunner)this.parentContextRunner().withPropertyValues(new String[]{"spring.datasource.url:jdbc:h2:mem:testdb-baz?sleuthServiceName=aaaabbbb"})).run(context -> {
            DataSource dataSource = (DataSource)context.getBean(DataSource.class);
            TestSpanHandler spanReporter = (TestSpanHandler)context.getBean(TestSpanHandler.class);
            Connection connection = dataSource.getConnection();
            connection.commit();
            connection.rollback();
            connection.close();
            Assertions.assertThat(spanReporter.reportedSpans()).hasSize(1);
            FinishedSpan connectionSpan = spanReporter.reportedSpans().get(0);
            Assertions.assertThat((String)connectionSpan.getName()).isEqualTo("connection");
            Assertions.assertThat((String)connectionSpan.getRemoteServiceName()).isEqualTo("aaaabbbb");
            Assertions.assertThat((Iterable)connectionSpan.getEvents()).extracting("value").contains(new Object[]{"jdbc.commit"});
            Assertions.assertThat((Iterable)connectionSpan.getEvents()).extracting("value").contains(new Object[]{"jdbc.rollback"});
            Assertions.assertThat((Object)((Tracer)context.getBean(Tracer.class)).currentSpan()).isNull();
        });
    }

    @Test
    void testShouldAddSpanForPreparedStatementExecute() {
        this.parentContextRunner().run(context -> {
            DataSource dataSource = (DataSource)context.getBean(DataSource.class);
            TestSpanHandler spanReporter = (TestSpanHandler)context.getBean(TestSpanHandler.class);
            Connection connection = dataSource.getConnection();
            connection.prepareStatement("SELECT NOW()").execute();
            connection.close();
            Assertions.assertThat(spanReporter.reportedSpans()).hasSize(2);
            FinishedSpan connectionSpan = spanReporter.reportedSpans().get(1);
            FinishedSpan statementSpan = spanReporter.reportedSpans().get(0);
            Assertions.assertThat((String)connectionSpan.getName()).isEqualTo("connection");
            Assertions.assertThat((String)statementSpan.getName()).isEqualTo("select");
            Assertions.assertThat((String)statementSpan.getRemoteServiceName()).isEqualTo("TESTDB-BAZ");
            Assertions.assertThat((Map)statementSpan.getTags()).containsEntry((Object)SPAN_SQL_QUERY_TAG_NAME, (Object)"SELECT NOW()");
            Assertions.assertThat((Object)((Tracer)context.getBean(Tracer.class)).currentSpan()).isNull();
        });
    }

    @Test
    void testShouldAddSpanForPreparedStatementExecuteUpdate() {
        this.parentContextRunner().run(context -> {
            DataSource dataSource = (DataSource)context.getBean(DataSource.class);
            TestSpanHandler spanReporter = (TestSpanHandler)context.getBean(TestSpanHandler.class);
            Connection connection = dataSource.getConnection();
            connection.prepareStatement("UPDATE INFORMATION_SCHEMA.TABLES SET table_Name = '' WHERE 0 = 1").executeUpdate();
            connection.close();
            Assertions.assertThat(spanReporter.reportedSpans()).hasSize(2);
            FinishedSpan connectionSpan = spanReporter.reportedSpans().get(1);
            FinishedSpan statementSpan = spanReporter.reportedSpans().get(0);
            Assertions.assertThat((String)connectionSpan.getName()).isEqualTo("connection");
            Assertions.assertThat((String)statementSpan.getName()).isEqualTo("update");
            Assertions.assertThat((Map)statementSpan.getTags()).containsEntry((Object)SPAN_SQL_QUERY_TAG_NAME, (Object)"UPDATE INFORMATION_SCHEMA.TABLES SET table_Name = '' WHERE 0 = 1");
            Assertions.assertThat((Map)statementSpan.getTags()).containsEntry((Object)SPAN_ROW_COUNT_TAG_NAME, (Object)"0");
            Assertions.assertThat((Object)((Tracer)context.getBean(Tracer.class)).currentSpan()).isNull();
        });
    }

    @Test
    void testShouldAddSpanForStatementExecuteUpdate() {
        this.parentContextRunner().run(context -> {
            DataSource dataSource = (DataSource)context.getBean(DataSource.class);
            TestSpanHandler spanReporter = (TestSpanHandler)context.getBean(TestSpanHandler.class);
            Connection connection = dataSource.getConnection();
            connection.createStatement().executeUpdate("UPDATE INFORMATION_SCHEMA.TABLES SET table_Name = '' WHERE 0 = 1");
            connection.close();
            Assertions.assertThat(spanReporter.reportedSpans()).hasSize(2);
            FinishedSpan connectionSpan = spanReporter.reportedSpans().get(1);
            FinishedSpan statementSpan = spanReporter.reportedSpans().get(0);
            Assertions.assertThat((String)connectionSpan.getName()).isEqualTo("connection");
            Assertions.assertThat((String)statementSpan.getName()).isEqualTo("update");
            Assertions.assertThat((Map)statementSpan.getTags()).containsEntry((Object)SPAN_SQL_QUERY_TAG_NAME, (Object)"UPDATE INFORMATION_SCHEMA.TABLES SET table_Name = '' WHERE 0 = 1");
            Assertions.assertThat((Map)statementSpan.getTags()).containsEntry((Object)SPAN_ROW_COUNT_TAG_NAME, (Object)"0");
            Assertions.assertThat((Object)((Tracer)context.getBean(Tracer.class)).currentSpan()).isNull();
        });
    }

    @Test
    void testShouldAddSpanForPreparedStatementExecuteQueryIncludingTimeToCloseResultSet() {
        this.parentContextRunner().run(context -> {
            DataSource dataSource = (DataSource)context.getBean(DataSource.class);
            TestSpanHandler spanReporter = (TestSpanHandler)context.getBean(TestSpanHandler.class);
            Connection connection = dataSource.getConnection();
            ResultSet resultSet = connection.prepareStatement("SELECT NOW() UNION ALL select NOW()").executeQuery();
            resultSet.next();
            resultSet.next();
            resultSet.close();
            connection.close();
            Assertions.assertThat(spanReporter.reportedSpans()).hasSize(3);
            FinishedSpan connectionSpan = spanReporter.reportedSpans().get(2);
            FinishedSpan resultSetSpan = spanReporter.reportedSpans().get(1);
            FinishedSpan statementSpan = spanReporter.reportedSpans().get(0);
            Assertions.assertThat((String)connectionSpan.getName()).isEqualTo("connection");
            Assertions.assertThat((String)statementSpan.getName()).isEqualTo("select");
            Assertions.assertThat((Map)statementSpan.getTags()).containsEntry((Object)SPAN_SQL_QUERY_TAG_NAME, (Object)"SELECT NOW() UNION ALL select NOW()");
            Assertions.assertThat((String)resultSetSpan.getName()).isEqualTo("result-set");
            Assertions.assertThat((String)resultSetSpan.getRemoteServiceName()).isEqualTo("TESTDB-BAZ");
            if (this.isP6Spy((AssertableApplicationContext)context)) {
                Assertions.assertThat((Map)resultSetSpan.getTags()).containsEntry((Object)SPAN_ROW_COUNT_TAG_NAME, (Object)"2");
            }
            Assertions.assertThat((Object)((Tracer)context.getBean(Tracer.class)).currentSpan()).isNull();
        });
    }

    @Test
    void testShouldAddSpanForStatementAndResultSet() {
        this.parentContextRunner().run(context -> {
            DataSource dataSource = (DataSource)context.getBean(DataSource.class);
            TestSpanHandler spanReporter = (TestSpanHandler)context.getBean(TestSpanHandler.class);
            Connection connection = dataSource.getConnection();
            ResultSet resultSet = connection.createStatement().executeQuery("SELECT NOW()");
            resultSet.next();
            Thread.sleep(200L);
            resultSet.close();
            connection.close();
            Assertions.assertThat(spanReporter.reportedSpans()).hasSize(3);
            FinishedSpan connectionSpan = spanReporter.reportedSpans().get(2);
            FinishedSpan resultSetSpan = spanReporter.reportedSpans().get(1);
            FinishedSpan statementSpan = spanReporter.reportedSpans().get(0);
            Assertions.assertThat((String)connectionSpan.getName()).isEqualTo("connection");
            Assertions.assertThat((String)statementSpan.getName()).isEqualTo("select");
            Assertions.assertThat((Map)statementSpan.getTags()).containsEntry((Object)SPAN_SQL_QUERY_TAG_NAME, (Object)"SELECT NOW()");
            Assertions.assertThat((String)resultSetSpan.getName()).isEqualTo("result-set");
            if (this.isP6Spy((AssertableApplicationContext)context)) {
                Assertions.assertThat((Map)resultSetSpan.getTags()).containsEntry((Object)SPAN_ROW_COUNT_TAG_NAME, (Object)"1");
            }
            Assertions.assertThat((Object)((Tracer)context.getBean(Tracer.class)).currentSpan()).isNull();
        });
    }

    @Test
    void testShouldNotFailWhenStatementIsClosedWihoutResultSet() {
        this.parentContextRunner().run(context -> {
            DataSource dataSource = (DataSource)context.getBean(DataSource.class);
            TestSpanHandler spanReporter = (TestSpanHandler)context.getBean(TestSpanHandler.class);
            Connection connection = dataSource.getConnection();
            Statement statement = connection.createStatement();
            ResultSet resultSet = statement.executeQuery("SELECT NOW()");
            resultSet.next();
            statement.close();
            connection.close();
            Assertions.assertThat(spanReporter.reportedSpans()).hasSize(3);
            FinishedSpan connectionSpan = spanReporter.reportedSpans().get(2);
            FinishedSpan resultSetSpan = spanReporter.reportedSpans().get(1);
            FinishedSpan statementSpan = spanReporter.reportedSpans().get(0);
            Assertions.assertThat((String)connectionSpan.getName()).isEqualTo("connection");
            Assertions.assertThat((String)statementSpan.getName()).isEqualTo("select");
            Assertions.assertThat((String)resultSetSpan.getName()).isEqualTo("result-set");
            Assertions.assertThat((Map)statementSpan.getTags()).containsEntry((Object)SPAN_SQL_QUERY_TAG_NAME, (Object)"SELECT NOW()");
            Assertions.assertThat((Object)((Tracer)context.getBean(Tracer.class)).currentSpan()).isNull();
        });
    }

    @Test
    void testShouldNotFailWhenConnectionIsClosedWihoutResultSet() {
        this.parentContextRunner().run(context -> {
            DataSource dataSource = (DataSource)context.getBean(DataSource.class);
            TestSpanHandler spanReporter = (TestSpanHandler)context.getBean(TestSpanHandler.class);
            Connection connection = dataSource.getConnection();
            Statement statement = connection.createStatement();
            ResultSet resultSet = statement.executeQuery("SELECT NOW()");
            resultSet.next();
            connection.close();
            Assertions.assertThat(spanReporter.reportedSpans()).hasSize(3);
            FinishedSpan connectionSpan = spanReporter.reportedSpans().get(2);
            FinishedSpan resultSetSpan = spanReporter.reportedSpans().get(1);
            FinishedSpan statementSpan = spanReporter.reportedSpans().get(0);
            Assertions.assertThat((String)connectionSpan.getName()).isEqualTo("connection");
            Assertions.assertThat((String)statementSpan.getName()).isEqualTo("select");
            Assertions.assertThat((String)resultSetSpan.getName()).isEqualTo("result-set");
            Assertions.assertThat((Map)statementSpan.getTags()).containsEntry((Object)SPAN_SQL_QUERY_TAG_NAME, (Object)"SELECT NOW()");
            Assertions.assertThat((Object)((Tracer)context.getBean(Tracer.class)).currentSpan()).isNull();
        });
    }

    @Test
    void testShouldNotFailWhenResultSetNextWasNotCalled() {
        this.parentContextRunner().run(context -> {
            DataSource dataSource = (DataSource)context.getBean(DataSource.class);
            TestSpanHandler spanReporter = (TestSpanHandler)context.getBean(TestSpanHandler.class);
            Connection connection = dataSource.getConnection();
            Statement statement = connection.createStatement();
            ResultSet resultSet = statement.executeQuery("SELECT NOW()");
            resultSet.next();
            resultSet.close();
            statement.close();
            connection.close();
            Assertions.assertThat(spanReporter.reportedSpans()).hasSize(3);
            FinishedSpan connectionSpan = spanReporter.reportedSpans().get(2);
            FinishedSpan resultSetSpan = spanReporter.reportedSpans().get(1);
            FinishedSpan statementSpan = spanReporter.reportedSpans().get(0);
            Assertions.assertThat((String)connectionSpan.getName()).isEqualTo("connection");
            Assertions.assertThat((String)statementSpan.getName()).isEqualTo("select");
            Assertions.assertThat((String)resultSetSpan.getName()).isEqualTo("result-set");
            Assertions.assertThat((Map)statementSpan.getTags()).containsEntry((Object)SPAN_SQL_QUERY_TAG_NAME, (Object)"SELECT NOW()");
            Assertions.assertThat((Object)((Tracer)context.getBean(Tracer.class)).currentSpan()).isNull();
        });
    }

    @Test
    void testShouldNotFailWhenResourceIsAlreadyClosed() {
        this.parentContextRunner().run(context -> {
            DataSource dataSource = (DataSource)context.getBean(DataSource.class);
            TestSpanHandler spanReporter = (TestSpanHandler)context.getBean(TestSpanHandler.class);
            Connection connection = dataSource.getConnection();
            Statement statement = connection.createStatement();
            ResultSet resultSet = statement.executeQuery("SELECT NOW()");
            resultSet.next();
            resultSet.close();
            resultSet.close();
            statement.close();
            statement.close();
            connection.close();
            connection.close();
            Assertions.assertThat(spanReporter.reportedSpans()).hasSize(3);
            FinishedSpan connectionSpan = spanReporter.reportedSpans().get(2);
            FinishedSpan resultSetSpan = spanReporter.reportedSpans().get(1);
            FinishedSpan statementSpan = spanReporter.reportedSpans().get(0);
            Assertions.assertThat((String)connectionSpan.getName()).isEqualTo("connection");
            Assertions.assertThat((String)statementSpan.getName()).isEqualTo("select");
            Assertions.assertThat((String)resultSetSpan.getName()).isEqualTo("result-set");
            Assertions.assertThat((Map)statementSpan.getTags()).containsEntry((Object)SPAN_SQL_QUERY_TAG_NAME, (Object)"SELECT NOW()");
            Assertions.assertThat((Object)((Tracer)context.getBean(Tracer.class)).currentSpan()).isNull();
        });
    }

    @Test
    void testShouldNotFailWhenResourceIsAlreadyClosed2() {
        this.parentContextRunner().run(context -> {
            DataSource dataSource = (DataSource)context.getBean(DataSource.class);
            TestSpanHandler spanReporter = (TestSpanHandler)context.getBean(TestSpanHandler.class);
            Connection connection = dataSource.getConnection();
            Assertions.assertThatThrownBy(() -> {
                connection.close();
                connection.prepareStatement("SELECT NOW()");
            }).isInstanceOf(SQLException.class);
            Assertions.assertThat(spanReporter.reportedSpans()).hasSize(1);
            FinishedSpan connectionSpan = spanReporter.reportedSpans().get(0);
            Assertions.assertThat((String)connectionSpan.getName()).isEqualTo("connection");
            Assertions.assertThat((Object)((Tracer)context.getBean(Tracer.class)).currentSpan()).isNull();
        });
    }

    @Test
    void testShouldNotFailWhenResourceIsAlreadyClosed3() {
        this.parentContextRunner().run(context -> {
            DataSource dataSource = (DataSource)context.getBean(DataSource.class);
            Connection connection = dataSource.getConnection();
            Statement statement = connection.createStatement();
            Assertions.assertThatThrownBy(() -> {
                statement.close();
                statement.executeQuery("SELECT NOW()");
            }).isInstanceOf(SQLException.class);
            connection.close();
            Assertions.assertThat((Object)((Tracer)context.getBean(Tracer.class)).currentSpan()).isNull();
        });
    }

    @Test
    void testShouldNotFailWhenResourceIsAlreadyClosed4() {
        this.parentContextRunner().run(context -> {
            DataSource dataSource = (DataSource)context.getBean(DataSource.class);
            Connection connection = dataSource.getConnection();
            Statement statement = connection.createStatement();
            ResultSet resultSet = statement.executeQuery("SELECT NOW()");
            Assertions.assertThatThrownBy(() -> {
                resultSet.close();
                resultSet.next();
            }).isInstanceOf(SQLException.class);
            statement.close();
            connection.close();
            Assertions.assertThat((Object)((Tracer)context.getBean(Tracer.class)).currentSpan()).isNull();
        });
    }

    @Test
    void testShouldNotFailToCloseSpanForTwoConsecutiveConnections() {
        this.parentContextRunner().run(context -> {
            DataSource dataSource = (DataSource)context.getBean(DataSource.class);
            TestSpanHandler spanReporter = (TestSpanHandler)context.getBean(TestSpanHandler.class);
            Connection connection1 = dataSource.getConnection();
            Connection connection2 = dataSource.getConnection();
            connection2.close();
            connection1.close();
            Assertions.assertThat(spanReporter.reportedSpans()).hasSize(2);
            FinishedSpan connection1Span = spanReporter.reportedSpans().get(0);
            FinishedSpan connection2Span = spanReporter.reportedSpans().get(1);
            Assertions.assertThat((String)connection1Span.getName()).isEqualTo("connection");
            Assertions.assertThat((String)connection2Span.getName()).isEqualTo("connection");
            Assertions.assertThat((Object)((Tracer)context.getBean(Tracer.class)).currentSpan()).isNull();
        });
    }

    @Test
    void testShouldNotFailWhenClosedInReversedOrder() {
        this.parentContextRunner().run(context -> {
            DataSource dataSource = (DataSource)context.getBean(DataSource.class);
            TestSpanHandler spanReporter = (TestSpanHandler)context.getBean(TestSpanHandler.class);
            Connection connection = dataSource.getConnection();
            Statement statement = connection.createStatement();
            ResultSet resultSet = statement.executeQuery("SELECT NOW()");
            resultSet.next();
            connection.close();
            statement.close();
            resultSet.close();
            Assertions.assertThat(spanReporter.reportedSpans()).hasSize(3);
            FinishedSpan connectionSpan = spanReporter.reportedSpans().get(2);
            FinishedSpan resultSetSpan = spanReporter.reportedSpans().get(1);
            FinishedSpan statementSpan = spanReporter.reportedSpans().get(0);
            Assertions.assertThat((String)connectionSpan.getName()).isEqualTo("connection");
            Assertions.assertThat((String)statementSpan.getName()).isEqualTo("select");
            Assertions.assertThat((String)resultSetSpan.getName()).isEqualTo("result-set");
            Assertions.assertThat((Map)statementSpan.getTags()).containsEntry((Object)SPAN_SQL_QUERY_TAG_NAME, (Object)"SELECT NOW()");
            Assertions.assertThat((Object)((Tracer)context.getBean(Tracer.class)).currentSpan()).isNull();
        });
    }

    @Test
    void testShouldNotCauseMemoryLeakOnTomcatPool() {
        ((ApplicationContextRunner)this.parentContextRunner().withPropertyValues(new String[]{"spring.datasource.type:org.apache.tomcat.jdbc.pool.DataSource"})).run(context -> {
            DataSource dataSource = (DataSource)context.getBean(DataSource.class);
            Object listener = this.isP6Spy((AssertableApplicationContext)context) ? context.getBean(TraceJdbcEventListener.class) : context.getBean(TraceQueryExecutionListener.class);
            Connection connection = dataSource.getConnection();
            Statement statement = connection.createStatement();
            ResultSet resultSet = statement.executeQuery("select 1 FROM dual");
            resultSet.next();
            resultSet.close();
            statement.close();
            connection.close();
            Assertions.assertThat((Object)listener).extracting("strategy").extracting("openConnections").isInstanceOfSatisfying(Map.class, map -> Assertions.assertThat((Map)map).isEmpty());
            Assertions.assertThat((Object)((Tracer)context.getBean(Tracer.class)).currentSpan()).isNull();
        });
    }

    private boolean isP6Spy(AssertableApplicationContext context) {
        if (context.getBeansOfType(TraceJdbcEventListener.class).size() == 1) {
            return true;
        }
        if (context.getBeansOfType(TraceQueryExecutionListener.class).size() == 1) {
            return false;
        }
        throw new IllegalStateException("Expected exactly 1 tracing listener bean in the context.");
    }

    @Test
    void testSingleConnectionAcrossMultipleThreads() {
        this.parentContextRunner().run(context -> {
            DataSource dataSource = (DataSource)context.getBean(DataSource.class);
            TestSpanHandler spanReporter = (TestSpanHandler)context.getBean(TestSpanHandler.class);
            Connection connection = dataSource.getConnection();
            IntStream.range(0, 5).mapToObj(i -> CompletableFuture.runAsync(() -> {
                try {
                    Statement statement = connection.createStatement();
                    ResultSet resultSet = statement.executeQuery("SELECT NOW()");
                    resultSet.next();
                    statement.close();
                    resultSet.close();
                }
                catch (SQLException e) {
                    throw new IllegalStateException(e);
                }
            })).collect(Collectors.toList()).forEach(CompletableFuture::join);
            connection.close();
            Assertions.assertThat(spanReporter.reportedSpans()).hasSize(11);
            Assertions.assertThat(spanReporter.reportedSpans()).extracting("name").contains(new Object[]{"select", "result-set", "connection"});
            Assertions.assertThat((Object)((Tracer)context.getBean(Tracer.class)).currentSpan()).isNull();
        });
    }

    @Test
    void testShouldIncludeOnlyConnectionTraces() {
        ((ApplicationContextRunner)this.parentContextRunner().withPropertyValues(new String[]{"spring.sleuth.jdbc.includes: connection"})).run(context -> {
            DataSource dataSource = (DataSource)context.getBean(DataSource.class);
            TestSpanHandler spanReporter = (TestSpanHandler)context.getBean(TestSpanHandler.class);
            Connection connection = dataSource.getConnection();
            Statement statement = connection.createStatement();
            ResultSet resultSet = statement.executeQuery("select 1 FROM dual");
            resultSet.next();
            resultSet.close();
            statement.close();
            connection.close();
            Assertions.assertThat(spanReporter.reportedSpans()).hasSize(1);
            FinishedSpan connectionSpan = spanReporter.reportedSpans().get(0);
            Assertions.assertThat((String)connectionSpan.getName()).isEqualTo("connection");
            Assertions.assertThat((Object)((Tracer)context.getBean(Tracer.class)).currentSpan()).isNull();
        });
    }

    @Test
    void testShouldIncludeOnlyQueryTraces() {
        ((ApplicationContextRunner)this.parentContextRunner().withPropertyValues(new String[]{"spring.sleuth.jdbc.includes: query"})).run(context -> {
            DataSource dataSource = (DataSource)context.getBean(DataSource.class);
            TestSpanHandler spanReporter = (TestSpanHandler)context.getBean(TestSpanHandler.class);
            Connection connection = dataSource.getConnection();
            Statement statement = connection.createStatement();
            ResultSet resultSet = statement.executeQuery("select 1 FROM dual");
            resultSet.next();
            resultSet.close();
            statement.close();
            connection.close();
            Assertions.assertThat(spanReporter.reportedSpans()).hasSize(1);
            FinishedSpan statementSpan = spanReporter.reportedSpans().get(0);
            Assertions.assertThat((String)statementSpan.getName()).isEqualTo("select");
            Assertions.assertThat((Object)((Tracer)context.getBean(Tracer.class)).currentSpan()).isNull();
        });
    }

    @Test
    void testShouldIncludeOnlyFetchTraces() {
        ((ApplicationContextRunner)this.parentContextRunner().withPropertyValues(new String[]{"spring.sleuth.jdbc.includes: fetch"})).run(context -> {
            DataSource dataSource = (DataSource)context.getBean(DataSource.class);
            TestSpanHandler spanReporter = (TestSpanHandler)context.getBean(TestSpanHandler.class);
            Connection connection = dataSource.getConnection();
            Statement statement = connection.createStatement();
            ResultSet resultSet = statement.executeQuery("select 1 FROM dual");
            resultSet.next();
            resultSet.close();
            statement.close();
            connection.close();
            Assertions.assertThat(spanReporter.reportedSpans()).hasSize(1);
            FinishedSpan resultSetSpan = spanReporter.reportedSpans().get(0);
            Assertions.assertThat((String)resultSetSpan.getName()).isEqualTo("result-set");
            Assertions.assertThat((Object)((Tracer)context.getBean(Tracer.class)).currentSpan()).isNull();
        });
    }

    @Test
    void testShouldIncludeOnlyConnectionAndQueryTraces() {
        ((ApplicationContextRunner)this.parentContextRunner().withPropertyValues(new String[]{"spring.sleuth.jdbc.includes: connection, query"})).run(context -> {
            DataSource dataSource = (DataSource)context.getBean(DataSource.class);
            TestSpanHandler spanReporter = (TestSpanHandler)context.getBean(TestSpanHandler.class);
            Connection connection = dataSource.getConnection();
            Statement statement = connection.createStatement();
            ResultSet resultSet = statement.executeQuery("select 1 FROM dual");
            resultSet.next();
            resultSet.close();
            statement.close();
            connection.close();
            Assertions.assertThat(spanReporter.reportedSpans()).hasSize(2);
            FinishedSpan connectionSpan = spanReporter.reportedSpans().get(1);
            FinishedSpan statementSpan = spanReporter.reportedSpans().get(0);
            Assertions.assertThat((String)connectionSpan.getName()).isEqualTo("connection");
            Assertions.assertThat((String)statementSpan.getName()).isEqualTo("select");
            Assertions.assertThat((Object)((Tracer)context.getBean(Tracer.class)).currentSpan()).isNull();
        });
    }

    @Test
    void testShouldIncludeOnlyConnectionAndFetchTraces() {
        ((ApplicationContextRunner)this.parentContextRunner().withPropertyValues(new String[]{"spring.sleuth.jdbc.includes: connection, fetch"})).run(context -> {
            DataSource dataSource = (DataSource)context.getBean(DataSource.class);
            TestSpanHandler spanReporter = (TestSpanHandler)context.getBean(TestSpanHandler.class);
            Connection connection = dataSource.getConnection();
            Statement statement = connection.createStatement();
            ResultSet resultSet = statement.executeQuery("select 1 FROM dual");
            resultSet.next();
            resultSet.close();
            statement.close();
            connection.close();
            Assertions.assertThat(spanReporter.reportedSpans()).hasSize(2);
            FinishedSpan connectionSpan = spanReporter.reportedSpans().get(1);
            FinishedSpan resultSetSpan = spanReporter.reportedSpans().get(0);
            Assertions.assertThat((String)connectionSpan.getName()).isEqualTo("connection");
            Assertions.assertThat((String)resultSetSpan.getName()).isEqualTo("result-set");
            Assertions.assertThat((Object)((Tracer)context.getBean(Tracer.class)).currentSpan()).isNull();
        });
    }

    @Test
    void testShouldIncludeOnlyQueryAndFetchTraces() {
        ((ApplicationContextRunner)this.parentContextRunner().withPropertyValues(new String[]{"spring.sleuth.jdbc.includes: query, fetch"})).run(context -> {
            DataSource dataSource = (DataSource)context.getBean(DataSource.class);
            TestSpanHandler spanReporter = (TestSpanHandler)context.getBean(TestSpanHandler.class);
            Connection connection = dataSource.getConnection();
            Statement statement = connection.createStatement();
            ResultSet resultSet = statement.executeQuery("select 1 FROM dual");
            resultSet.next();
            resultSet.close();
            statement.close();
            connection.close();
            Assertions.assertThat(spanReporter.reportedSpans()).hasSize(2);
            FinishedSpan resultSetSpan = spanReporter.reportedSpans().get(1);
            FinishedSpan statementSpan = spanReporter.reportedSpans().get(0);
            Assertions.assertThat((String)statementSpan.getName()).isEqualTo("select");
            Assertions.assertThat((String)resultSetSpan.getName()).isEqualTo("result-set");
            Assertions.assertThat((Object)((Tracer)context.getBean(Tracer.class)).currentSpan()).isNull();
        });
    }

    @Test
    void testShouldNotOverrideExceptionWhenConnectionWasClosedBeforeExecutingQuery() {
        this.parentContextRunner().run(context -> {
            DataSource dataSource = (DataSource)context.getBean(DataSource.class);
            TestSpanHandler spanReporter = (TestSpanHandler)context.getBean(TestSpanHandler.class);
            Connection connection = dataSource.getConnection();
            PreparedStatement statement = connection.prepareStatement("SELECT NOW()");
            connection.close();
            Assertions.assertThat((Object)((Tracer)context.getBean(Tracer.class)).currentSpan()).isNull();
            Assertions.assertThatThrownBy(statement::executeQuery).isInstanceOf(SQLException.class);
            Assertions.assertThat(spanReporter.reportedSpans()).hasSize(1);
            FinishedSpan connectionSpan = spanReporter.reportedSpans().get(0);
            Assertions.assertThat((String)connectionSpan.getName()).isEqualTo("connection");
            Assertions.assertThat((Object)((Tracer)context.getBean(Tracer.class)).currentSpan()).isNull();
        });
    }

    @Test
    void testShouldNotOverrideExceptionWhenStatementWasClosedBeforeExecutingQuery() {
        this.parentContextRunner().run(context -> {
            DataSource dataSource = (DataSource)context.getBean(DataSource.class);
            TestSpanHandler spanReporter = (TestSpanHandler)context.getBean(TestSpanHandler.class);
            Connection connection = dataSource.getConnection();
            PreparedStatement statement = connection.prepareStatement("SELECT NOW()");
            statement.close();
            Assertions.assertThatThrownBy(statement::executeQuery).isInstanceOf(SQLException.class);
            connection.close();
            Assertions.assertThat(spanReporter.reportedSpans()).hasSize(2);
            FinishedSpan connectionSpan = spanReporter.reportedSpans().get(1);
            FinishedSpan statementSpan = spanReporter.reportedSpans().get(0);
            Assertions.assertThat((String)connectionSpan.getName()).isEqualTo("connection");
            Assertions.assertThat((String)statementSpan.getName()).isEqualTo("select");
            Assertions.assertThat((Object)((Tracer)context.getBean(Tracer.class)).currentSpan()).isNull();
        });
    }

    @Test
    void testShouldNotOverrideExceptionWhenResultSetWasClosedBeforeNext() {
        this.parentContextRunner().run(context -> {
            DataSource dataSource = (DataSource)context.getBean(DataSource.class);
            TestSpanHandler spanReporter = (TestSpanHandler)context.getBean(TestSpanHandler.class);
            Connection connection = dataSource.getConnection();
            Statement statement = connection.createStatement();
            ResultSet resultSet = statement.executeQuery("SELECT NOW()");
            resultSet.close();
            Assertions.assertThatThrownBy(resultSet::next).isInstanceOf(SQLException.class);
            statement.close();
            connection.close();
            Assertions.assertThat(spanReporter.reportedSpans()).hasSize(3);
            FinishedSpan connectionSpan = spanReporter.reportedSpans().get(2);
            FinishedSpan resultSetSpan = spanReporter.reportedSpans().get(1);
            FinishedSpan statementSpan = spanReporter.reportedSpans().get(0);
            Assertions.assertThat((String)connectionSpan.getName()).isEqualTo("connection");
            Assertions.assertThat((String)statementSpan.getName()).isEqualTo("select");
            Assertions.assertThat((String)resultSetSpan.getName()).isEqualTo("result-set");
            Assertions.assertThat((Map)statementSpan.getTags()).containsEntry((Object)SPAN_SQL_QUERY_TAG_NAME, (Object)"SELECT NOW()");
            Assertions.assertThat((Object)((Tracer)context.getBean(Tracer.class)).currentSpan()).isNull();
        });
    }

    @Test
    void testShouldNotFailWhenClosingConnectionFromDifferentDataSource() {
        ApplicationContextRunner contextRunner = (ApplicationContextRunner)this.parentContextRunner().withUserConfiguration(new Class[]{MultiDataSourceConfiguration.class});
        contextRunner.run(context -> {
            DataSource dataSource1 = (DataSource)context.getBean("test1", DataSource.class);
            DataSource dataSource2 = (DataSource)context.getBean("test2", DataSource.class);
            dataSource1.getConnection().close();
            dataSource2.getConnection().close();
            CompletableFuture<Void> future = CompletableFuture.runAsync(() -> {
                try {
                    Connection connection1 = dataSource1.getConnection();
                    PreparedStatement statement = connection1.prepareStatement("SELECT NOW()");
                    ResultSet resultSet = statement.executeQuery();
                    Thread.sleep(200L);
                    resultSet.close();
                    statement.close();
                    connection1.close();
                }
                catch (InterruptedException | SQLException e) {
                    throw new IllegalStateException(e);
                }
            });
            Thread.sleep(100L);
            Connection connection2 = dataSource2.getConnection();
            Thread.sleep(300L);
            connection2.close();
            future.join();
        });
    }

    private static class MultiDataSourceConfiguration {
        private MultiDataSourceConfiguration() {
        }

        @Bean
        public HikariDataSource test1() {
            HikariDataSource dataSource = new HikariDataSource();
            dataSource.setJdbcUrl("jdbc:h2:mem:testdb-1-foo");
            dataSource.setPoolName("test1");
            return dataSource;
        }

        @Bean
        public HikariDataSource test2() {
            HikariDataSource dataSource = new HikariDataSource();
            dataSource.setJdbcUrl("jdbc:h2:mem:testdb-2-bar");
            dataSource.setPoolName("test2");
            return dataSource;
        }
    }
}

