package org.springframework.data.rest.webmvc;

import static org.springframework.data.rest.core.util.UriUtils.*;

import java.io.Serializable;
import java.net.URI;
import java.util.ArrayList;
import java.util.List;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.convert.ConversionService;
import org.springframework.core.convert.TypeDescriptor;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.mapping.PersistentProperty;
import org.springframework.data.mapping.model.BeanWrapper;
import org.springframework.data.repository.support.DomainClassConverter;
import org.springframework.data.repository.support.Repositories;
import org.springframework.data.rest.config.RepositoryRestConfiguration;
import org.springframework.data.rest.config.ResourceMapping;
import org.springframework.data.rest.repository.PageableResources;
import org.springframework.data.rest.repository.PersistentEntityResource;
import org.springframework.data.rest.repository.context.AfterCreateEvent;
import org.springframework.data.rest.repository.context.AfterDeleteEvent;
import org.springframework.data.rest.repository.context.AfterSaveEvent;
import org.springframework.data.rest.repository.context.BeforeCreateEvent;
import org.springframework.data.rest.repository.context.BeforeDeleteEvent;
import org.springframework.data.rest.repository.context.BeforeSaveEvent;
import org.springframework.data.rest.repository.invoke.RepositoryMethodInvoker;
import org.springframework.data.rest.repository.json.JsonSchema;
import org.springframework.data.rest.repository.json.PersistentEntityToJsonSchemaConverter;
import org.springframework.data.rest.repository.support.DomainObjectMerger;
import org.springframework.hateoas.EntityLinks;
import org.springframework.hateoas.Link;
import org.springframework.hateoas.Resource;
import org.springframework.hateoas.Resources;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.ResponseBody;

/**
 * @author Jon Brisbin
 */
@Controller
@RequestMapping("/{repository}")
public class RepositoryEntityController extends AbstractRepositoryRestController {

	@Autowired
	private DomainObjectMerger                    domainObjectMerger;
	@Autowired
	private PersistentEntityToJsonSchemaConverter jsonSchemaConverter;

	public RepositoryEntityController(Repositories repositories,
	                                  RepositoryRestConfiguration config,
	                                  DomainClassConverter domainClassConverter,
	                                  ConversionService conversionService,
	                                  EntityLinks entityLinks) {
		super(repositories,
		      config,
		      domainClassConverter,
		      conversionService,
		      entityLinks);
	}

	@RequestMapping(
			value = "/schema",
			method = RequestMethod.GET,
			produces = {
					"application/schema+json"
			}
	)
	@ResponseBody
	public JsonSchema schema(RepositoryRestRequest repoRequest) {
		return jsonSchemaConverter.convert(repoRequest.getPersistentEntity().getType());
	}

	@SuppressWarnings({"unchecked"})
	@RequestMapping(
			method = RequestMethod.GET,
			produces = {
					"application/json",
					"application/x-spring-data-verbose+json"
			}
	)
	@ResponseBody
	public Resources<Resource<?>> listEntities(final RepositoryRestRequest repoRequest)
			throws ResourceNotFoundException {
		List<Resource<?>> resources = new ArrayList<Resource<?>>();
		List<Link> links = new ArrayList<Link>();

		Iterable<?> results;
		RepositoryMethodInvoker repoMethodInvoker = repoRequest.getRepositoryMethodInvoker();
		if(null == repoMethodInvoker) {
			throw new ResourceNotFoundException();
		}
		boolean hasPagingParams = (null != repoRequest.getRequest().getParameter(config.getPageParamName()));
		boolean hasSortParams = (null != repoRequest.getRequest().getParameter(config.getSortParamName()));
		if(repoMethodInvoker.hasFindAllPageable() && hasPagingParams) {
			results = repoMethodInvoker.findAll(new PageRequest(repoRequest.getPagingAndSorting().getPageNumber(),
			                                                    repoRequest.getPagingAndSorting().getPageSize(),
			                                                    repoRequest.getPagingAndSorting().getSort()));
		} else if(repoMethodInvoker.hasFindAllSorted() && hasSortParams) {
			results = repoMethodInvoker.findAll(repoRequest.getPagingAndSorting().getSort());
		} else if(repoMethodInvoker.hasFindAll()) {
			results = repoMethodInvoker.findAll();
		} else {
			throw new ResourceNotFoundException();
		}

		for(Object o : results) {
			BeanWrapper wrapper = BeanWrapper.create(o, conversionService);
			Link selfLink = entityLinks.linkForSingleResource(repoRequest.getPersistentEntity().getType(),
			                                                  wrapper.getProperty(repoRequest.getPersistentEntity()
			                                                                                 .getIdProperty()))
			                           .withSelfRel();
			resources.add(new PersistentEntityResource<Object>(repoRequest.getPersistentEntity(),
			                                                   o,
			                                                   selfLink)
					              .setBaseUri(repoRequest.getBaseUri()));
		}


		if(!repoMethodInvoker.getQueryMethods().isEmpty()) {
			ResourceMapping repoMapping = repoRequest.getRepositoryResourceMapping();
			links.add(new Link(buildUri(repoRequest.getBaseUri(), repoMapping.getPath(), "search").toString(),
			                   repoMapping.getRel() + ".search"));
		}

		if(hasPagingParams || hasSortParams) {
			PageRequest pr = new PageRequest(repoRequest.getPagingAndSorting().getPageNumber() + 1,
			                                 repoRequest.getPagingAndSorting().getPageSize(),
			                                 repoRequest.getPagingAndSorting().getSort()) {
				@Override public int getOffset() {
					return super.getOffset() - repoRequest.getPagingAndSorting().getPageSize();
				}
			};
			return new PageableResources<Resource<?>>(resources, pr, links);
		} else {
			return new Resources<Resource<?>>(resources, links);
		}
	}

	@SuppressWarnings({"unchecked"})
	@RequestMapping(
			method = RequestMethod.GET,
			produces = {
					"application/x-spring-data-compact+json",
					"text/uri-list"
			}
	)
	@ResponseBody
	public Resources<Resource<?>> listEntitiesCompact(RepositoryRestRequest repoRequest)
			throws ResourceNotFoundException {
		Resources<Resource<?>> resources = listEntities(repoRequest);
		List<Link> links = new ArrayList<Link>(resources.getLinks());

		for(Resource<?> resource : resources.getContent()) {
			PersistentEntityResource<?> persistentEntityResource = (PersistentEntityResource<?>)resource;
			links.add(resourceLink(repoRequest, persistentEntityResource));
		}

		boolean hasPagingParams = (null != repoRequest.getRequest().getParameter(config.getPageParamName()));
		boolean hasSortParams = (null != repoRequest.getRequest().getParameter(config.getSortParamName()));
		if(hasPagingParams || hasSortParams) {
			return new PageableResources<Resource<?>>(EMPTY_RESOURCE_LIST, repoRequest.getPagingAndSorting(), links);
		} else {
			return new Resources<Resource<?>>(EMPTY_RESOURCE_LIST, links);
		}
	}

	@SuppressWarnings({"unchecked"})
	@RequestMapping(
			method = RequestMethod.POST,
			consumes = {
					"application/json"
			},
			produces = {
					"application/json",
					"text/uri-list"
			}
	)
	@ResponseBody
	public ResponseEntity<Resource<?>> createNewEntity(RepositoryRestRequest repoRequest,
	                                                   PersistentEntityResource<?> incoming) {
		RepositoryMethodInvoker repoMethodInvoker = repoRequest.getRepositoryMethodInvoker();
		if(null == repoMethodInvoker || !repoMethodInvoker.hasSaveOne()) {
			throw new NoSuchMethodError();
		}

		applicationContext.publishEvent(new BeforeCreateEvent(incoming.getContent()));
		Object obj = repoMethodInvoker.save(incoming.getContent());
		applicationContext.publishEvent(new AfterCreateEvent(obj));

		BeanWrapper wrapper = BeanWrapper.create(obj, conversionService);
		Link selfLink = entityLinks.linkForSingleResource(repoRequest.getPersistentEntity().getType(),
		                                                  wrapper.getProperty(repoRequest.getPersistentEntity()
		                                                                                 .getIdProperty()))
		                           .withSelfRel();
		HttpHeaders headers = new HttpHeaders();
		headers.setLocation(URI.create(selfLink.getHref()));

		if(config.isReturnBodyOnCreate()) {
			return resourceResponse(headers,
			                        new PersistentEntityResource<Object>(repoRequest.getPersistentEntity(),
			                                                             obj,
			                                                             selfLink)
					                        .setBaseUri(repoRequest.getBaseUri()),
			                        HttpStatus.CREATED);
		} else {
			return resourceResponse(headers,
			                        null,
			                        HttpStatus.CREATED);
		}
	}

	@SuppressWarnings({"unchecked"})
	@RequestMapping(
			value = "/{id}",
			method = RequestMethod.GET,
			produces = {
					"application/json",
					"application/x-spring-data-compact+json",
					"text/uri-list"
			}
	)
	@ResponseBody
	public Resource<?> getSingleEntity(RepositoryRestRequest repoRequest,
	                                   @PathVariable String id)
			throws ResourceNotFoundException {
		RepositoryMethodInvoker repoMethodInvoker = repoRequest.getRepositoryMethodInvoker();
		if(null == repoMethodInvoker || !repoMethodInvoker.hasFindOne()) {
			throw new ResourceNotFoundException();
		}

		Object domainObj = domainClassConverter.convert(id,
		                                                STRING_TYPE,
		                                                TypeDescriptor.valueOf(repoRequest.getPersistentEntity()
		                                                                                  .getType()));
		if(null == domainObj) {
			throw new ResourceNotFoundException();
		}

		PersistentEntityResource per = PersistentEntityResource.wrap(repoRequest.getPersistentEntity(),
		                                                             domainObj,
		                                                             repoRequest.getBaseUri());
		BeanWrapper wrapper = BeanWrapper.create(domainObj, conversionService);
		Link selfLink = entityLinks.linkForSingleResource(repoRequest.getPersistentEntity().getType(),
		                                                  wrapper.getProperty(repoRequest.getPersistentEntity()
		                                                                                 .getIdProperty()))
		                           .withSelfRel();
		per.add(selfLink);
		return per;
	}

	@SuppressWarnings({"unchecked"})
	@RequestMapping(
			value = "/{id}",
			method = RequestMethod.PUT,
			consumes = {
					"application/json"
			},
			produces = {
					"application/json",
					"text/uri-list"
			}
	)
	@ResponseBody
	public ResponseEntity<Resource<?>> updateEntity(RepositoryRestRequest repoRequest,
	                                                PersistentEntityResource<?> incoming,
	                                                @PathVariable String id)
			throws ResourceNotFoundException {
		RepositoryMethodInvoker repoMethodInvoker = repoRequest.getRepositoryMethodInvoker();
		if(null == repoMethodInvoker || !repoMethodInvoker.hasSaveOne() || !repoMethodInvoker.hasFindOne()) {
			throw new NoSuchMethodError();
		}

		Object domainObj = domainClassConverter.convert(id,
		                                                STRING_TYPE,
		                                                TypeDescriptor.valueOf(repoRequest.getPersistentEntity()
		                                                                                  .getType()));
		if(null == domainObj) {
			BeanWrapper incomingWrapper = BeanWrapper.create(incoming.getContent(), conversionService);
			PersistentProperty idProp = incoming.getPersistentEntity().getIdProperty();
			incomingWrapper.setProperty(idProp, conversionService.convert(id, idProp.getType()));
			return createNewEntity(repoRequest, incoming);
		}

		domainObjectMerger.merge(incoming.getContent(), domainObj);

		applicationContext.publishEvent(new BeforeSaveEvent(incoming.getContent()));
		Object obj = repoMethodInvoker.save(domainObj);
		applicationContext.publishEvent(new AfterSaveEvent(obj));

		if(config.isReturnBodyOnUpdate()) {
			PersistentEntityResource per = PersistentEntityResource.wrap(repoRequest.getPersistentEntity(),
			                                                             obj,
			                                                             repoRequest.getBaseUri());
			BeanWrapper wrapper = BeanWrapper.create(obj, conversionService);
			Link selfLink = entityLinks.linkForSingleResource(repoRequest.getPersistentEntity().getType(),
			                                                  wrapper.getProperty(repoRequest.getPersistentEntity()
			                                                                                 .getIdProperty()))
			                           .withSelfRel();
			per.add(selfLink);
			return resourceResponse(null,
			                        per,
			                        HttpStatus.OK);
		} else {
			return resourceResponse(null,
			                        null,
			                        HttpStatus.NO_CONTENT);
		}
	}

	@SuppressWarnings({"unchecked"})
	@RequestMapping(
			value = "/{id}",
			method = RequestMethod.DELETE
	)
	@ResponseBody
	public ResponseEntity<?> deleteEntity(RepositoryRestRequest repoRequest,
	                                      @PathVariable String id)
			throws ResourceNotFoundException {
		RepositoryMethodInvoker repoMethodInvoker = repoRequest.getRepositoryMethodInvoker();
		if(null == repoMethodInvoker || (!repoMethodInvoker.hasFindOne()
				&& !(repoMethodInvoker.hasDeleteOne() || repoMethodInvoker.hasDeleteOneById()))) {
			throw new NoSuchMethodError();
		}

		Object domainObj = domainClassConverter.convert(id,
		                                                STRING_TYPE,
		                                                TypeDescriptor.valueOf(repoRequest.getPersistentEntity()
		                                                                                  .getType()));
		if(null == domainObj) {
			throw new ResourceNotFoundException();
		}

		applicationContext.publishEvent(new BeforeDeleteEvent(domainObj));
		if(repoMethodInvoker.hasDeleteOneById()) {
			Class<? extends Serializable> idType = (Class<? extends Serializable>)repoRequest.getPersistentEntity()
			                                                                                 .getIdProperty()
			                                                                                 .getType();
			Object idVal = conversionService.convert(id, idType);
			repoMethodInvoker.delete((Serializable)idVal);
		} else if(repoMethodInvoker.hasDeleteOne()) {
			repoMethodInvoker.delete(domainObj);
		}
		applicationContext.publishEvent(new AfterDeleteEvent(domainObj));

		return new ResponseEntity<Object>(HttpStatus.NO_CONTENT);
	}

}
