/*
 * Decompiled with CFR 0.152.
 */
package org.springframework.session.data.redis;

import java.time.Duration;
import java.time.Instant;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.context.ApplicationEvent;
import org.springframework.context.ApplicationEventPublisher;
import org.springframework.data.redis.connection.Message;
import org.springframework.data.redis.connection.MessageListener;
import org.springframework.data.redis.core.BoundHashOperations;
import org.springframework.data.redis.core.RedisOperations;
import org.springframework.data.redis.serializer.JdkSerializationRedisSerializer;
import org.springframework.data.redis.serializer.RedisSerializer;
import org.springframework.expression.Expression;
import org.springframework.expression.spel.standard.SpelExpressionParser;
import org.springframework.session.FindByIndexNameSessionRepository;
import org.springframework.session.MapSession;
import org.springframework.session.Session;
import org.springframework.session.data.redis.RedisFlushMode;
import org.springframework.session.data.redis.RedisSessionExpirationPolicy;
import org.springframework.session.events.SessionCreatedEvent;
import org.springframework.session.events.SessionDeletedEvent;
import org.springframework.session.events.SessionExpiredEvent;
import org.springframework.util.Assert;

public class RedisOperationsSessionRepository
implements FindByIndexNameSessionRepository<RedisSession>,
MessageListener {
    private static final Log logger = LogFactory.getLog(RedisOperationsSessionRepository.class);
    private static final String SPRING_SECURITY_CONTEXT = "SPRING_SECURITY_CONTEXT";
    static PrincipalNameResolver PRINCIPAL_NAME_RESOLVER = new PrincipalNameResolver();
    public static final String DEFAULT_NAMESPACE = "spring:session";
    static final String CREATION_TIME_ATTR = "creationTime";
    static final String MAX_INACTIVE_ATTR = "maxInactiveInterval";
    static final String LAST_ACCESSED_ATTR = "lastAccessedTime";
    static final String SESSION_ATTR_PREFIX = "sessionAttr:";
    private String namespace = "spring:session:";
    private final RedisOperations<Object, Object> sessionRedisOperations;
    private final RedisSessionExpirationPolicy expirationPolicy;
    private ApplicationEventPublisher eventPublisher = new ApplicationEventPublisher(){

        public void publishEvent(ApplicationEvent event) {
        }

        public void publishEvent(Object event) {
        }
    };
    private Integer defaultMaxInactiveInterval;
    private RedisSerializer<Object> defaultSerializer = new JdkSerializationRedisSerializer();
    private RedisFlushMode redisFlushMode = RedisFlushMode.ON_SAVE;

    public RedisOperationsSessionRepository(RedisOperations<Object, Object> sessionRedisOperations) {
        Assert.notNull(sessionRedisOperations, (String)"sessionRedisOperations cannot be null");
        this.sessionRedisOperations = sessionRedisOperations;
        this.expirationPolicy = new RedisSessionExpirationPolicy(sessionRedisOperations, this::getExpirationsKey, this::getSessionKey);
    }

    public void setApplicationEventPublisher(ApplicationEventPublisher applicationEventPublisher) {
        Assert.notNull((Object)applicationEventPublisher, (String)"applicationEventPublisher cannot be null");
        this.eventPublisher = applicationEventPublisher;
    }

    public void setDefaultMaxInactiveInterval(int defaultMaxInactiveInterval) {
        this.defaultMaxInactiveInterval = defaultMaxInactiveInterval;
    }

    public void setDefaultSerializer(RedisSerializer<Object> defaultSerializer) {
        Assert.notNull(defaultSerializer, (String)"defaultSerializer cannot be null");
        this.defaultSerializer = defaultSerializer;
    }

    public void setRedisFlushMode(RedisFlushMode redisFlushMode) {
        Assert.notNull((Object)((Object)redisFlushMode), (String)"redisFlushMode cannot be null");
        this.redisFlushMode = redisFlushMode;
    }

    public RedisOperations<Object, Object> getSessionRedisOperations() {
        return this.sessionRedisOperations;
    }

    public void save(RedisSession session) {
        session.saveDelta();
        if (session.isNew()) {
            String sessionCreatedKey = this.getSessionCreatedChannel(session.getId());
            this.sessionRedisOperations.convertAndSend(sessionCreatedKey, (Object)session.delta);
            session.setNew(false);
        }
    }

    public void cleanupExpiredSessions() {
        this.expirationPolicy.cleanExpiredSessions();
    }

    public RedisSession findById(String id) {
        return this.getSession(id, false);
    }

    public Map<String, RedisSession> findByIndexNameAndIndexValue(String indexName, String indexValue) {
        if (!PRINCIPAL_NAME_INDEX_NAME.equals(indexName)) {
            return Collections.emptyMap();
        }
        String principalKey = this.getPrincipalKey(indexValue);
        Set sessionIds = this.sessionRedisOperations.boundSetOps((Object)principalKey).members();
        HashMap<String, RedisSession> sessions = new HashMap<String, RedisSession>(sessionIds.size());
        for (Object id : sessionIds) {
            RedisSession session = this.findById((String)id);
            if (session == null) continue;
            sessions.put(session.getId(), session);
        }
        return sessions;
    }

    private RedisSession getSession(String id, boolean allowExpired) {
        Map entries = this.getSessionBoundHashOperations(id).entries();
        if (entries.isEmpty()) {
            return null;
        }
        MapSession loaded = this.loadSession(id, entries);
        if (!allowExpired && loaded.isExpired()) {
            return null;
        }
        RedisSession result = new RedisSession(loaded);
        result.originalLastAccessTime = loaded.getLastAccessedTime();
        return result;
    }

    private MapSession loadSession(String id, Map<Object, Object> entries) {
        MapSession loaded = new MapSession(id);
        for (Map.Entry<Object, Object> entry : entries.entrySet()) {
            String key = (String)entry.getKey();
            if (CREATION_TIME_ATTR.equals(key)) {
                loaded.setCreationTime(Instant.ofEpochMilli((Long)entry.getValue()));
                continue;
            }
            if (MAX_INACTIVE_ATTR.equals(key)) {
                loaded.setMaxInactiveInterval(Duration.ofSeconds(((Integer)entry.getValue()).intValue()));
                continue;
            }
            if (LAST_ACCESSED_ATTR.equals(key)) {
                loaded.setLastAccessedTime(Instant.ofEpochMilli((Long)entry.getValue()));
                continue;
            }
            if (!key.startsWith(SESSION_ATTR_PREFIX)) continue;
            loaded.setAttribute(key.substring(SESSION_ATTR_PREFIX.length()), entry.getValue());
        }
        return loaded;
    }

    public void deleteById(String sessionId) {
        RedisSession session = this.getSession(sessionId, true);
        if (session == null) {
            return;
        }
        this.cleanupPrincipalIndex(session);
        this.expirationPolicy.onDelete(session);
        String expireKey = this.getExpiredKey(session.getId());
        this.sessionRedisOperations.delete((Object)expireKey);
        session.setMaxInactiveInterval(Duration.ZERO);
        this.save(session);
    }

    public RedisSession createSession() {
        RedisSession redisSession = new RedisSession();
        if (this.defaultMaxInactiveInterval != null) {
            redisSession.setMaxInactiveInterval(Duration.ofSeconds(this.defaultMaxInactiveInterval.intValue()));
        }
        return redisSession;
    }

    public void onMessage(Message message, byte[] pattern) {
        byte[] messageChannel = message.getChannel();
        byte[] messageBody = message.getBody();
        String channel = new String(messageChannel);
        if (channel.startsWith(this.getSessionCreatedChannelPrefix())) {
            Map loaded = (Map)this.defaultSerializer.deserialize(message.getBody());
            this.handleCreated(loaded, channel);
            return;
        }
        String body = new String(messageBody);
        if (!body.startsWith(this.getExpiredKeyPrefix())) {
            return;
        }
        boolean isDeleted = channel.endsWith(":del");
        if (isDeleted || channel.endsWith(":expired")) {
            int endIndex;
            int beginIndex = body.lastIndexOf(":") + 1;
            String sessionId = body.substring(beginIndex, endIndex = body.length());
            RedisSession session = this.getSession(sessionId, true);
            if (session == null) {
                logger.warn((Object)("Unable to publish SessionDestroyedEvent for session " + sessionId));
                return;
            }
            if (logger.isDebugEnabled()) {
                logger.debug((Object)("Publishing SessionDestroyedEvent for session " + sessionId));
            }
            this.cleanupPrincipalIndex(session);
            if (isDeleted) {
                this.handleDeleted(session);
            } else {
                this.handleExpired(session);
            }
        }
    }

    private void cleanupPrincipalIndex(RedisSession session) {
        String sessionId = session.getId();
        String principal = PRINCIPAL_NAME_RESOLVER.resolvePrincipal(session);
        if (principal != null) {
            this.sessionRedisOperations.boundSetOps((Object)this.getPrincipalKey(principal)).remove(new Object[]{sessionId});
        }
    }

    private void handleCreated(Map<Object, Object> loaded, String channel) {
        String id = channel.substring(channel.lastIndexOf(":") + 1);
        MapSession session = this.loadSession(id, loaded);
        this.publishEvent((ApplicationEvent)new SessionCreatedEvent((Object)this, (Session)session));
    }

    private void handleDeleted(RedisSession session) {
        this.publishEvent((ApplicationEvent)new SessionDeletedEvent((Object)this, (Session)session));
    }

    private void handleExpired(RedisSession session) {
        this.publishEvent((ApplicationEvent)new SessionExpiredEvent((Object)this, (Session)session));
    }

    private void publishEvent(ApplicationEvent event) {
        try {
            this.eventPublisher.publishEvent(event);
        }
        catch (Throwable ex) {
            logger.error((Object)("Error publishing " + event + "."), ex);
        }
    }

    public void setRedisKeyNamespace(String namespace) {
        Assert.hasText((String)namespace, (String)"namespace cannot be null or empty");
        this.namespace = namespace.trim() + ":";
    }

    String getSessionKey(String sessionId) {
        return this.namespace + "sessions:" + sessionId;
    }

    String getPrincipalKey(String principalName) {
        return this.namespace + "index:" + FindByIndexNameSessionRepository.PRINCIPAL_NAME_INDEX_NAME + ":" + principalName;
    }

    String getExpirationsKey(long expiration) {
        return this.namespace + "expirations:" + expiration;
    }

    private String getExpiredKey(String sessionId) {
        return this.getExpiredKeyPrefix() + sessionId;
    }

    private String getSessionCreatedChannel(String sessionId) {
        return this.getSessionCreatedChannelPrefix() + sessionId;
    }

    private String getExpiredKeyPrefix() {
        return this.namespace + "sessions:expires:";
    }

    public String getSessionCreatedChannelPrefix() {
        return this.namespace + "event:created:";
    }

    private BoundHashOperations<Object, Object, Object> getSessionBoundHashOperations(String sessionId) {
        String key = this.getSessionKey(sessionId);
        return this.sessionRedisOperations.boundHashOps((Object)key);
    }

    static String getSessionAttrNameKey(String attributeName) {
        return SESSION_ATTR_PREFIX + attributeName;
    }

    static class PrincipalNameResolver {
        private SpelExpressionParser parser = new SpelExpressionParser();

        PrincipalNameResolver() {
        }

        public String resolvePrincipal(Session session) {
            String principalName = (String)session.getAttribute(FindByIndexNameSessionRepository.PRINCIPAL_NAME_INDEX_NAME);
            if (principalName != null) {
                return principalName;
            }
            Object authentication = session.getAttribute(RedisOperationsSessionRepository.SPRING_SECURITY_CONTEXT);
            if (authentication != null) {
                Expression expression = this.parser.parseExpression("authentication?.name");
                return (String)expression.getValue(authentication, String.class);
            }
            return null;
        }
    }

    final class RedisSession
    implements Session {
        private final MapSession cached;
        private Instant originalLastAccessTime;
        private Map<String, Object> delta = new HashMap<String, Object>();
        private boolean isNew;
        private String originalPrincipalName;
        private String originalSessionId;

        RedisSession() {
            this(new MapSession());
            this.delta.put(RedisOperationsSessionRepository.CREATION_TIME_ATTR, this.getCreationTime().toEpochMilli());
            this.delta.put(RedisOperationsSessionRepository.MAX_INACTIVE_ATTR, (int)this.getMaxInactiveInterval().getSeconds());
            this.delta.put(RedisOperationsSessionRepository.LAST_ACCESSED_ATTR, this.getLastAccessedTime().toEpochMilli());
            this.isNew = true;
            this.flushImmediateIfNecessary();
        }

        RedisSession(MapSession cached) {
            Assert.notNull((Object)cached, (String)"MapSession cannot be null");
            this.cached = cached;
            this.originalPrincipalName = PRINCIPAL_NAME_RESOLVER.resolvePrincipal(this);
            this.originalSessionId = cached.getId();
        }

        public void setNew(boolean isNew) {
            this.isNew = isNew;
        }

        public void setLastAccessedTime(Instant lastAccessedTime) {
            this.cached.setLastAccessedTime(lastAccessedTime);
            this.putAndFlush(RedisOperationsSessionRepository.LAST_ACCESSED_ATTR, this.getLastAccessedTime().toEpochMilli());
        }

        public boolean isExpired() {
            return this.cached.isExpired();
        }

        public boolean isNew() {
            return this.isNew;
        }

        public Instant getCreationTime() {
            return this.cached.getCreationTime();
        }

        public String getId() {
            return this.cached.getId();
        }

        public String changeSessionId() {
            return this.cached.changeSessionId();
        }

        public Instant getLastAccessedTime() {
            return this.cached.getLastAccessedTime();
        }

        public void setMaxInactiveInterval(Duration interval) {
            this.cached.setMaxInactiveInterval(interval);
            this.putAndFlush(RedisOperationsSessionRepository.MAX_INACTIVE_ATTR, (int)this.getMaxInactiveInterval().getSeconds());
        }

        public Duration getMaxInactiveInterval() {
            return this.cached.getMaxInactiveInterval();
        }

        public <T> T getAttribute(String attributeName) {
            return (T)this.cached.getAttribute(attributeName);
        }

        public Set<String> getAttributeNames() {
            return this.cached.getAttributeNames();
        }

        public void setAttribute(String attributeName, Object attributeValue) {
            this.cached.setAttribute(attributeName, attributeValue);
            this.putAndFlush(RedisOperationsSessionRepository.getSessionAttrNameKey(attributeName), attributeValue);
        }

        public void removeAttribute(String attributeName) {
            this.cached.removeAttribute(attributeName);
            this.putAndFlush(RedisOperationsSessionRepository.getSessionAttrNameKey(attributeName), null);
        }

        private void flushImmediateIfNecessary() {
            if (RedisOperationsSessionRepository.this.redisFlushMode == RedisFlushMode.IMMEDIATE) {
                this.saveDelta();
            }
        }

        private void putAndFlush(String a, Object v) {
            this.delta.put(a, v);
            this.flushImmediateIfNecessary();
        }

        private void saveDelta() {
            String sessionId = this.getId();
            this.saveChangeSessionId(sessionId);
            if (this.delta.isEmpty()) {
                return;
            }
            RedisOperationsSessionRepository.this.getSessionBoundHashOperations(sessionId).putAll(this.delta);
            String principalSessionKey = RedisOperationsSessionRepository.getSessionAttrNameKey(FindByIndexNameSessionRepository.PRINCIPAL_NAME_INDEX_NAME);
            String securityPrincipalSessionKey = RedisOperationsSessionRepository.getSessionAttrNameKey(RedisOperationsSessionRepository.SPRING_SECURITY_CONTEXT);
            if (this.delta.containsKey(principalSessionKey) || this.delta.containsKey(securityPrincipalSessionKey)) {
                String principal;
                if (this.originalPrincipalName != null) {
                    String originalPrincipalRedisKey = RedisOperationsSessionRepository.this.getPrincipalKey(this.originalPrincipalName);
                    RedisOperationsSessionRepository.this.sessionRedisOperations.boundSetOps((Object)originalPrincipalRedisKey).remove(new Object[]{sessionId});
                }
                this.originalPrincipalName = principal = PRINCIPAL_NAME_RESOLVER.resolvePrincipal(this);
                if (principal != null) {
                    String principalRedisKey = RedisOperationsSessionRepository.this.getPrincipalKey(principal);
                    RedisOperationsSessionRepository.this.sessionRedisOperations.boundSetOps((Object)principalRedisKey).add(new Object[]{sessionId});
                }
            }
            this.delta = new HashMap<String, Object>(this.delta.size());
            Long originalExpiration = this.originalLastAccessTime == null ? null : Long.valueOf(this.originalLastAccessTime.plus(this.getMaxInactiveInterval()).toEpochMilli());
            RedisOperationsSessionRepository.this.expirationPolicy.onExpirationUpdated(originalExpiration, this);
        }

        private void saveChangeSessionId(String sessionId) {
            if (!this.isNew() && !sessionId.equals(this.originalSessionId)) {
                String originalSessionIdKey = RedisOperationsSessionRepository.this.getSessionKey(this.originalSessionId);
                String sessionIdKey = RedisOperationsSessionRepository.this.getSessionKey(sessionId);
                RedisOperationsSessionRepository.this.sessionRedisOperations.rename((Object)originalSessionIdKey, (Object)sessionIdKey);
                this.originalSessionId = sessionId;
            }
        }
    }
}

