/*
 * Copyright 2013-2019 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.springframework.cloud.contract.verifier

import java.nio.file.Files
import java.nio.file.Path
import java.nio.file.Paths
import java.nio.file.StandardOpenOption

import groovy.transform.CompileStatic
import groovy.transform.PackageScope
import groovy.util.logging.Commons

import org.springframework.cloud.contract.verifier.builder.SingleTestGenerator

import static org.springframework.cloud.contract.verifier.util.NamesUtil.beforeLast
import static org.springframework.cloud.contract.verifier.util.NamesUtil.capitalize
import static org.springframework.cloud.contract.verifier.util.NamesUtil.packageToDirectory

@CompileStatic
@PackageScope
@Commons
class FileSaver {

	private final File targetDirectory
	private final SingleTestGenerator generator
	private final String fileExtension

	FileSaver(File targetDirectory, String fileExtension, SingleTestGenerator generator) {
		this.targetDirectory = targetDirectory
		this.generator = generator
		this.fileExtension = fileExtension
	}

	void saveClassFile(Path classPath, byte[] classBytes) {
		log.info("Creating new class file [$classPath]")
		Files.
				write(classPath, classBytes, StandardOpenOption.CREATE, StandardOpenOption.TRUNCATE_EXISTING)
	}

	protected Path pathToClass(Path testBaseDir, String fileName) {
		return Paths.get(testBaseDir.toString(),
				capitalize(fileName) + fileExtension).toAbsolutePath()
	}

	protected Path generateTestBaseDir(String basePackageClass, String includedDirectoryRelativePath) {
		Path testBaseDir = Paths.
				get(targetDirectory.absolutePath, packageToDirectory(basePackageClass),
						beforeLast(includedDirectoryRelativePath, File.separator))
		Files.createDirectories(testBaseDir)
		return testBaseDir
	}

}
