/*
 * Decompiled with CFR 0.152.
 */
package org.cloudfoundry.util;

import java.io.IOException;
import java.io.InputStream;
import java.math.BigInteger;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.nio.file.attribute.FileTime;
import java.nio.file.attribute.PosixFileAttributes;
import java.nio.file.attribute.PosixFilePermission;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.Enumeration;
import java.util.Map;
import java.util.Optional;
import java.util.function.Predicate;
import java.util.stream.Stream;
import org.apache.commons.compress.archivers.ArchiveEntry;
import org.apache.commons.compress.archivers.zip.ZipArchiveEntry;
import org.apache.commons.compress.archivers.zip.ZipArchiveOutputStream;
import org.apache.commons.compress.archivers.zip.ZipFile;
import org.cloudfoundry.util.ByteArrayPool;
import org.cloudfoundry.util.FluentMap;
import reactor.core.Exceptions;
import reactor.core.publisher.Mono;
import reactor.core.scheduler.Schedulers;

public final class FileUtils {
    private static final Integer DEFAULT_PERMISSIONS = 484;
    private static final int MIBIBYTE = 0x100000;
    private static final Map<PosixFilePermission, Integer> PERMISSION_MODES = FluentMap.builder().entry(PosixFilePermission.OWNER_READ, 256).entry(PosixFilePermission.OWNER_WRITE, 128).entry(PosixFilePermission.OWNER_EXECUTE, 64).entry(PosixFilePermission.GROUP_READ, 32).entry(PosixFilePermission.GROUP_WRITE, 16).entry(PosixFilePermission.GROUP_EXECUTE, 8).entry(PosixFilePermission.OTHERS_READ, 4).entry(PosixFilePermission.OTHERS_WRITE, 2).entry(PosixFilePermission.OTHERS_EXECUTE, 1).build();

    private FileUtils() {
    }

    public static Mono<Path> compress(Path candidate) {
        return FileUtils.compress(candidate, path -> true);
    }

    public static Mono<Path> compress(Path candidate, Predicate<String> filter) {
        return Mono.defer(() -> {
            try {
                Path staging = Files.createTempFile(String.format("resource-matched-%s-", candidate.getFileName()), ".zip", new FileAttribute[0]);
                try (ZipArchiveOutputStream out = new ZipArchiveOutputStream(staging.toFile());){
                    if (Files.isDirectory(candidate, new LinkOption[0])) {
                        FileUtils.compressFromDirectory(candidate, filter, out);
                    } else {
                        FileUtils.compressFromZip(candidate, filter, out);
                    }
                }
                return Mono.just((Object)staging);
            }
            catch (IOException e) {
                throw Exceptions.propagate((Throwable)e);
            }
        }).subscribeOn(Schedulers.elastic());
    }

    public static String getRelativePathName(Path root, Path path) {
        Path relative = root.relativize(path);
        return Files.isDirectory(path, new LinkOption[0]) && !relative.toString().endsWith("/") ? String.format("%s/", relative.toString()) : relative.toString();
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public static String hash(Path path) {
        try (InputStream in = Files.newInputStream(path, new OpenOption[0]);){
            String string = FileUtils.hash(in);
            return string;
        }
        catch (IOException e) {
            throw Exceptions.propagate((Throwable)e);
        }
    }

    public static String hash(InputStream in) {
        try {
            MessageDigest digest = MessageDigest.getInstance("sha1");
            ByteArrayPool.withByteArray(buffer -> {
                try {
                    int length;
                    while ((length = in.read((byte[])buffer)) != -1) {
                        digest.update((byte[])buffer, 0, length);
                    }
                }
                catch (IOException e) {
                    throw Exceptions.propagate((Throwable)e);
                }
            });
            return String.format("%040x", new BigInteger(1, digest.digest()));
        }
        catch (NoSuchAlgorithmException e) {
            throw Exceptions.propagate((Throwable)e);
        }
    }

    public static String permissions(Path path) {
        try {
            return FileUtils.permissions(FileUtils.getUnixMode(path));
        }
        catch (IOException e) {
            throw Exceptions.propagate((Throwable)e);
        }
    }

    public static String permissions(int mode) {
        return Integer.toOctalString(mode == 0 ? DEFAULT_PERMISSIONS : mode);
    }

    public static int size(Path path) {
        try {
            return (int)Files.size(path);
        }
        catch (IOException e) {
            throw Exceptions.propagate((Throwable)e);
        }
    }

    private static void compressFromDirectory(Path candidate, Predicate<String> filter, ZipArchiveOutputStream out) {
        try (Stream<Path> contents = Files.walk(candidate, new FileVisitOption[0]);){
            contents.filter(path -> {
                try {
                    return !Files.isSameFile(candidate, path);
                }
                catch (IOException e) {
                    throw Exceptions.propagate((Throwable)e);
                }
            }).filter(path -> filter.test(FileUtils.getRelativePathName(candidate, path))).forEach(path -> {
                try (InputStream in = Files.isDirectory(path, new LinkOption[0]) ? null : Files.newInputStream(path, new OpenOption[0]);){
                    FileUtils.write(in, Files.getLastModifiedTime(path, new LinkOption[0]), FileUtils.getUnixMode(path), out, FileUtils.getRelativePathName(candidate, path));
                }
                catch (IOException e) {
                    throw Exceptions.propagate((Throwable)e);
                }
            });
        }
        catch (IOException e) {
            throw Exceptions.propagate((Throwable)e);
        }
    }

    private static void compressFromZip(Path candidate, Predicate<String> filter, ZipArchiveOutputStream out) {
        try (ZipFile zipFile = new ZipFile(candidate.toFile());){
            Enumeration entries = zipFile.getEntries();
            while (entries.hasMoreElements()) {
                ZipArchiveEntry entry = (ZipArchiveEntry)entries.nextElement();
                if (!filter.test(entry.getName())) continue;
                InputStream in = zipFile.getInputStream(entry);
                Throwable throwable = null;
                try {
                    int mode = entry.getUnixMode();
                    FileUtils.write(in, entry.getLastModifiedTime(), mode == 0 ? DEFAULT_PERMISSIONS : mode, out, entry.getName());
                }
                catch (Throwable throwable2) {
                    throwable = throwable2;
                    throw throwable2;
                }
                finally {
                    if (in == null) continue;
                    if (throwable != null) {
                        try {
                            in.close();
                        }
                        catch (Throwable throwable3) {
                            throwable.addSuppressed(throwable3);
                        }
                        continue;
                    }
                    in.close();
                }
            }
        }
        catch (IOException e) {
            throw Exceptions.propagate((Throwable)e);
        }
    }

    private static int getUnixMode(Path path) throws IOException {
        return Optional.ofNullable(Files.readAttributes(path, PosixFileAttributes.class, new LinkOption[0])).map(attributes -> attributes.permissions().stream().map(PERMISSION_MODES::get).mapToInt(i -> i).sum()).orElse(DEFAULT_PERMISSIONS);
    }

    private static void write(InputStream in, FileTime lastModifiedTime, int mode, ZipArchiveOutputStream out, String path) {
        try {
            ZipArchiveEntry entry = new ZipArchiveEntry(path);
            entry.setUnixMode(mode);
            entry.setLastModifiedTime(lastModifiedTime);
            out.putArchiveEntry((ArchiveEntry)entry);
            if (in != null) {
                ByteArrayPool.withByteArray(buffer -> {
                    try {
                        int length;
                        while ((length = in.read((byte[])buffer)) != -1) {
                            out.write(buffer, 0, length);
                        }
                    }
                    catch (IOException e) {
                        throw Exceptions.propagate((Throwable)e);
                    }
                });
            }
            out.closeArchiveEntry();
        }
        catch (IOException e) {
            throw Exceptions.propagate((Throwable)e);
        }
    }
}

