package me.alexisevelyn.randomtech.api.blocks.cables;

import com.google.common.collect.Maps;
import me.alexisevelyn.randomtech.api.Main;
import me.alexisevelyn.randomtech.api.utilities.CalculationHelper;
import me.alexisevelyn.randomtech.api.utilities.pathfinding.dijkstra.*;
import net.minecraft.block.*;
import net.minecraft.class_1268;
import net.minecraft.class_1269;
import net.minecraft.class_1309;
import net.minecraft.class_156;
import net.minecraft.class_1657;
import net.minecraft.class_1799;
import net.minecraft.class_1802;
import net.minecraft.class_1922;
import net.minecraft.class_1936;
import net.minecraft.class_1937;
import net.minecraft.class_2248;
import net.minecraft.class_2338;
import net.minecraft.class_2350;
import net.minecraft.class_2382;
import net.minecraft.class_2429;
import net.minecraft.class_2561;
import net.minecraft.class_2588;
import net.minecraft.class_265;
import net.minecraft.class_2680;
import net.minecraft.class_2689;
import net.minecraft.class_2741;
import net.minecraft.class_2746;
import net.minecraft.class_2754;
import net.minecraft.class_2769;
import net.minecraft.class_3610;
import net.minecraft.class_3612;
import net.minecraft.class_3726;
import net.minecraft.class_3737;
import net.minecraft.class_3965;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.*;
import java.util.function.Consumer;

/**
 * The type Generic cable.
 */
public abstract class GenericCable extends class_2429 implements class_3737 {
    public static class_2754<CableConnection> CABLE_CONNECTION_NORTH = class_2754.method_11849("north", CableConnection.class, CableConnection.NONE, CableConnection.CABLE, CableConnection.INTERFACEABLE);
    public static class_2754<CableConnection> CABLE_CONNECTION_SOUTH = class_2754.method_11849("south", CableConnection.class, CableConnection.NONE, CableConnection.CABLE, CableConnection.INTERFACEABLE);
    public static class_2754<CableConnection> CABLE_CONNECTION_EAST = class_2754.method_11849("east", CableConnection.class, CableConnection.NONE, CableConnection.CABLE, CableConnection.INTERFACEABLE);
    public static class_2754<CableConnection> CABLE_CONNECTION_WEST = class_2754.method_11849("west", CableConnection.class, CableConnection.NONE, CableConnection.CABLE, CableConnection.INTERFACEABLE);
    public static class_2754<CableConnection> CABLE_CONNECTION_UP = class_2754.method_11849("up", CableConnection.class, CableConnection.NONE, CableConnection.CABLE, CableConnection.INTERFACEABLE);
    public static class_2754<CableConnection> CABLE_CONNECTION_DOWN = class_2754.method_11849("down", CableConnection.class, CableConnection.NONE, CableConnection.CABLE, CableConnection.INTERFACEABLE);

    // Allows to be Waterlogged
    public static final class_2746 WATERLOGGED = class_2741.field_12508;

    private static final class_2382 northVector = class_2350.field_11043.method_10163();
    private static final class_2382 southVector = class_2350.field_11035.method_10163();
    private static final class_2382 eastVector = class_2350.field_11034.method_10163();
    private static final class_2382 westVector = class_2350.field_11039.method_10163();
    private static final class_2382 upVector = class_2350.field_11036.method_10163();
    private static final class_2382 downVector = class_2350.field_11033.method_10163();

    private static final Map<class_2350, class_2754<CableConnection>> FACING_PROPERTIES = class_156.method_654(Maps.newEnumMap(class_2350.class), (enumMap) -> {
        enumMap.put(class_2350.field_11043, CABLE_CONNECTION_NORTH);
        enumMap.put(class_2350.field_11034, CABLE_CONNECTION_EAST);
        enumMap.put(class_2350.field_11035, CABLE_CONNECTION_SOUTH);
        enumMap.put(class_2350.field_11039, CABLE_CONNECTION_WEST);
        enumMap.put(class_2350.field_11036, CABLE_CONNECTION_UP);
        enumMap.put(class_2350.field_11033, CABLE_CONNECTION_DOWN);
    });

    public final int maxCount = 5000; // Integer.MAX_VALUE;

    /**
     *
     * @param radius
     * @param settings
     */
    public GenericCable(float radius, @NotNull class_2251 settings) {
        super(radius, settings);

        this.method_9590(this.field_10647.method_11664()
                .method_11657(CABLE_CONNECTION_NORTH, CableConnection.NONE)
                .method_11657(CABLE_CONNECTION_SOUTH, CableConnection.NONE)
                .method_11657(CABLE_CONNECTION_EAST, CableConnection.NONE)
                .method_11657(CABLE_CONNECTION_WEST, CableConnection.NONE)
                .method_11657(CABLE_CONNECTION_UP, CableConnection.NONE)
                .method_11657(CABLE_CONNECTION_DOWN, CableConnection.NONE)
                .method_11657(WATERLOGGED, false)
        );
    }

    @Override
    public int method_10368(class_2680 state) {
        int i = 0;

        for(int j = 0; j < class_2350.values().length; ++j) {
            if (isConnected(state, class_2350.values()[j])) {
                i |= 1 << j;
            }
        }

        return i;
    }

    @Override
    public class_265 method_9549(class_2680 state, class_1922 world, class_2338 pos, class_3726 context) {
        return this.method_9530(state, world, pos, context);
    }

    /**
     * Makes it where light levels aren't blocked
     *
     * @param state
     * @param world
     * @param pos
     * @return
     */
    public int method_9505(class_2680 state, class_1922 world, class_2338 pos) {
        return 0;
    }

    /**
     * Append properties.
     *
     * @param builder the builder
     */
    @Override
    protected void method_9515(class_2689.class_2690<class_2248, class_2680> builder) {
        builder.method_11667(CABLE_CONNECTION_NORTH, CABLE_CONNECTION_SOUTH, CABLE_CONNECTION_EAST, CABLE_CONNECTION_WEST, CABLE_CONNECTION_UP, CABLE_CONNECTION_DOWN, WATERLOGGED);
    }

    /**
     * Is instance of cable boolean.
     *
     * @param block    the block
     * @param world    the world
     * @param blockPos the block pos
     * @return the boolean
     */
    // This is to make sure that we only connect to the proper cable instance.
    // If you really want, you can connect to any cable by just checking if it's an instance of GenericCable.
    // Don't expect it to look right though if the other cable doesn't do the same.
    public abstract boolean isInstanceOfCable(class_2248 block, class_1936 world, class_2338 blockPos);

    /**
     * Is instance of interfaceable block boolean.
     *
     * @param block    the block
     * @param world    the world
     * @param blockPos the block pos
     * @return the boolean
     */
    // This is to make sure that we only connect to blocks that are supposed to connect to this cable.
    public abstract boolean isInstanceOfInterfaceableBlock(class_2248 block, class_1936 world, class_2338 blockPos);

    /**
     * Is valid side boolean.
     *
     * @param block    the block
     * @param world    the world
     * @param blockPos the block pos
     * @param side     the side
     * @return the boolean
     */
    // This is to make sure that we only connect to blocks that are supposed to connect to this cable.
    public abstract boolean isValidSide(class_2248 block, class_1936 world, class_2338 blockPos, class_2350 side);

    /**
     * Allows for Opening Gui
     *
     * @param state
     * @param world
     * @param pos
     * @param player
     * @param hand
     * @param hit
     * @return
     */
    public abstract class_1269 openGui(class_2680 state, class_1937 world, class_2338 pos, class_1657 player, class_1268 hand, class_3965 hit);

    /**
     * On placed.
     *
     * @param world     the world
     * @param pos       the pos
     * @param state     the state
     * @param placer    the placer
     * @param itemStack the item stack
     */
    @Override
    public void method_9567(class_1937 world, class_2338 pos, class_2680 state, @Nullable class_1309 placer, class_1799 itemStack) {
        // Responsible for Visually Connecting Cables Together
        setupCableStates(world, pos, state);
    }

    /**
     * On broken.
     *
     * @param world the world
     * @param pos   the pos
     * @param state the state
     */
    @Override
    public void method_9585(class_1936 world, class_2338 pos, class_2680 state) {
        super.method_9585(world, pos, state);

        // Responsible for Visually Disconnecting Cables on Block Breaking
        setupCableStates(world, pos, state, true);
    }

    /**
     * Gets state for neighbor update.
     *
     * @param state     the state
     * @param direction the direction
     * @param newState  the new state
     * @param world     the world
     * @param pos       the pos
     * @param posFrom   the pos from
     * @return the state for neighbor update
     */
    // This runs for every block update that occurs to the cables
    @Override
    public class_2680 method_9559(class_2680 state, class_2350 direction, class_2680 newState, class_1936 world, class_2338 pos, class_2338 posFrom) {
        // Try to support generic fluids if possible
        if (state.method_11654(WATERLOGGED))
            world.method_8405().method_8676(pos, class_3612.field_15910, class_3612.field_15910.method_15789(world));

        // This sets up the cable blockstates for each cable
        return setupCableStates(world, pos, state);
    }

    /**
     * Gets no interfaceable cables text.
     *
     * @return the no interfaceable cables text
     */
    // Override this to change the message text
    public class_2561 getNoInterfaceableCablesText() {
        return new class_2588(Main.MODID + ".no_interfaceable_cables_found");
    }

    /**
     * Gets cable position header text.
     *
     * @return the cable position header text
     */
    // Override this to change the message text
    public class_2561 getCablePositionHeaderText() {
        return new class_2588(Main.MODID + ".cable_position_header");
    }

    /**
     * Gets cable position text.
     *
     * @param cablePos the cable pos
     * @return the cable position text
     */
    // Override this to change the message text
    public class_2561 getCablePositionText(class_2338 cablePos) {
        return new class_2588(Main.MODID + ".cable_position", cablePos.method_10263(), cablePos.method_10264(), cablePos.method_10260());
    }

    /**
     * On use action result.
     *
     * @param state  the state
     * @param world  the world
     * @param pos    the pos
     * @param player the player
     * @param hand   the hand
     * @param hit    the hit
     * @return the action result
     */
    @Override
    public class_1269 method_9534(class_2680 state, class_1937 world, class_2338 pos, class_1657 player, class_1268 hand, class_3965 hit) {
        // So player has to click with an empty hand. This makes it easy to place blocks against the cable.
        if (!player.method_5998(hand).method_7909().equals(class_1802.field_8162)) // This causes weird behavior. Try again later. --> player.getStackInHand(hand).getItem() instanceof BlockItem
            return class_1269.field_5811;

        // Only the server needs to bother with the connection of the network.
        // Success takes away the block placement sound, but also makes it where onUse is only called once.
        //  That's why the check for isClient is after checking if the hand is empty
        if (world.field_9236)
            return class_1269.field_5812;

        // We want to allow the player to access the gui if they are not sneaking
        if (!player.method_5715())
            return openGui(state, world, pos, player, hand, hit);

        List<class_2338> allCables = getAllInterfacingCables(world, pos);

        if (allCables.size() == 0) {
            player.method_7353(getNoInterfaceableCablesText(), false);
            return class_1269.field_21466;
        }

        player.method_7353(getCablePositionHeaderText(), false);
        for (class_2338 cablePos : allCables) {
            player.method_7353(getCablePositionText(cablePos), false);
        }

        return class_1269.field_21466;
    }

    /**
     * Gets all interfacing cables.
     *
     * @param world the world
     * @param pos   the pos
     * @return the all interfacing cables
     */
    public List<class_2338> getAllInterfacingCables(@NotNull class_1936 world, @NotNull class_2338 pos) {
        List<class_2338> knownCables = getAllCables(world, pos);

        // Fixes problem with single cable setups
        if (knownCables.size() == 0) {
            knownCables.add(pos);
        }

        return getAllInterfacingCables(world, knownCables);
    }

    /**
     * Gets all interfacing cables.
     *
     * @param world       the world
     * @param knownCables the known cables
     * @return the all interfacing cables
     */
    public List<class_2338> getAllInterfacingCables(@NotNull class_1936 world, @NotNull List<class_2338> knownCables) {
        List<class_2338> interfacingCables = new ArrayList<>();

        for (class_2338 cablePos : knownCables) {
            if (isInterfacing(world.method_8320(cablePos)))
                interfacingCables.add(cablePos);
        }

        return interfacingCables;
    }

    /**
     * Gets all cables.
     *
     * @param world the world
     * @param pos   the pos
     * @return the all cables
     */
    public List<class_2338> getAllCables(@NotNull class_1936 world, @NotNull class_2338 pos) {
        Set<class_2338> passedCables = new HashSet<>();
        List<class_2338> knownCables = new ArrayList<>();

        visitNeighbors(world, pos, passedCables, knownCables::add, this.maxCount);
        return knownCables;
    }

    /**
     * Visit neighbors.
     *
     * @param world        the world
     * @param pos          the pos
     * @param passedCables the passed cables
     * @param visitor      the visitor
     * @param maxCount     the max count
     */
    private void visitNeighbors(@NotNull class_1936 world, @NotNull class_2338 pos, @NotNull Set<class_2338> passedCables, @NotNull Consumer<class_2338> visitor, int maxCount) {
        visitNeighbors(world, pos, passedCables, visitor, 0, maxCount);
    }

    /**
     * Visit neighbors.
     *
     * @param world        the world
     * @param pos          the pos
     * @param passedCables the passed cables
     * @param visitor      the visitor
     * @param counter      the counter
     * @param maxCount     the max count
     */
    private void visitNeighbors(@NotNull class_1936 world, @NotNull class_2338 pos, @NotNull Set<class_2338> passedCables, @NotNull Consumer<class_2338> visitor, int counter, int maxCount) {
        if (counter >= maxCount)
            return;

        counter++;

        // This check is so I can avoid duplicate results provided from outside this function.
        if (!(pos instanceof class_2338.class_2339))
            visitor.accept(pos);

        List<class_2338> validNeighbors = getValidNeighbors(world, pos);

        for(class_2338 attachedNeighbor : validNeighbors) {
            if (!passedCables.contains(attachedNeighbor)) {
                passedCables.add(attachedNeighbor);

                // TODO (Important): Figure out how to tail this recursion
                visitNeighbors(world, attachedNeighbor, passedCables, visitor, counter, maxCount);
            }
        }
    }

    /**
     * Is interfacing boolean.
     *
     * @param blockState the block state
     * @return the boolean
     */
    public boolean isInterfacing(@NotNull class_2680 blockState) {
        return blockState.method_11654(CABLE_CONNECTION_NORTH).equals(CableConnection.INTERFACEABLE) ||
                blockState.method_11654(CABLE_CONNECTION_SOUTH).equals(CableConnection.INTERFACEABLE) ||
                blockState.method_11654(CABLE_CONNECTION_EAST).equals(CableConnection.INTERFACEABLE) ||
                blockState.method_11654(CABLE_CONNECTION_WEST).equals(CableConnection.INTERFACEABLE) ||
                blockState.method_11654(CABLE_CONNECTION_UP).equals(CableConnection.INTERFACEABLE) ||
                blockState.method_11654(CABLE_CONNECTION_DOWN).equals(CableConnection.INTERFACEABLE);
    }

    public boolean isConnected(@NotNull class_2680 blockState, @NotNull class_2350 direction) {
        return isConnected(blockState, field_11329.get(direction));
    }

    private boolean isConnected(@NotNull class_2680 blockState, @NotNull class_2769<CableConnection> cableConnectionProperty) {
        return blockState.method_11654(cableConnectionProperty).equals(CableConnection.INTERFACEABLE) || blockState.method_11654(cableConnectionProperty).equals(CableConnection.CABLE);
    }

    /**
     * Gets valid neighbors.
     *
     * @param world the world
     * @param pos   the pos
     * @return the valid neighbors
     */
    private List<class_2338> getValidNeighbors(@NotNull class_1936 world, @NotNull class_2338 pos) {
        class_2338 north, south, east, west, up, down;
        class_2680 northBlockState, southBlockState, eastBlockState, westBlockState, upBlockState, downBlockState;
        boolean isNorthValid, isSouthValid, isEastValid, isWestValid, isUpValid, isDownValid;

        // Neighbor Positions
        north = CalculationHelper.addVectors(pos, northVector);
        south = CalculationHelper.addVectors(pos, southVector);
        east = CalculationHelper.addVectors(pos, eastVector);
        west = CalculationHelper.addVectors(pos, westVector);
        up = CalculationHelper.addVectors(pos, upVector);
        down = CalculationHelper.addVectors(pos, downVector);

        // Neighbor BlockStates
        northBlockState = world.method_8320(north);
        southBlockState = world.method_8320(south);
        eastBlockState = world.method_8320(east);
        westBlockState = world.method_8320(west);
        upBlockState = world.method_8320(up);
        downBlockState = world.method_8320(down);

        isNorthValid = isNeighborValidForContinuance(world, northBlockState, north);
        isSouthValid = isNeighborValidForContinuance(world, southBlockState, north);
        isEastValid = isNeighborValidForContinuance(world, eastBlockState, north);
        isWestValid = isNeighborValidForContinuance(world, westBlockState, north);
        isUpValid = isNeighborValidForContinuance(world, upBlockState, north);
        isDownValid = isNeighborValidForContinuance(world, downBlockState, north);

        List<class_2338> cables = new ArrayList<>();
        if (isNorthValid)
            cables.add(north);

        if (isSouthValid)
            cables.add(south);

        if (isEastValid)
            cables.add(east);

        if (isWestValid)
            cables.add(west);

        if (isUpValid)
            cables.add(up);

        if (isDownValid)
            cables.add(down);

        return cables;
    }

    /**
     * Is neighbor valid for continuance boolean.
     *
     * @param world              the world
     * @param neighborBlockState the neighbor block state
     * @param neighborPos        the neighbor pos
     * @return the boolean
     */
    private boolean isNeighborValidForContinuance(@NotNull class_1936 world, @NotNull class_2680 neighborBlockState, @NotNull class_2338 neighborPos) {
        if (!isInstanceOfCable(neighborBlockState.method_26204(), world, neighborPos))
            return false;

        CableConnection northConnection = neighborBlockState.method_11654(CABLE_CONNECTION_NORTH);
        CableConnection southConnection = neighborBlockState.method_11654(CABLE_CONNECTION_SOUTH);
        CableConnection eastConnection = neighborBlockState.method_11654(CABLE_CONNECTION_EAST);
        CableConnection westConnection = neighborBlockState.method_11654(CABLE_CONNECTION_WEST);
        CableConnection upConnection = neighborBlockState.method_11654(CABLE_CONNECTION_UP);
        CableConnection downConnection = neighborBlockState.method_11654(CABLE_CONNECTION_DOWN);

        int isViable = 0;
        isViable = getIsViable(northConnection, isViable);
        isViable = getIsViable(southConnection, isViable);
        isViable = getIsViable(eastConnection, isViable);
        isViable = getIsViable(westConnection, isViable);
        isViable = getIsViable(upConnection, isViable);
        isViable = getIsViable(downConnection, isViable);

        // If only one, then we are seeing our own cable. We need more than one.
        return isViable >= 2;
    }

    /**
     * Gets is viable.
     *
     * @param neighborConnection the neighbor connection
     * @param isViable           the is viable
     * @return the is viable
     */
    // This is so the IDE will shutup about duplicate code.
    private int getIsViable(@NotNull CableConnection neighborConnection, int isViable) {
        if (neighborConnection.equals(CableConnection.CABLE) || neighborConnection.equals(CableConnection.INTERFACEABLE)) {
            isViable++;
        }

        return isViable;
    }

    /**
     * Sets cable state.
     *
     * @param ourBlockState      the our block state
     * @param neighborBlockState the neighbor block state
     * @param ourProperty        the our property
     * @param neighborProperty   the neighbor property
     * @param world              the world
     * @param ourPos             the our pos
     * @param neighborPos        the neighbor pos
     * @param broken             the broken
     * @return the cable state
     */
    protected class_2680 setCableState(class_2680 ourBlockState, class_2680 neighborBlockState, class_2754<CableConnection> ourProperty, class_2754<CableConnection> neighborProperty, class_1936 world, class_2338 ourPos, class_2338 neighborPos, boolean broken) {
        if (isInstanceOfCable(ourBlockState.method_26204(), world, ourPos) && !broken) {
            if (isInstanceOfCable(neighborBlockState.method_26204(), world, neighborPos)) {
                // Connected to Cable
                world.method_8652(neighborPos, neighborBlockState.method_11657(neighborProperty, CableConnection.CABLE), 0x1); // Flag 0x1 = 0b0000001 which means Propagate Changes. More info in net.minecraft.world.ModifiableWorld
                world.method_8652(ourPos, ourBlockState.method_11657(ourProperty, CableConnection.CABLE), 0x1);
            } else if (isInstanceOfInterfaceableBlock(neighborBlockState.method_26204(), world, neighborPos) && isValidSide(neighborBlockState.method_26204(), world, neighborPos, CalculationHelper.getDirection(ourPos, neighborPos))) {
                // Connected to Interfaceable Block
                world.method_8652(ourPos, ourBlockState.method_11657(ourProperty, CableConnection.INTERFACEABLE), 0x1);
            } else {
                // Removed Interfaceable Block
                world.method_8652(ourPos, ourBlockState.method_11657(ourProperty, CableConnection.NONE), 0x1);
            }
        } else if (isInstanceOfCable(neighborBlockState.method_26204(), world, neighborPos)) {
            // Broken Our Cable
            world.method_8652(neighborPos, neighborBlockState.method_11657(neighborProperty, CableConnection.NONE), 0x1);
        }

        // Return Our Latest Changes So Changes Can Stack
        return world.method_8320(ourPos);
    }

    /**
     * Sets cable states.
     *
     * @param world the world
     * @param pos   the pos
     * @param state the state
     * @return the cable states
     */
    protected class_2680 setupCableStates(class_1936 world, class_2338 pos, class_2680 state) {
        return setupCableStates(world, pos, state, false);
    }

    /**
     * Sets cable states.
     *
     * @param world  the world
     * @param pos    the pos
     * @param state  the state
     * @param broken the broken
     * @return the cable states
     */
    protected class_2680 setupCableStates(class_1936 world, class_2338 pos, class_2680 state, boolean broken) {
        // Neighbor Positions
        class_2338 north = CalculationHelper.addVectors(pos, northVector);
        class_2338 south = CalculationHelper.addVectors(pos, southVector);
        class_2338 east = CalculationHelper.addVectors(pos, eastVector);
        class_2338 west = CalculationHelper.addVectors(pos, westVector);
        class_2338 up = CalculationHelper.addVectors(pos, upVector);
        class_2338 down = CalculationHelper.addVectors(pos, downVector);

        // Neighbor BlockStates
        class_2680 northBlockState = world.method_8320(north);
        class_2680 southBlockState = world.method_8320(south);
        class_2680 eastBlockState = world.method_8320(east);
        class_2680 westBlockState = world.method_8320(west);
        class_2680 upBlockState = world.method_8320(up);
        class_2680 downBlockState = world.method_8320(down);

        // Set Cable States
        state = setCableState(state, northBlockState, CABLE_CONNECTION_NORTH, CABLE_CONNECTION_SOUTH, world, pos, north, broken);
        state = setCableState(state, southBlockState, CABLE_CONNECTION_SOUTH, CABLE_CONNECTION_NORTH, world, pos, south, broken);
        state = setCableState(state, eastBlockState, CABLE_CONNECTION_EAST, CABLE_CONNECTION_WEST, world, pos, east, broken);
        state = setCableState(state, westBlockState, CABLE_CONNECTION_WEST, CABLE_CONNECTION_EAST, world, pos, west, broken);
        state = setCableState(state, upBlockState, CABLE_CONNECTION_UP, CABLE_CONNECTION_DOWN, world, pos, up, broken);
        state = setCableState(state, downBlockState, CABLE_CONNECTION_DOWN, CABLE_CONNECTION_UP, world, pos, down, broken);

        return state;
    }

    /**
     * Gets fluid state.
     *
     * @param state the state
     * @return the fluid state
     */
    // Used to visually indicate if waterlogged
    @Override
    public class_3610 method_9545(class_2680 state) {
        return state.method_11654(WATERLOGGED) ? class_3612.field_15910.method_15729(false) : super.method_9545(state);
    }

    /**
     * Dijkstra algorithm vertex path.
     *
     * @param currentKnownCables    the current known cables
     * @param startingBlockPosition the starting block position
     * @param endingBlockPosition   the ending block position
     * @return the vertex path
     */
    @NotNull
    public static VertexPath dijkstraAlgorithm(List<class_2338> currentKnownCables, class_2338 startingBlockPosition, class_2338 endingBlockPosition) {
        List<Vertex> nodes = new ArrayList<>();
        List<Edge> edges = new ArrayList<>();

        Vertex startingPosition = null;
        Vertex endingPosition = null;

        for(class_2338 currentCable : currentKnownCables) {
            Vertex location = new Vertex(currentCable, getCableName(currentCable));
            nodes.add(location);

            // Add Starting Position to Variable To Track Later
            if (currentCable.equals(startingBlockPosition))
                startingPosition = location;

            // Add Ending Position to Variable To Track Later
            if (currentCable.equals(endingBlockPosition))
                endingPosition = location;
        }
        
        if (startingPosition == null || endingPosition == null)
            return new VertexPath();

        // TODO (Important): Add Edges
        addLane(nodes, edges, "Lane_0", nodes.indexOf(startingPosition), nodes.indexOf(endingPosition), 1);
        addLane(nodes, edges, "Lane_1", nodes.indexOf(endingPosition), nodes.indexOf(startingPosition), 1);

        Graph graph = new Graph(nodes, edges);
        DijkstraAlgorithm dijkstraAlgorithm = new DijkstraAlgorithm(graph);

        dijkstraAlgorithm.execute(startingPosition);

        return dijkstraAlgorithm.getPath(endingPosition);
    }

    /**
     * Add lane.
     *
     * @param nodes       the nodes
     * @param edges       the edges
     * @param laneId      the lane id
     * @param source      the source
     * @param destination the destination
     * @param weight      the weight
     */
    private static void addLane(List<Vertex> nodes, List<Edge> edges, String laneId, int source, int destination, int weight) {
        Edge lane = new Edge(laneId, nodes.get(source), nodes.get(destination), weight);
        edges.add(lane);
    }

    /**
     * Gets cable name.
     *
     * @param blockPos the block pos
     * @return the cable name
     */
    private static String getCableName(class_2338 blockPos) {
        return "(" + blockPos.method_10263() + ", " + blockPos.method_10264() + ", " + blockPos.method_10260() + ")";
    }
}
