/*
 * Copyright (c) 2018, apexes.net. All rights reserved.
 *
 *         http://www.apexes.net
 *
 */
package net.apexes.codegen.core;

import com.j256.ormlite.field.DataType;
import com.mysema.codegen.CodeWriter;
import com.mysema.codegen.model.ClassType;
import com.mysema.codegen.model.Parameter;
import com.mysema.codegen.model.SimpleType;
import com.mysema.codegen.model.Type;
import com.mysema.codegen.model.TypeCategory;
import com.mysema.codegen.model.Types;
import com.querydsl.codegen.EntityType;
import com.querydsl.codegen.Property;
import com.querydsl.codegen.Serializer;
import com.querydsl.codegen.SerializerConfig;
import com.querydsl.codegen.TypeMappings;
import com.querydsl.sql.ColumnMetadata;
import com.querydsl.sql.codegen.NamingStrategy;
import com.querydsl.sql.codegen.SQLCodegenModule;
import net.apexes.commons.ormlite.Column;
import net.apexes.commons.ormlite.Table;

import javax.inject.Inject;
import javax.inject.Named;
import java.io.IOException;
import java.lang.annotation.Annotation;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashSet;
import java.util.Set;
import java.util.TreeSet;

import static com.mysema.codegen.Symbols.NEW;

/**
 * 
 * @author <a href="mailto:hedyn@foxmail.com">HeDYn</a>
 *
 */
public class ColumnMetadataSerializer implements Serializer {
    
    protected final TypeMappings typeMappings;

    protected final Collection<String> keywords;

    private final NamingStrategy namingStrategy;

    private final Set<String> imports;

    @Inject
    public ColumnMetadataSerializer(TypeMappings typeMappings, NamingStrategy namingStrategy,
            @Named(SQLCodegenModule.INNER_CLASSES_FOR_KEYS) boolean innerClassesForKeys,
            @Named(SQLCodegenModule.IMPORTS) Set<String> imports,
            @Named(SQLCodegenModule.COLUMN_COMPARATOR) Comparator<Property> columnComparator,
            @Named(SQLCodegenModule.ENTITYPATH_TYPE) Class<?> entityPathType) {
        this.typeMappings = typeMappings;
        this.keywords = Collections.<String> emptyList();
        this.namingStrategy = namingStrategy;
        this.imports = new HashSet<String>(imports);
    }

    @Override
    public void serialize(EntityType model, SerializerConfig config, CodeWriter writer) throws IOException {
        Set<Property> properties = CodegenUtils.getProperties(model);
        
        Type queryType = typeMappings.getPathType(model, model, false);
        if (!queryType.getPackageName().isEmpty()) {
            writer.packageDecl(queryType.getPackageName());
        }
        
        Set<String> importClasses = new TreeSet<>();
        if (!model.getPackageName().isEmpty() && !queryType.getPackageName().equals(model.getPackageName())
                && !queryType.getSimpleName().equals(model.getSimpleName())) {
            String fullName = model.getFullName();
            String packageName = model.getPackageName();
            if (fullName.substring(packageName.length() + 1).contains(".")) {
                fullName = fullName.substring(0, fullName.lastIndexOf('.'));
            }
            importClasses.add(fullName);
        }
        importClasses.add(Column.class.getName());
        for (Property property : properties) {
            switch(property.getType().getCategory()) {
                case DATE:
                case DATETIME:
                case TIME:
                    importClasses.add(DataType.class.getName());
                    break;
                default:
                    break;
            }
        }
        writer.importClasses(importClasses.toArray(new String[0]));
        writeUserImports(writer);

//        writer.nl();

        introJavadoc(writer, model);
        introClassHeader(writer, model);
        if (config.createDefaultVariable()) {
            introDefaultInstance(writer, model, config.defaultVariableName());
        }
        
        serializeProperties(model, properties, writer);
        
        // constructors
        constructors(model, config, writer);

        outro(model, writer);
    }

    protected void writeUserImports(CodeWriter writer) throws IOException {
        Set<String> packages = new HashSet<>();
        Set<String> classes = new HashSet<>();

        for (String javaImport : imports){
            //true if the character next to the dot is an upper case or if no dot is found (-1+1=0) the first character
            boolean isClass = Character.isUpperCase(javaImport.charAt(javaImport.lastIndexOf(".") + 1));
            if (isClass){
                classes.add(javaImport);
            }else{
                packages.add(javaImport);
            }
        }

        if (!packages.isEmpty()) {
            writer.importPackages(packages.toArray(new String[0]));
        }
        if (!classes.isEmpty()) {
            writer.importClasses(classes.toArray(new String[0]));
        }
    }
    
    protected void introJavadoc(CodeWriter writer, EntityType model) throws IOException {
        Type queryType = typeMappings.getPathType(model, model, true);
        writer.javadoc(queryType.getSimpleName() + " is a ORMLite query type for " + model.getSimpleName(),
                "@see " + model.getFullName());
    }
    
    protected void introClassHeader(CodeWriter writer, EntityType model) throws IOException {
        for (Annotation annotation : model.getAnnotations()) {
            writer.annotation(annotation);
        }
        
        Type queryType = typeMappings.getPathType(model, model, true);

        Type superType;
        TypeCategory category = model.getOriginalCategory();
        com.querydsl.codegen.Supertype supertype = model.getSuperType();
        if (supertype != null) {
            superType = supertype.getType();
            superType = typeMappings.getPathType(superType, null, false);
        } else {
            superType = new ClassType(category, Table.class, model);
        }

        writer.beginClass(queryType, superType);
    }
    
    protected void introDefaultInstance(CodeWriter writer, EntityType entityType, String defaultName) 
            throws IOException {
        String variableName = !defaultName.isEmpty() ? defaultName : namingStrategy.getDefaultVariableName(entityType);
        Type queryType = typeMappings.getPathType(entityType, entityType, true);
        String table = (String) entityType.getData().get("table");
        writer.publicStaticFinal(queryType, variableName, NEW + queryType.getSimpleName() + "(\"" + table + "\")");
    }
    
    protected void serializeProperties(EntityType model, Set<Property> properties, CodeWriter writer) 
            throws IOException {
        // fields
        Type colmumType = new SimpleType(Column.class.getSimpleName());
        StringBuilder sb = new StringBuilder();
        for (Property property : properties) {
            String propertyName = property.getEscapedName();
            sb.setLength(0);
            sb.append("field(\"").append(propertyName).append("\")");
            switch(property.getType().getCategory()) {
                case DATE:
                case DATETIME:
                case TIME:
                    sb.append(".dataType(DataType.DATE)");
                    break;
                default:
                    break;
            }
            ColumnMetadata metadata = (ColumnMetadata) property.getData().get("COLUMN");
            sb.append(".columnName(\"").append(metadata.getName()).append("\")");
            if (!metadata.isNullable()) {
                sb.append(".notNull()");
            }
            sb.append(".build()");
            writer.javadoc(metadata.getName(), "@see " + model.getFullName() + "#" + propertyName);
            writer.publicFinal(colmumType, propertyName, sb.toString());
        }
    }
    
    protected void constructors(EntityType model, SerializerConfig config, CodeWriter writer)
            throws IOException {
        String localName = writer.getRawName(model);
        writer.beginConstructor(new Parameter("tableName", Types.STRING));
        writer.line("super(", writer.getClassConstant(localName), ", tableName);");
        constructorContent(writer, model);
        writer.end();
    }
    
    protected void constructorContent(CodeWriter writer, EntityType model) throws IOException {
        // override in subclasses
    }
    
    protected void outro(EntityType model, CodeWriter writer) throws IOException {
        writer.end();
    }

}
