/*
 * Copyright (c) 2018, apexes.net. All rights reserved.
 *
 *         http://www.apexes.net
 *
 */
package net.apexes.codegen.core;

import com.google.common.base.Function;
import com.google.common.collect.Lists;
import com.mysema.codegen.CodeWriter;
import com.mysema.codegen.model.ClassType;
import com.mysema.codegen.model.Parameter;
import com.mysema.codegen.model.Type;
import com.mysema.codegen.model.Types;
import com.querydsl.codegen.*;
import com.querydsl.core.util.BeanUtils;
import com.querydsl.sql.ColumnMetadata;

import java.io.IOException;
import java.io.Serializable;
import java.lang.annotation.Annotation;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * 
 * @see com.querydsl.codegen.BeanSerializer
 * @author <a href=mailto:hedyn@foxmail.com>HeDYn</a>
 *
 */
public class JavabeanSerializer implements Serializer {
	
	private static final Function<Property, Parameter> propertyToParameter = new Function<Property, Parameter>() {
        @Override
        public Parameter apply(Property input) {
            return new Parameter(input.getName(), input.getType());
        }
    };

    private final boolean propertyAnnotations;

    private final List<Type> interfaces = Lists.newArrayList();

    private final String javadocSuffix;

    private boolean addToString, addFullConstructor, addCloneFrom;

    private boolean printSupertype = false;

    /**
     * Create a new BeanSerializer
     */
    public JavabeanSerializer() {
        this(true, " is a Querydsl bean type");
    }

    /**
     * Create a new BeanSerializer with the given javadoc suffix
     *
     * @param javadocSuffix
     */
    public JavabeanSerializer(String javadocSuffix) {
        this(true, javadocSuffix);
    }

    /**
     * Create a new BeanSerializer
     *
     * @param propertyAnnotations
     */
    public JavabeanSerializer(boolean propertyAnnotations) {
        this(propertyAnnotations, " is a Querydsl bean type");
    }

    /**
     * Create a new BeanSerializer
     *
     * @param propertyAnnotations
     * @param javadocSuffix
     */
    public JavabeanSerializer(boolean propertyAnnotations, String javadocSuffix) {
        this.propertyAnnotations = propertyAnnotations;
        this.javadocSuffix = javadocSuffix;
        setPrintSupertype(true);
    }

    @Override
    public void serialize(EntityType model, SerializerConfig serializerConfig,
            CodeWriter writer) throws IOException {
        String simpleName = model.getSimpleName();

        // package
        if (!model.getPackageName().isEmpty()) {
            writer.packageDecl(model.getPackageName());
        }

        // imports
        Set<String> importedClasses = getAnnotationTypes(model);
        for (Type iface : interfaces) {
            importedClasses.add(iface.getFullName());
        }
        //importedClasses.add(Generated.class.getName());
        if (model.hasLists()) {
            importedClasses.add(List.class.getName());
        }
        if (model.hasCollections()) {
            importedClasses.add(Collection.class.getName());
        }
        if (model.hasSets()) {
            importedClasses.add(Set.class.getName());
        }
        if (model.hasMaps()) {
            importedClasses.add(Map.class.getName());
        }
        if (addToString && model.hasArrays()) {
            importedClasses.add(Arrays.class.getName());
        }
        writer.importClasses(importedClasses.toArray(new String[importedClasses.size()]));

        // javadoc
        writer.javadoc(simpleName + javadocSuffix
                + ". the database table \"" 
                + model.getData().get("table") + "\"");

        // header
        for (Annotation annotation : model.getAnnotations()) {
            writer.annotation(annotation);
        }

        //writer.line("@Generated(\"", getClass().getName(), "\")");

        if (!interfaces.isEmpty()) {
            Type superType = null;
            if (printSupertype && model.getSuperType() != null) {
                superType = model.getSuperType().getType();
            }
            Type[] ifaces = interfaces.toArray(new Type[interfaces.size()]);
            writer.beginClass(model, superType, ifaces);
        } else if (printSupertype && model.getSuperType() != null) {
            writer.beginClass(model, model.getSuperType().getType());
        } else {
            writer.beginClass(model);
        }


        bodyStart(model, writer);

        if (addFullConstructor) {
            addFullConstructor(model, writer);
        }
        
        Set<Property> properties = CodegenUtils.getProperties(model);
        serializeProperties(model, properties, writer);
                
        if (addCloneFrom) {
            addCloneFrom(model, writer);
        }
        if (addToString) {
            addToString(properties, writer);
        }

        bodyEnd(model, writer);

        writer.end();
    }
    
    protected void serializeProperties(EntityType model, Set<Property> properties, CodeWriter writer) throws IOException {
        // fields
        for (Property property : properties) {
            if (propertyAnnotations) {
                for (Annotation annotation : property.getAnnotations()) {
                    writer.annotation(annotation);
                }
            }
            ColumnMetadata metadata = (ColumnMetadata) property.getData().get("COLUMN");
            writer.javadoc(metadata.getName());
            writer.privateField(property.getType(), property.getEscapedName());
        }

        // accessors
        for (Property property : properties) {
            String propertyName = property.getEscapedName();
            // getter
            writer.beginPublicMethod(property.getType(), "get"+BeanUtils.capitalize(propertyName));
            writer.line("return ", propertyName, ";");
            writer.end();
            // setter
            Parameter parameter = new Parameter(propertyName, property.getType());
            writer.beginPublicMethod(Types.VOID, "set"+BeanUtils.capitalize(propertyName), parameter);
            writer.line("this.", propertyName, " = ", propertyName, ";");
            writer.end();
        }
    }

    protected void addFullConstructor(EntityType model, CodeWriter writer) throws IOException {
        // public empty constructor
        writer.beginConstructor();
        writer.end();

        // full constructor
        writer.beginConstructor(model.getProperties(), propertyToParameter);
        for (Property property : model.getProperties()) {
            writer.line("this.", property.getEscapedName(), " = ", property.getEscapedName(), ";");
        }
        writer.end();
    }

    protected void addToString(Set<Property> properties, CodeWriter writer) throws IOException {
        if (!properties.isEmpty()) {
            writer.line("@Override");
            writer.beginPublicMethod(Types.STRING, "toString");
            writer.line("StringBuilder sb = new StringBuilder();");
            writer.line("sb.append(super.toString());");
            for (Property property : properties) {
                String propertyName = property.getEscapedName();
                writer.line("sb.append(\",", propertyName, "=\").append(", propertyName, ");");
            }
            writer.line("return sb.toString();");
            writer.end();
        }
    }
    
    protected void addCloneFrom(EntityType model, CodeWriter writer) throws IOException {
        Parameter parameter = new Parameter("from", model);
        writer.beginPublicMethod(Types.VOID, "cloneFrom", parameter);

        Supertype supertype = model.getSuperType();
        if (supertype != null) {
            Class<?> superClass = supertype.getType().getJavaClass();
            try {
                boolean superHasCloneFrom = false;
                for (Method method : superClass.getDeclaredMethods()){
                    if ("cloneFrom".equals(method.getName())) {
                        int m = method.getModifiers();
                        if (Modifier.isPublic(m) || Modifier.isProtected(m)) {
                            superHasCloneFrom = true;
                            break;
                        }
                    }
                }
                if (superHasCloneFrom) {
                    writer.line("super.cloneFrom(from);");
                }
            } catch (Exception e) {
            };
        }

        for (Property property : CodegenUtils.getProperties(model)) {
            String propertyName = property.getEscapedName();
            Class<?> classType = property.getType().getJavaClass();
            boolean canClone = Cloneable.class.isAssignableFrom(classType);
            if (canClone) {
                writer.line("this.", propertyName, 
                        " = from.", propertyName, " == null ? null : (",
                        CodegenUtils.getTypeName(classType), ") from.", propertyName, ".clone();");
            } else {
                writer.line("this.", propertyName, " = from.", propertyName, ";");
            }
        }
        writer.end();
    }
    
    protected void bodyStart(EntityType model, CodeWriter writer) throws IOException {
        com.querydsl.codegen.Supertype supertype = model.getSuperType();
        if (supertype != null) {
            Class<?> superClass = supertype.getType().getJavaClass();
            if (Serializable.class.isAssignableFrom(superClass)) {
                writer.privateStaticFinal(Types.LONG_P, "serialVersionUID", "1L");
            }
        }
	}

	protected void bodyEnd(EntityType model, CodeWriter writer) throws IOException {
	}

    private Set<String> getAnnotationTypes(EntityType model) {
        Set<String> imports = new HashSet<String>();
        for (Annotation annotation : model.getAnnotations()) {
            imports.add(annotation.annotationType().getName());
        }
        if (propertyAnnotations) {
            for (Property property : model.getProperties()) {
                for (Annotation annotation : property.getAnnotations()) {
                    imports.add(annotation.annotationType().getName());
                }
            }
        }
        return imports;
    }

    public void addInterface(Class<?> iface) {
        interfaces.add(new ClassType(iface));
    }

    public void addInterface(Type type) {
        interfaces.add(type);
    }

    public void setAddToString(boolean addToString) {
        this.addToString = addToString;
    }

    public void setAddFullConstructor(boolean addFullConstructor) {
        this.addFullConstructor = addFullConstructor;
    }

    public void setAddCloneFrom(boolean addCloneFrom) {
        this.addCloneFrom = addCloneFrom;
    }

    public void setPrintSupertype(boolean printSupertype) {
        this.printSupertype = printSupertype;
    }

}
