/*
 * Copyright (c) 2018, apexes.net. All rights reserved.
 *
 *         http://www.apexes.net
 *
 */
package net.apexes.codegen.core;

import com.google.common.base.Function;
import com.google.common.collect.ImmutableList;
import com.google.common.io.Files;
import com.mysema.codegen.CodeWriter;
import com.mysema.codegen.JavaWriter;
import com.mysema.codegen.ScalaWriter;
import com.mysema.codegen.model.ClassType;
import com.mysema.codegen.model.SimpleType;
import com.mysema.codegen.model.Type;
import com.mysema.codegen.model.TypeCategory;
import com.querydsl.codegen.*;
import com.querydsl.sql.*;
import com.querydsl.sql.codegen.MetaDataExporter;
import com.querydsl.sql.codegen.NamingStrategy;
import com.querydsl.sql.codegen.SQLCodegenModule;
import com.querydsl.sql.codegen.SpatialSupport;
import com.querydsl.sql.codegen.support.*;
import net.apexes.commons.querydsl.info.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nullable;
import java.io.File;
import java.io.IOException;
import java.io.StringWriter;
import java.nio.charset.Charset;
import java.sql.DatabaseMetaData;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.*;

/**
 * 
 * @see com.querydsl.sql.codegen.MetaDataExporter
 * @author <a href="mailto:hedyn@foxmail.com">HeDYn</a>
 *
 */
public class MetadataExporter {
    
    private static final Logger logger = LoggerFactory.getLogger(MetaDataExporter.class);

    private final SQLTemplatesRegistry sqlTemplatesRegistry = new SQLTemplatesRegistry();

    private final SQLCodegenModule module = new SQLCodegenModule();

    private final Set<String> classes = new HashSet<String>();

    private File targetFolder;

    private File beansTargetFolder;

    @Nullable
    private String beanPackageName;

    @Nullable
    private String schemaPattern, tableNamePattern;

    @Nullable
    private Serializer beanSerializer;

    private boolean createScalaSources = false;

    private final Map<EntityType, Type> entityToWrapped = new HashMap<EntityType, Type>();

    private Serializer serializer;

    private TypeMappings typeMappings;

    private QueryTypeFactory queryTypeFactory;

    private NamingStrategy namingStrategy;

    private Configuration configuration;
    
    // EXT #######################
//    private KeyDataFactory keyDataFactory;
    private KeyDataConvertor keyDataConvertor;
    // ####################### EXT

    private boolean columnAnnotations = false;

    private boolean validationAnnotations = false;

    private boolean schemaToPackage = false;

    private String sourceEncoding = "UTF-8";

    private boolean lowerCase = false;

    private boolean exportTables = true;

    private boolean exportViews = true;

    private boolean exportAll = false;
    
    // EXT #######################
//    private boolean exportPrimaryKeys = true;
//    private boolean exportForeignKeys = true;    
    private boolean exportKeys = true;
    private boolean ignoreMetadata = false; 
    // ####################### EXT

    private boolean spatial = false;

    @Nullable
    private String tableTypesToExport;

    public MetadataExporter() { }

    @SuppressWarnings("unchecked")
    protected EntityType createEntityType(SchemaAndTable schemaAndTable,
            final String className) {
        EntityType classModel;

        if (beanSerializer == null) {
            String packageName = normalizePackage(module.getPackageName(), schemaAndTable);
            String simpleName = module.getPrefix() + className + module.getSuffix();
            Type classTypeModel = new SimpleType(TypeCategory.ENTITY,
                    packageName + "." + simpleName,  packageName, simpleName, false, false);
            classModel = new EntityType(classTypeModel, module.get(Function.class, CodegenModule.VARIABLE_NAME_FUNCTION_CLASS));
            typeMappings.register(classModel, classModel);

        } else {
            String beanPackage = normalizePackage(beanPackageName, schemaAndTable);
            String simpleName = module.getBeanPrefix() + className + module.getBeanSuffix();
            Type classTypeModel = new SimpleType(TypeCategory.ENTITY,
                    beanPackage + "." + simpleName, beanPackage, simpleName, false, false);
            classModel = new EntityType(classTypeModel, module.get(Function.class, CodegenModule.VARIABLE_NAME_FUNCTION_CLASS));

            Type mappedType = queryTypeFactory.create(classModel);
            entityToWrapped.put(classModel, mappedType);
            typeMappings.register(classModel, mappedType);
        }

        classModel.getData().put("schema", schemaAndTable.getSchema());
        classModel.getData().put("table", schemaAndTable.getTable());
        
        // EXT #######################
        SuperType supertype = getSuperType(classModel);
        if (supertype != null) {
            classModel.addSupertype(supertype);
            typeMappings.register(supertype.getType(), supertype.getQueryType());
        }
        // ####################### EXT
        
        return classModel;
    }


    private String normalizePackage(String packageName, SchemaAndTable schemaAndTable) {
        String rval = packageName;
        if (schemaToPackage) {
            rval = namingStrategy.getPackage(rval, schemaAndTable);
        }
        return rval;
    }

    protected Property createProperty(EntityType classModel, String normalizedColumnName,
            String propertyName, Type typeModel) {
        return new Property(
                classModel,
                propertyName,
                propertyName,
                typeModel,
                Collections.<String>emptyList(),
                false);
    }

    /**
     * Export the tables based on the given database metadata
     *
     * @param md database metadata
     * @throws SQLException
     */
    public void export(DatabaseMetaData md) throws Exception {
        export(md, null);
    }

    public void export(DatabaseMetaData md, TableFilter tableFilter) throws Exception {
        if (beanPackageName == null) {
            beanPackageName =  module.getPackageName();
        }
        if (beansTargetFolder == null) {
            beansTargetFolder = targetFolder;
        }
        module.bind(SQLCodegenModule.BEAN_PACKAGE_NAME, beanPackageName);

        if (spatial) {
            SpatialSupport.addSupport(module);
        }

        classes.clear();
        typeMappings = module.get(TypeMappings.class);
        queryTypeFactory = module.get(QueryTypeFactory.class);
        serializer = module.get(Serializer.class);
        beanSerializer = module.get(Serializer.class, SQLCodegenModule.BEAN_SERIALIZER);
        namingStrategy = module.get(NamingStrategy.class);
        configuration = module.get(Configuration.class);

        SQLTemplates templates = sqlTemplatesRegistry.getTemplates(md);
        if (templates != null) {
            configuration.setTemplates(templates);
        } else {
            logger.info("Found no specific dialect for " + md.getDatabaseProductName());
        }

        // EXT #######################
//        if (beanSerializer == null) {
//            keyDataFactory = new KeyDataFactory(namingStrategy,  module.getPackageName(),
//                    module.getPrefix(), module.getSuffix(), schemaToPackage);
//        } else {
//            keyDataFactory = new KeyDataFactory(namingStrategy, beanPackageName,
//                    module.getBeanPrefix(), module.getBeanSuffix(), schemaToPackage);
//        }

        if (beanSerializer == null) {
            keyDataConvertor = new KeyDataConvertor(namingStrategy,  module.getPackageName(),
                    module.getPrefix(), module.getSuffix(), schemaToPackage);
        } else {
            keyDataConvertor = new KeyDataConvertor(namingStrategy,  beanPackageName,
                    module.getBeanPrefix(), module.getBeanSuffix(), schemaToPackage);
        }
        // ####################### EXT

        String[] typesArray = null;

        if (tableTypesToExport != null && !tableTypesToExport.isEmpty()) {
            List<String> types = new ArrayList<String>();
            for (String tableType : tableTypesToExport.split(",")) {
                types.add(tableType.trim());
            }
            typesArray = types.toArray(new String[types.size()]);
        } else if (!exportAll) {
            List<String> types = new ArrayList<String>(2);
            if (exportTables) {
                types.add("TABLE");
            }
            if (exportViews) {
                types.add("VIEW");
            }
            typesArray = types.toArray(new String[types.size()]);
        }

        List<String> schemaPatterns = Arrays.asList(schemaPattern);
        if (schemaPattern != null && schemaPattern.contains(",")) {
            schemaPatterns = ImmutableList.copyOf(schemaPattern.split(","));
        }
        List<String> tablePatterns = Arrays.asList(tableNamePattern);
        if (tableNamePattern != null && tableNamePattern.contains(",")) {
            tablePatterns = ImmutableList.copyOf(tableNamePattern.split(","));
        }

        for (String schemaPattern : schemaPatterns) {
            schemaPattern = schemaPattern != null ? schemaPattern.trim() : null;
            for (String tablePattern : tablePatterns) {
                tablePattern = tablePattern != null ? tablePattern.trim() : null;
                ResultSet tables = md.getTables(null, schemaPattern, tablePattern, typesArray);
                try {
                    while (tables.next()) {
                        String catalog = tables.getString("TABLE_CAT");
                        String schema = tables.getString("TABLE_SCHEM");
                        String tableName = normalize(tables.getString("TABLE_NAME"));
                        if (tableFilter != null) {
                            if (tableFilter.accept(catalog, schema, tableName)) {
                                handleTable(md, catalog, schema, tableName);
                            }
                        } else {
                            handleTable(md, catalog, schema, tableName);
                        }
                    }
                } finally {
                    tables.close();
                }
            }
        }
    }

    private void handleTable(DatabaseMetaData md, String catalog, String schema, String tableName) throws SQLException {
        logger.info("{}.{} ...", schema, tableName);

        String schemaName = normalize(schema);
        String normalizedSchemaName = namingStrategy.normalizeSchemaName(schemaName);
        String normalizedTableName = namingStrategy.normalizeTableName(tableName);

        SchemaAndTable schemaAndTable = new SchemaAndTable(normalizedSchemaName, normalizedTableName);

        if (!namingStrategy.shouldGenerateClass(schemaAndTable)) {
            return;
        }

        String className = namingStrategy.getClassName(schemaAndTable);
        EntityType classModel = createEntityType(schemaAndTable, className);

        // EXT #######################
        TableInfo tableInfo = new TableInfo(md, catalog, schema, tableName, lowerCase);
        if (exportKeys) {
            // primary key
            PrimaryKeyInfo pkInfo = tableInfo.getPrimaryKey();
            if (pkInfo != null) {
                PrimaryKeyData pkData = new PrimaryKeyData(pkInfo.getName(), pkInfo.getColumns().toArray(new String[0]));
                classModel.getData().put(PrimaryKeyData.class, Arrays.asList(pkData));
            }
            // collect foreign keys
            List<ForeignKeyInfo> fkInfos = tableInfo.getImportedKeys();
            if (!fkInfos.isEmpty()) {
                Collection<ForeignKeyData> fkDatas = keyDataConvertor.toForeignKeyData(fkInfos);
                for (ForeignKeyData fkd : fkDatas) {
                    if (namingStrategy.shouldGenerateForeignKey(schemaAndTable, fkd)) {
                        classModel.getData().put(ForeignKeyData.class, fkDatas);
                    }
                }
            }
            // collect inverse foreign keys
            List<ForeignKeyInfo> invFkInfos = tableInfo.getExportedKeys();
            if (!invFkInfos.isEmpty()) {
                classModel.getData().put(InverseForeignKeyData.class, keyDataConvertor.toInverseForeignKeyData(invFkInfos));
            }
            // collect indexs
            List<IndexInfo> ixInfos = tableInfo.getIndexs();
            if (!ixInfos.isEmpty()) {
                classModel.getData().put(net.apexes.codegen.core.IndexData.class, keyDataConvertor.toIndexData(ixInfos));
            }
        }
        // collect columns
        for (ColumnInfo columnInfo : tableInfo.getColumns()) {
            handleColumn(classModel, tableName, columnInfo);
        }
        
//        if (exportPrimaryKeys) {
//            // collect primary keys
//            Map<String,PrimaryKeyData> primaryKeyData = keyDataFactory
//                    .getPrimaryKeys(md, catalog, schema, tableName);
//            if (!primaryKeyData.isEmpty()) {
//                classModel.getData().put(PrimaryKeyData.class, primaryKeyData.values());
//            }
//        }
//
//        if (exportForeignKeys) {
//            // collect foreign keys
//            Map<String,ForeignKeyData> foreignKeyData = keyDataFactory
//                    .getImportedKeys(md, catalog, schema, tableName);
//            if (!foreignKeyData.isEmpty()) {
//                for (ForeignKeyData fkd : foreignKeyData.values()) {
//                    if (namingStrategy.shouldGenerateForeignKey(schemaAndTable, fkd)) {
//                        classModel.getData().put(ForeignKeyData.class, foreignKeyData.values());
//                    }
//                }
//            }
//
//            // collect inverse foreign keys
//            Map<String,InverseForeignKeyData> inverseForeignKeyData = keyDataFactory
//                    .getExportedKeys(md, catalog, schema, tableName);
//            if (!inverseForeignKeyData.isEmpty()) {
//                classModel.getData().put(InverseForeignKeyData.class, inverseForeignKeyData.values());
//            }
//        }
//        
//        // collect columns
//        ResultSet columns = md.getColumns(catalog, schema, tableName.replace("/", "//"), null);
//        try {
//            while (columns.next()) {
//                handleColumn(classModel, tableName, columns);
//            }
//        } finally {
//            columns.close();
//        }
        // ####################### EXT

        // serialize model
        serialize(classModel, schemaAndTable);

        logger.info("{}.{} successfully!", schema, tableName);
    }

    private void handleColumn(EntityType classModel, String tableName, ColumnInfo columnInfo) throws SQLException {
        String columnName = columnInfo.getName();
        String normalizedColumnName = columnInfo.getNormalizedName();
        int columnType = columnInfo.getJdbcType();
        String typeName = columnInfo.getTypeName();
        Number columnSize = columnInfo.getSize();
        Number columnDigits = columnInfo.getDigits();
        int columnIndex = columnInfo.getIndex();
        String defaultValue = columnInfo.getDefaultValue();
        String describe = columnInfo.getDescribe();
        
        ColumnMetadata column = ColumnMetadata.named(normalizedColumnName).ofType(columnType).withIndex(columnIndex);
        if (columnInfo.isNotNullable()) {
            column = column.notNull();
        }
        if (columnSize != null && columnSize.intValue() > 0) {
            column = column.withSize(columnSize.intValue());
        }
        if (columnDigits != null) {
            column = column.withDigits(columnDigits.intValue());
        }
        
        String propertyName = namingStrategy.getPropertyName(normalizedColumnName, classModel);
        Class<?> clazz = configuration.getJavaType(columnType,
                typeName,
                columnSize != null ? columnSize.intValue() : 0,
                columnDigits != null ? columnDigits.intValue() : 0,
                tableName, columnName);
        if (clazz == null) {
            clazz = Object.class;
        }
        TypeCategory fieldType = TypeCategory.get(clazz.getName());
        if (Number.class.isAssignableFrom(clazz)) {
            fieldType = TypeCategory.NUMERIC;
        } else if (Enum.class.isAssignableFrom(clazz)) {
            fieldType = TypeCategory.ENUM;
        }
        Type typeModel = new ClassType(fieldType, clazz);
        Property property = createProperty(classModel, normalizedColumnName, propertyName, typeModel);
        property.getData().put("COLUMN", column);
        if (defaultValue != null) {
            property.getData().put("COLUMN_DEF", defaultValue);
        }
        if (describe != null && !describe.trim().isEmpty()) {
            property.getData().put("REMARKS", describe);
        }
        
        if (columnAnnotations) {
            property.addAnnotation(new ColumnImpl(normalizedColumnName));
        }
        if (validationAnnotations) {
            if (columnInfo.isNotNullable()) {
                property.addAnnotation(new NotNullImpl());
            }
            int size = (columnSize == null ? 0 : columnSize.intValue());
            if (size > 0 && clazz.equals(String.class)) {
                property.addAnnotation(new SizeImpl(0, size));
            }
        }
        classModel.addProperty(property);
    }
    
    private String normalize(String str) {
        if (lowerCase && str != null) {
            return str.toLowerCase();
        } else {
            return str;
        }
    }
    
    private void serialize(EntityType type, SchemaAndTable schemaAndTable) {
        try {
            String fileSuffix = createScalaSources ? ".scala" : ".java";

            if (beanSerializer != null) {
                String packageName = normalizePackage(beanPackageName, schemaAndTable);
                String path = packageName.replace('.', '/') + "/" + type.getSimpleName() + fileSuffix;
                write(beanSerializer, new File(beansTargetFolder, path), type);
                
                if (!ignoreMetadata) {
                	String otherPath = entityToWrapped.get(type).getFullName().replace('.', '/') + fileSuffix;
                    write(serializer, new File(targetFolder, otherPath), type);
                }
            } else {
                String packageName = normalizePackage(module.getPackageName(), schemaAndTable);
                String path =  packageName.replace('.', '/') + "/" + type.getSimpleName() + fileSuffix;
                write(serializer, new File(targetFolder, path), type);
            }

        } catch (IOException e) {
            throw new RuntimeException(e.getMessage(), e);
        }
    }
    
    private void write(Serializer serializer, File targetFile, EntityType type) throws IOException {
        if (!classes.add(targetFile.getPath())) {
            throw new IllegalStateException("Attempted to write multiple times to " +
                    targetFile.getPath() + ", please check your configuration");
        }
        StringWriter w = new StringWriter();
        CodeWriter writer = createScalaSources ? new ScalaWriter(w) : new JavaWriter(w);
        serializer.serialize(type, SimpleSerializerConfig.DEFAULT, writer);

        // conditional creation
        boolean generate = true;
        byte[] bytes = w.toString().getBytes(sourceEncoding);
        if (targetFile.exists() && targetFile.length() == bytes.length) {
            String str = Files.toString(targetFile, Charset.forName(sourceEncoding));
            if (str.equals(w.toString())) {
                generate = false;
            }
        } else {
            targetFile.getParentFile().mkdirs();
        }

        if (generate) {
            Files.write(bytes, targetFile);
        }
    }

    Set<String> getClasses() {
        return classes;
    }

    /**
     * Set the schema pattern filter to be used
     *
     * @param schemaPattern a schema name pattern; must match the schema name
     *        as it is stored in the database; "" retrieves those without a schema;
     *        {@code null} means that the schema name should not be used to narrow
     *        the search (default: null)
     */
    public void setSchemaPattern(@Nullable String schemaPattern) {
        this.schemaPattern = schemaPattern;
    }

    /**
     * Set the table name pattern filter to be used
     *
    * @param tableNamePattern a table name pattern; must match the
    *        table name as it is stored in the database (default: null)
    */
    public void setTableNamePattern(@Nullable String tableNamePattern) {
        this.tableNamePattern = tableNamePattern;
    }

    /**
     * Override the configuration
     *
     * @param configuration override configuration for custom type mappings etc
     */
    public void setConfiguration(Configuration configuration) {
        module.bind(Configuration.class, configuration);
    }

    /**
     * Set true to create Scala sources instead of Java sources
     *
     * @param createScalaSources whether to create Scala sources (default: false)
     */
    public void setCreateScalaSources(boolean createScalaSources) {
        this.createScalaSources = createScalaSources;
    }

    /**
     * Set the target folder
     *
     * @param targetFolder target source folder to create the sources into
     *        (e.g. target/generated-sources/java)
     */
    public void setTargetFolder(File targetFolder) {
        this.targetFolder = targetFolder;
    }

    /**
     * Set the target folder for beans
     *
     * <p>defaults to the targetFolder value</p>
     *
     * @param targetFolder target source folder to create the bean sources into
     */
    public void setBeansTargetFolder(File targetFolder) {
        this.beansTargetFolder = targetFolder;
    }

    /**
     * Set the package name
     *
     * @param packageName package name for sources
     */
    public void setPackageName(String packageName) {
        module.bind(SQLCodegenModule.PACKAGE_NAME, packageName);
    }

    /**
     * Override the bean package name (default: packageName)
     *
     * @param beanPackageName package name for bean sources
     */
    public void setBeanPackageName(@Nullable String beanPackageName) {
        this.beanPackageName = beanPackageName;
    }

    /**
     * Override the name prefix for the classes (default: Q)
     *
     * @param namePrefix name prefix for querydsl-types (default: Q)
     */
    public void setNamePrefix(String namePrefix) {
        module.bind(CodegenModule.PREFIX, namePrefix);
    }

    /**
     * Override the name suffix for the classes (default: "")
     *
     * @param nameSuffix name suffix for querydsl-types (default: "")
     */
    public void setNameSuffix(String nameSuffix) {
        module.bind(CodegenModule.SUFFIX, nameSuffix);
    }

    /**
     * Override the bean prefix for the classes (default: "")
     *
     * @param beanPrefix bean prefix for bean-types (default: "")
     */
    public void setBeanPrefix(String beanPrefix) {
        module.bind(SQLCodegenModule.BEAN_PREFIX, beanPrefix);
    }

    /**
     * Override the bean suffix for the classes (default: "")
     *
     * @param beanSuffix bean suffix for bean-types (default: "")
     */
    public void setBeanSuffix(String beanSuffix) {
        module.bind(SQLCodegenModule.BEAN_SUFFIX, beanSuffix);
    }

    /**
     * Override the NamingStrategy (default: new DefaultNamingStrategy())
     *
     * @param namingStrategy naming strategy to override (default: new DefaultNamingStrategy())
     */
    public void setNamingStrategy(NamingStrategy namingStrategy) {
        module.bind(NamingStrategy.class, namingStrategy);
    }

    /**
     * Set the Bean serializer to create bean types as well
     *
     * @param beanSerializer serializer for JavaBeans (default: null)
     */
    public void setBeanSerializer(@Nullable Serializer beanSerializer) {
    	setBeanSerializer(beanSerializer, false);
    }
    
    public void setBeanSerializer(@Nullable Serializer beanSerializer, boolean ignoreMetadata) {
    	module.bind(SQLCodegenModule.BEAN_SERIALIZER, beanSerializer);
    	this.ignoreMetadata = ignoreMetadata;
    }

    /**
     * Set whether inner classes should be created for keys
     *
     * @param innerClassesForKeys
     */
    public void setInnerClassesForKeys(boolean innerClassesForKeys) {
        module.bind(SQLCodegenModule.INNER_CLASSES_FOR_KEYS, innerClassesForKeys);
    }

    /**
     * Set the column comparator class
     *
     * @param columnComparatorClass
     */
    public void setColumnComparatorClass(Class<? extends Comparator<Property>> columnComparatorClass) {
        module.bind(SQLCodegenModule.COLUMN_COMPARATOR, columnComparatorClass);
    }

    /**
     * Set the serializer class
     *
     * @param serializerClass
     */
    public void setSerializerClass(Class<? extends Serializer> serializerClass) {
        module.bind(Serializer.class, serializerClass);
    }
    
    public void setEntityPathType(Class<?> classType) {
        module.bindInstance(SQLCodegenModule.ENTITYPATH_TYPE, classType);
    }

    /**
     * Set the type mappings to use
     *
     * @param typeMappings
     */
    public void setTypeMappings(TypeMappings typeMappings) {
        module.bind(TypeMappings.class, typeMappings);
    }
    
    public TypeMappings getTypeMappings() {
        return module.get(TypeMappings.class);
    }

    /**
     * Set whether column annotations should be created
     *
     * @param columnAnnotations
     */
    public void setColumnAnnotations(boolean columnAnnotations) {
        this.columnAnnotations = columnAnnotations;
    }

    /**
     * Set whether validation annotations should be created
     *
     * @param validationAnnotations
     */
    public void setValidationAnnotations(boolean validationAnnotations) {
        this.validationAnnotations = validationAnnotations;
    }

    /**
     * Set the source encoding
     *
     * @param sourceEncoding
     */
    public void setSourceEncoding(String sourceEncoding) {
        this.sourceEncoding = sourceEncoding;
    }

    /**
     * Set whether schema names should be appended to the package name.
     *
     * <p><b>!!! Important !!!</b><i> {@link NamingStrategy#getPackage(String, SchemaAndTable)}
     * will be invoked only if <code>schemaToPackage</code> is set to <code>true</code>.</i></p>
     *
     * @deprecated This flag will not be necessary in the future because the generated package name
     * can be controlled in method {@link NamingStrategy#getPackage(String, SchemaAndTable)}.
     *
     * @param schemaToPackage
     */
    @Deprecated
    public void setSchemaToPackage(boolean schemaToPackage) {
        this.schemaToPackage = schemaToPackage;
        module.bind(SQLCodegenModule.SCHEMA_TO_PACKAGE, schemaToPackage);
    }

    /**
     * Set whether names should be normalized to lowercase
     *
     * @param lowerCase
     */
    public void setLowerCase(boolean lowerCase) {
        this.lowerCase = lowerCase;
    }

    /**
     * Set whether tables should be exported
     *
     * @param exportTables
     */
    public void setExportTables(boolean exportTables) {
        this.exportTables = exportTables;
    }

    /**
     * Set whether views should be exported
     *
     * @param exportViews
     */
    public void setExportViews(boolean exportViews) {
        this.exportViews = exportViews;
    }

    /**
     * Set whether all table types should be exported
     *
     * @param exportAll
     */
    public void setExportAll(boolean exportAll) {
        this.exportAll = exportAll;
    }

//    /**
//     * Set whether primary keys should be exported
//     *
//     * @param exportPrimaryKeys
//     */
//    public void setExportPrimaryKeys(boolean exportPrimaryKeys) {
//        this.exportPrimaryKeys = exportPrimaryKeys;
//    }
//
//    /**
//     * Set whether foreign keys should be exported
//     *
//     * @param exportForeignKeys
//     */
//    public void setExportForeignKeys(boolean exportForeignKeys) {
//        this.exportForeignKeys = exportForeignKeys;
//    }
    
    public void setExportKeys(boolean exportKeys) {
        this.exportKeys = exportKeys;
    }

    /**
     * Set the java imports
     *
     * @param imports
     *            java imports array
     */
    public void setImports(String[] imports) {
        module.bind(CodegenModule.IMPORTS, new HashSet<String>(Arrays.asList(imports)));
    }

    /**
     * Set whether spatial type support should be used
     *
     * @param spatial
     */
    public void setSpatial(boolean spatial) {
        this.spatial = spatial;
    }

    /**
     * Set the table types to export as a comma separated string
     *
     * @param tableTypesToExport
     */
    public void setTableTypesToExport(String tableTypesToExport) {
        this.tableTypesToExport = tableTypesToExport;
    }
    
    /****************************************************
     * 
     * EXT
     * 
     ***************************************************/
    
    protected SuperType getSuperType(EntityType entityType) {
        return null;
    }

    /**
     * 
     * @author <a href="mailto:hedyn@foxmail.com">HeDYn</a>
     *
     */
    private static class KeyDataConvertor {
        
        private final NamingStrategy namingStrategy;
        
        private final String packageName, prefix, suffix;

        private final boolean schemaToPackage;

        public KeyDataConvertor(NamingStrategy namingStrategy, 
                String packageName, String prefix, String suffix, boolean schemaToPackage) {
            this.namingStrategy = namingStrategy;
            this.packageName = packageName;
            this.prefix = prefix;
            this.suffix = suffix;
            this.schemaToPackage = schemaToPackage;
        }
        
        public Collection<ForeignKeyData> toForeignKeyData(List<ForeignKeyInfo> list) {
            List<ForeignKeyData> result = new ArrayList<>();
            for (ForeignKeyInfo info : list) {
                String name = info.getName();
                String parentSchemaName = info.getParentSchema();
                String parentTableName = info.getParentTable();
                
                ForeignKeyData keyData = new ForeignKeyData(name, parentSchemaName, parentTableName,
                        createType(parentSchemaName, parentTableName));
                int size = info.getForeignColumns().size();
                for (int i = 0; i < size; i++) {
                    String foreignColumnName = info.getForeignColumns().get(i);
                    String parentColumnName = info.getParentColumns().get(i);
                    keyData.add(foreignColumnName, parentColumnName);
                }
                result.add(keyData);
            }
            return result;
        }
        
        public Collection<InverseForeignKeyData> toInverseForeignKeyData(List<ForeignKeyInfo> list) {
            List<InverseForeignKeyData> result = new ArrayList<>();
            for (ForeignKeyInfo info : list) {
                String name = info.getName();
                String foreignSchemaName = info.getForeignSchema();
                String foreignTableName = info.getForeignTable();
                
                InverseForeignKeyData keyData = new InverseForeignKeyData(name, foreignSchemaName, foreignTableName,
                        createType(foreignSchemaName, foreignTableName));
                int size = info.getForeignColumns().size();
                for (int i = 0; i < size; i++) {
                    String foreignColumnName = info.getForeignColumns().get(i);
                    String parentColumnName = info.getParentColumns().get(i);
                    keyData.add(parentColumnName, foreignColumnName);
                }
                result.add(keyData);
            }
            return result;
        }
        
        public Collection<net.apexes.codegen.core.IndexData> toIndexData(List<IndexInfo> list) {
            List<net.apexes.codegen.core.IndexData> result = new ArrayList<>();
            for (IndexInfo info : list) {
                String indexSchemaName = info.getSchema();
                String indexTableName = info.getTable();
                net.apexes.codegen.core.IndexData indexData = new net.apexes.codegen.core.IndexData(info, createType(indexSchemaName, indexTableName));
                result.add(indexData);
            }
            return result;
        }

        private Type createType(String schemaName, String table) {
            SchemaAndTable schemaAndTable = new SchemaAndTable(schemaName, table);
            String packageName = this.packageName;
            if (schemaToPackage) {
                packageName = namingStrategy.getPackage(packageName, schemaAndTable);
            }
            String simpleName = prefix + namingStrategy.getClassName(schemaAndTable) + suffix;
            return new SimpleType(packageName + "." + simpleName, packageName, simpleName);
        }
        
    }
}
