/*
 * Copyright (C) 2015, apexes.net. All rights reserved.
 * 
 *        http://www.apexes.net
 * 
 */
package net.apexes.wsonrpc.core;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;

/**
 * 
 * @author <a href=mailto:hedyn@foxmail.com>HeDYn</a>
 *
 */
public class WsonrpcEndpoint implements WsonrpcRemote {

    private static final Logger LOG = LoggerFactory.getLogger(WsonrpcEndpoint.class);
    
    protected final WsonrpcEngine wsonrpcEngine;
    private WebSocketSession session;

    protected WsonrpcEndpoint(WsonrpcConfig config) {
        this(new WsonrpcEngine(config));
    }

    protected WsonrpcEndpoint(WsonrpcEngine wsonrpcEngine) {
        this.wsonrpcEngine = wsonrpcEngine;
    }

    protected final void openSession(WebSocketSession session) {
        this.session = session;
    }

    protected void closeSession() throws Exception {
        if (session != null) {
            session.close();
            session = null;
        }
    }

    protected WebSocketSession getSession() {
        return session;
    }
    
    protected void verifyOnline() throws WsonrpcException {
        if (!isConnected()) {
            throw new WsonrpcException("Connection is closed.");
        }
    }
    
    @Override
    public boolean isConnected() {
        return session != null && session.isOpen();
    }

    @Override
    public String getSessionId() {
        if (session != null) {
            return session.getId();
        }
        return null;
    }

    @Override
    public void disconnect() {
        try {
            closeSession();
        } catch (Exception e) {
            LOG.warn("disconnect error!");
        }
    }
    
    @Override
    public void ping() throws Exception {
        verifyOnline();
        session.ping();
    }

    @Override
    public void notify(String serviceName, String methodName, Object[] args)
            throws IOException, WsonrpcException {
        verifyOnline();
        wsonrpcEngine.notify(getSession(), serviceName, methodName, args);
    }

    @Override
    public <T> T request(String serviceName, String methodName, Object[] args, Class<T> returnType, int timeout)
            throws IOException, WsonrpcException, RemoteException {
        Future<T> future = request(serviceName, methodName, args, returnType);
        try {
            if (timeout <= 0) {
                return future.get();
            } else {
                return future.get(timeout, TimeUnit.MILLISECONDS);
            }
        } catch (Throwable e) {
            if (e instanceof ExecutionException) {
                e = e.getCause();
            }
            if (e instanceof RemoteException) {
                throw (RemoteException) e;
            }
            if (e instanceof WsonrpcException) {
                throw (WsonrpcException) e;
            }
            throw new WsonrpcException(e);
        }
    }

    @Override
    public <T> Future<T> request(String serviceName, String methodName, Object[] args, Class<T> returnType)
            throws IOException, WsonrpcException {
        verifyOnline();
        return wsonrpcEngine.request(session, serviceName, methodName, args, returnType);
    }

    @Override
    public void request(String serviceName, String methodName, Object[] args, WsonrpcCallback callback)
            throws IOException, WsonrpcException {
        wsonrpcEngine.request(session, serviceName, methodName, args, callback);
    }
    
}
