/*
 * Copyright (C) 2016, apexes.net. All rights reserved.
 * 
 *        http://www.apexes.net
 * 
 */
package net.apexes.wsonrpc.json.support;

import java.io.StringReader;
import java.util.List;

import com.google.gson.Gson;
import com.google.gson.JsonArray;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;

import net.apexes.wsonrpc.core.JsonException;
import net.apexes.wsonrpc.json.JsonImplementor;
import net.apexes.wsonrpc.json.JsonNode;

/**
 * 
 * @author <a href="mailto:hedyn@foxmail.com">HeDYn</a>
 */
public class GsonImplementor implements JsonImplementor {
    
    private final Gson gson;
    
    public GsonImplementor() {
        this(new Gson());
    }
    
    public GsonImplementor(Gson gson) {
        this.gson = gson;
    }
    
    @Override
    public JsonNode fromJson(String json) throws JsonException {
        JsonParser parser = new JsonParser();
        JsonElement jsonElement = parser.parse(new StringReader(json));
        return new GsonJsonNode(jsonElement);
    }

    @Override
    public String toJson(JsonNode node) throws JsonException {
        GsonJsonNode gsonNode = (GsonJsonNode) node;
        return gsonNode.toString();
    }

    @Override
    public String toJson(List<JsonNode> nodeList) throws JsonException {
        JsonArray array = new JsonArray();
        for (JsonNode node : nodeList) {
            GsonJsonNode gsonNode = (GsonJsonNode) node;
            array.add(gsonNode.jsonElement);
        }
        return array.toString();
    }

    @Override
    public JsonNode createNode() {
        return new GsonJsonNode(new JsonObject());
    }

    @Override
    public boolean isCompatible(JsonNode node, Class<?> classType) {
        return true;
    }

    @Override
    public <T> T convert(JsonNode node, Class<T> classType) {
        GsonJsonNode gsonNode = (GsonJsonNode) node;
        return gson.fromJson(gsonNode.jsonElement, classType);
    }
    
    @Override
    public JsonNode convert(Object object) {
        JsonElement jsonElement = gson.toJsonTree(object);
        return new GsonJsonNode(jsonElement);
    }

    /**
     * 
     * @author <a href="mailto:hedyn@foxmail.com">HeDYn</a>
     *
     */
    private static class GsonJsonNode implements JsonNode {
        
        private final JsonElement jsonElement;
        private final JsonObject jsonObject;
        
        public GsonJsonNode(JsonElement jsonElement) {
            this.jsonElement = jsonElement;
            if (jsonElement instanceof JsonObject) {
                jsonObject = (JsonObject) jsonElement;
            } else {
                jsonObject = null;
            }
        }

        @Override
        public boolean has(String name) {
            if (jsonObject == null) {
                return false;
            }
            return jsonObject.has(name);
        }

        @Override
        public JsonNode get(String name) {
            if (jsonObject == null) {
                return null;
            }
            return new GsonJsonNode(jsonObject.get(name));
        }

        @Override
        public Integer getInteger(String name) {
            if (jsonObject == null) {
                return null;
            }
            JsonElement je = jsonObject.get(name);
            if (je.isJsonNull()) {
                return null;
            }
            return je.getAsInt();
        }

        @Override
        public String getString(String name) {
            if (jsonObject == null) {
                return null;
            }
            JsonElement je = jsonObject.get(name);
            if (je.isJsonNull()) {
                return null;
            }
            return je.getAsString();
        }

        @Override
        public JsonNode[] getArray(String name) {
            if (jsonObject == null) {
                return null;
            }
            JsonElement je = jsonObject.get(name);
            if (je.isJsonNull()) {
                return new JsonNode[0];
            }
            JsonArray jsonArray = je.getAsJsonArray();
            int size = jsonArray.size();
            JsonNode[] results = new JsonNode[size];
            for (int i = 0; i < size; i++) {
                results[i] = new GsonJsonNode(jsonArray.get(i));
            }
            return results;
        }

        @Override
        public void put(String name, int value) {
            if (jsonObject == null) {
                throw new UnsupportedOperationException();
            }
            jsonObject.addProperty(name, value);
        }

        @Override
        public void put(String name, String value) {
            if (jsonObject == null) {
                throw new UnsupportedOperationException();
            }
            jsonObject.addProperty(name, value);
        }

        @Override
        public void put(String name, JsonNode value) {
            if (jsonObject == null) {
                throw new UnsupportedOperationException();
            }
            if (value instanceof GsonJsonNode) {
                GsonJsonNode gsonNode = (GsonJsonNode) value;
                jsonObject.add(name, gsonNode.jsonElement);
            } else {
                throw new UnsupportedOperationException("value must be GsonNode");
            }
        }

        @Override
        public void put(String name, JsonNode[] array) {
            if (jsonObject == null) {
                throw new UnsupportedOperationException();
            }
            JsonArray jsonArray = new JsonArray();
            for (int i = 0; i < array.length; i++) {
                if (array[i] instanceof GsonJsonNode) {
                    GsonJsonNode gsonNode = (GsonJsonNode) array[i];
                    jsonArray.add(gsonNode.jsonElement);
                } else {
                    throw new UnsupportedOperationException("array[" + i + "] must be GsonNode");
                }
            }
            jsonObject.add(name, jsonArray);
        }

        @Override
        public String toJson() {
            return jsonElement.toString();
        }

        @Override
        public String toString() {
            return toJson();
        }
        
    }

}
