/*
 * Copyright (C) 2016, apexes.net. All rights reserved.
 * 
 *        http://www.apexes.net
 * 
 */
package net.apexes.wsonrpc.json.support;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.node.JsonNodeFactory;
import com.fasterxml.jackson.databind.node.ObjectNode;
import net.apexes.wsonrpc.core.JsonException;
import net.apexes.wsonrpc.json.JsonImplementor;
import net.apexes.wsonrpc.json.JsonNode;

import java.io.IOException;
import java.util.List;

/**
 * 
 * @author <a href="mailto:hedyn@foxmail.com">HeDYn</a>
 */
public class JacksonImplementor implements JsonImplementor {
    
    private final ObjectMapper objectMapper;
    
    public JacksonImplementor() {
        this(new ObjectMapper());
    }
    
    public JacksonImplementor(ObjectMapper objectMapper) {
        this.objectMapper = objectMapper;
    }

    @Override
    public JsonNode fromJson(String json) throws JsonException {
        try {
            return new JacksonJsonNode(objectMapper.readTree(json));
        } catch (IOException e) {
            throw new JsonException(e.getMessage(), e.getCause());
        }
    }

    @Override
    public String toJson(JsonNode node) throws JsonException {
        JacksonJsonNode jacksonNode = (JacksonJsonNode) node;
        try {
            return objectMapper.writeValueAsString(jacksonNode.jsonNode);
        } catch (JsonProcessingException e) {
            throw new JsonException(e.getMessage(), e.getCause());
        }
    }

    @Override
    public String toJson(List<JsonNode> nodeList) throws JsonException {
        ArrayNode array = objectMapper.createArrayNode();
        for (JsonNode node : nodeList) {
            JacksonJsonNode jacksonNode = (JacksonJsonNode) node;
            array.add(jacksonNode.jsonNode);
        }
        return array.toString();
    }

    @Override
    public JsonNode createNode() {
        JsonNodeFactory factory = new JsonNodeFactory(false);  
        return new JacksonJsonNode(factory.objectNode());
    }

    @Override
    public boolean isCompatible(JsonNode node, Class<?> classType) {
        return true;
    }

    @Override
    public <T> T convert(JsonNode node, Class<T> classType) {
        JacksonJsonNode jacksonNode = (JacksonJsonNode) node;
        return objectMapper.convertValue(jacksonNode.jsonNode, classType);
    }

    @Override
    public JsonNode convert(Object object) {
        return new JacksonJsonNode(objectMapper.valueToTree(object));
    }
    
    /**
     * 
     * @author <a href="mailto:hedyn@foxmail.com">HeDYn</a>
     *
     */
    private static class JacksonJsonNode implements JsonNode {
        
        private final com.fasterxml.jackson.databind.JsonNode jsonNode;
        
        public JacksonJsonNode(com.fasterxml.jackson.databind.JsonNode jsonNode) {
            this.jsonNode = jsonNode;
        }

        @Override
        public boolean has(String name) {
            return jsonNode.has(name);
        }

        @Override
        public JsonNode get(String name) {
            return new JacksonJsonNode(jsonNode.get(name));
        }

        @Override
        public Integer getInteger(String name) {
            com.fasterxml.jackson.databind.JsonNode jn = jsonNode.get(name);
            if (jn.isNull()) {
                return null;
            }
            return jn.asInt();
        }

        @Override
        public String getString(String name) {
            com.fasterxml.jackson.databind.JsonNode jn = jsonNode.get(name);
            if (jn.isNull()) {
                return null;
            }
            return jn.asText();
        }

        @Override
        public JsonNode[] getArray(String name) {
            com.fasterxml.jackson.databind.JsonNode paramNode = jsonNode.get(name);
            int size = paramNode.size();
            JsonNode[] results = new JsonNode[size];
            for (int i = 0; i < size; i++) {
                results[i] = new JacksonJsonNode(paramNode.get(i));
            }
            return results;
        }

        @Override
        public void put(String name, int value) {
            if (jsonNode instanceof ObjectNode) {
                ObjectNode objectNode = (ObjectNode) jsonNode;
                objectNode.put(name, value);
            } else {
                throw new UnsupportedOperationException(jsonNode.getClass().getName());
            }
        }

        @Override
        public void put(String name, String value) {
            if (jsonNode instanceof ObjectNode) {
                ObjectNode objectNode = (ObjectNode) jsonNode;
                objectNode.put(name, value);
            } else {
                throw new UnsupportedOperationException(jsonNode.getClass().getName());
            }
        }

        @Override
        public void put(String name, JsonNode value) {
            if (jsonNode instanceof ObjectNode) {
                ObjectNode objectNode = (ObjectNode) jsonNode;
                if (value instanceof JacksonJsonNode) {
                    JacksonJsonNode jacksonNode = (JacksonJsonNode) value;
                    objectNode.set(name, jacksonNode.jsonNode);
                } else {
                    throw new UnsupportedOperationException("value must be JacksonNode");
                }
            } else {
                throw new UnsupportedOperationException(jsonNode.getClass().getName());
            }
        }

        @Override
        public void put(String name, JsonNode[] array) {
            if (jsonNode instanceof ObjectNode) {
                ObjectNode objectNode = (ObjectNode) jsonNode;
                ArrayNode arrayNode = objectNode.arrayNode();
                for (int i = 0; i < array.length; i++) {
                    if (array[i] instanceof JacksonJsonNode) {
                        JacksonJsonNode jacksonNode = (JacksonJsonNode) array[i];
                        arrayNode.add(jacksonNode.jsonNode);
                    } else {
                        throw new UnsupportedOperationException("array[" + i + "] must be JacksonNode");
                    }
                }
                objectNode.set(name, arrayNode);
            } else {
                throw new UnsupportedOperationException(jsonNode.getClass().getName());
            }
        }

        @Override
        public String toJson() {
            return jsonNode.toString();
        }

        @Override
        public String toString() {
            return toJson();
        }
    }

}
