/*
 * Copyright (C) 2016, apexes.net. All rights reserved.
 * 
 *        http://www.apexes.net
 * 
 */
package net.apexes.wsonrpc.core;

import net.apexes.wsonrpc.json.JsonImplementor;
import net.apexes.wsonrpc.json.JsonRpcMessage;
import net.apexes.wsonrpc.json.JsonRpcRequest;
import net.apexes.wsonrpc.json.JsonRpcResponse;
import net.apexes.wsonrpc.util.JsonRpcErrors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Type;

/**
 * @author <a href="mailto:hedyn@foxmail.com">HeDYn</a>
 */
public class JsonRpcEngine {
    private static final Logger LOG = LoggerFactory.getLogger(JsonRpcEngine.class);

    private final JsonImplementor jsonImpl;
    private final BinaryWrapper binaryWrapper;
    private final ServiceRegistry serviceRegistry;
    private JsonRpcLogger jsonRpcLogger;

    public JsonRpcEngine(JsonImplementor jsonImpl) {
        this(jsonImpl, null);
    }

    public JsonRpcEngine(JsonImplementor jsonImpl, BinaryWrapper binaryWrapper) {
        if (jsonImpl == null) {
            throw new NullPointerException("jsonImpl");
        }
        this.jsonImpl = jsonImpl;
        this.binaryWrapper = binaryWrapper;
        this.serviceRegistry = new ServiceRegistry();
    }

    /**
     * @return 返回 {@link JsonImplementor} 对象
     */
    protected JsonImplementor getJsonImplementor() {
        return jsonImpl;
    }

    public ServiceRegistry getServiceRegistry() {
        return serviceRegistry;
    }

    public JsonRpcLogger getJsonRpcLogger() {
        return jsonRpcLogger;
    }

    public void setJsonRpcLogger(JsonRpcLogger jsonRpcLogger) {
        this.jsonRpcLogger = jsonRpcLogger;
    }

    /**
     * 远程调用方法。
     *
     * @param serviceName 服务名
     * @param methodName  方法名
     * @param args        参数
     * @param id          请求ID
     * @param transport   {@link Transport}实例
     * @throws IOException
     * @throws JsonException
     */
    public void invoke(String serviceName, String methodName, Object[] args, String id, Transport transport)
            throws IOException, JsonException {
        transmit(transport, createRequest(serviceName, methodName, args, id));
    }

    protected JsonRpcRequest createRequest(String serviceName, String methodName, Object[] args, String id) {
        if (methodName == null) {
            throw new NullPointerException("methodName");
        }

        String method;
        if (serviceName == null || serviceName.isEmpty()) {
            method = methodName;
        } else {
            method = serviceName + "." + methodName;
        }
        return jsonImpl.createRequest(id, method, args);
    }

    /**
     * 接收远端的调用请求，并将回复执行结果。
     *
     * @param bytes     接收到的数据
     * @param transport {@link Transport} 实例
     * @throws IOException
     * @throws JsonException
     */
    public void receiveRequest(byte[] bytes, Transport transport) throws IOException, JsonException {
        JsonRpcMessage resp;
        try {
            JsonRpcMessage msg = receive(bytes);
            if (msg.isRequest()) {
                resp = execute((JsonRpcRequest) msg);
            } else {
                resp = jsonImpl.createResponse(JsonRpcErrors.invalidRequestError());
            }
        } catch (JsonException e) {
            resp = jsonImpl.createResponse(JsonRpcErrors.parseError(e));
        } catch (IOException e) {
            resp = jsonImpl.createResponse(JsonRpcErrors.internalError(e));
        }
        transmit(transport, resp);
    }

    /**
     * 接收远程调用得到的回复，从回复中返回指定类型的对象。
     *
     * @param bytes      接收到的字节数组
     * @param returnType 返回的对象类型
     * @return 返回指定类型的对象
     * @throws IOException      IO错误
     * @throws JsonException
     * @throws RemoteException  远程方法抛出异常
     */
    public <T> T receiveResponse(byte[] bytes, Type returnType) throws IOException, JsonException, RemoteException {
        JsonRpcMessage msg;
        try {
            msg = receive(bytes);
        } catch (JsonException e) {
            throw new JsonException("parse response error", e);
        }
        if (msg instanceof JsonRpcResponse) {
            return fromResponse((JsonRpcResponse) msg, returnType);
        } else {
            throw new JsonException("invalid response");
        }
    }

    /**
     * 处理远端的调用请求，执行相应的方法并返回执行结果。
     *
     * @param request
     * @return 如果request为通知将返回 null
     */
    protected JsonRpcResponse execute(JsonRpcRequest request) {
        if (request == null) {
            return jsonImpl.createResponse(JsonRpcErrors.parseError());
        }

        String id = request.getId();
        String serviceMethod = request.getMethod();

        ReflectServiceMethodInvoker invoker = serviceRegistry.getInvoker(serviceMethod);
        if (invoker == null) {
            return jsonImpl.createResponse(id, JsonRpcErrors.methodNotFoundError());
        }

        if (invoker.validateParameters(request)) {
            try {
                Object invokeValue = invoker.invoke();

                if (id == null) {
                    return null;
                }

                return jsonImpl.createResponse(id, invokeValue);
            } catch (Throwable t) {
                if (t instanceof InvocationTargetException) {
                    t = ((InvocationTargetException) t).getTargetException();
                }
                LOG.debug("executing error : " + serviceMethod, t);
                return jsonImpl.createResponse(id, JsonRpcErrors.serverError(2, "Server error", t));
            }
        } else {
            return jsonImpl.createResponse(id, JsonRpcErrors.invalidParamsError());
        }
    }

    protected <T> T fromResponse(JsonRpcResponse resp, Type returnType) throws JsonException, RemoteException {
        if (resp.getError() != null) {
            throw RemoteException.of(resp.getError());
        }

        Object resultValue = resp.getResultValue(returnType);
        if (resultValue == null) {
            return null;
        }

        try {
            return (T) resultValue;
        } catch (Throwable t) {
            throw new JsonException(t.getMessage(), t);
        }
    }

    protected JsonRpcMessage receive(byte[] bytes) throws IOException, JsonException {
        JsonRpcLogger logger = jsonRpcLogger;
        return receive(bytes, logger);
    }

    protected JsonRpcMessage receive(byte[] bytes, JsonRpcLogger logger) throws IOException, JsonException {
        if (binaryWrapper != null) {
            LOG.debug(" - {}", bytes.length);
            bytes = binaryWrapper.read(bytes);
            LOG.debug(" = {}", bytes.length);
        }

        String json = new String(bytes, "UTF-8");

        if (logger != null) {
            logger.onReceive(json);
        } else {
            LOG.debug(" <<  {}", json);
        }

        return jsonImpl.fromJson(json);
    }

    protected void transmit(Transport transport, JsonRpcMessage message)throws IOException, JsonException {
        JsonRpcLogger logger = jsonRpcLogger;
        transmit(transport, message, logger);
    }

    protected void transmit(Transport transport, JsonRpcMessage message, JsonRpcLogger logger)
            throws IOException, JsonException {
        String json = message.toJson();

        if (logger != null) {
            logger.onTransmit(json);
        } else {
            LOG.debug(" >>  {}", json);
        }

        byte[] bytes = json.getBytes("UTF-8");
        if (binaryWrapper != null) {
            LOG.debug(" = {}", bytes.length);
            bytes = binaryWrapper.write(bytes);
            LOG.debug(" - {}", bytes.length);
        }
        transport.sendBinary(bytes);
    }

}
