/*
 * Copyright (c) 2017, apexes.net. All rights reserved.
 *
 *         http://www.apexes.net
 *
 */
package net.apexes.wsonrpc.core;

import java.lang.reflect.Method;
import java.util.*;

/**
 * @author <a href=mailto:hedyn@foxmail.com>HeDYn</a>
 */
class ReflectServiceHolder<T> {

    private final T instance;
    private final Map<String, Set<Method>> methodFinder;

    ReflectServiceHolder(Class<T> interfaceClass, T instance) {
        if (interfaceClass == null) {
            throw new NullPointerException("interfaceClass");
        }
        if (instance == null) {
            throw new NullPointerException("instance");
        }
        if (!interfaceClass.isInterface()) {
            throw new IllegalArgumentException("class should be an interface : " + interfaceClass);
        }
        if (!interfaceClass.isInstance(instance)) {
            throw new IllegalArgumentException("instance should be implements " + interfaceClass);
        }

        this.instance = instance;

        Map<String, Set<Method>> map = new HashMap<>();

        for (Method method : interfaceClass.getMethods()) {
            String methodName = method.getName();
            Set<Method> set = map.get(methodName);
            if (set == null) {
                set = new HashSet<>();
                map.put(methodName, set);
            }
            set.add(method);
        }

        this.methodFinder = Collections.unmodifiableMap(map);
    }

    ReflectServiceMethodInvoker findInvoker(String methodName) {
        Set<Method> methods = methodFinder.get(methodName);
        if (methods == null) {
            return null;
        }
        return new ReflectServiceMethodInvoker(instance, methods);
    }

    private static String argumentTypesToString(Class<?>[] argTypes) {
        StringBuilder buf = new StringBuilder();
        buf.append("(");
        if (argTypes != null) {
            for (int i = 0; i < argTypes.length; i++) {
                if (i > 0) {
                    buf.append(", ");
                }
                Class<?> c = argTypes[i];
                buf.append((c == null) ? "null" : c.getName());
            }
        }
        buf.append(")");
        return buf.toString();
    }
}
