/*
 * Copyright (C) 2016, apexes.net. All rights reserved.
 * 
 *        http://www.apexes.net
 * 
 */
package net.apexes.wsonrpc.core;

import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

/**
 * 
 * @author <a href="mailto:hedyn@foxmail.com">HeDYn</a>
 *
 */
public class ServiceRegistry {

    private final ConcurrentMap<String, ReflectServiceHolder> holders;

    public ServiceRegistry() {
        holders = new ConcurrentHashMap<>();
    }

    public <T> Binder<T> bind(Class<T> interfaceClass) {
        return new Binder<>(interfaceClass);
    }

    public <T> Binder<T> simpleNameIfNull(Class<T> interfaceClass) {
        return new Binder<>(interfaceClass).simpleNameIfNull();
    }

    public <T> ServiceRegistry register(String name, Class<T> interfaceClass, T instance) {
        if (holders.containsKey(name)) {
            throw new IllegalArgumentException("service already exists");
        }
        holders.put(name, new ReflectServiceHolder<>(interfaceClass, instance));
        return this;
    }

    public ServiceRegistry unregister(String name) {
        holders.remove(name);
        return this;
    }

    ReflectServiceMethodInvoker getInvoker(String serviceMethod) {
        String serviceName = null;
        String methodName = serviceMethod;
        int lastIndex = serviceMethod.lastIndexOf('.');
        if (lastIndex >= 0) {
            serviceName = serviceMethod.substring(0, lastIndex);
            methodName = serviceMethod.substring(lastIndex + 1);
        } else {
            return null;
        }

        ReflectServiceHolder holder = holders.get(serviceName);
        if (holder != null) {
            return holder.findInvoker(methodName);
        }
        return null;
    }

    /**
     *
     * @author <a href="mailto:hedyn@foxmail.com">HeDYn</a>
     *
     */
    public class Binder<T> {

        private final Class<T> interfaceClass;
        private String customServiceName;
        private boolean simpleNameIfNull = false;

        Binder(Class<T> interfaceClass) {
            this.interfaceClass = interfaceClass;
        }

        /**
         * 自定义服务名称
         * @param serviceName 服务名称
         * @return
         */
        public Binder<T> serviceName(String serviceName) {
            this.customServiceName = serviceName;
            this.simpleNameIfNull = false;
            return this;
        }

        /**
         * 如果未设置服务名称就使用serviceClass的simpleName
         * @return
         */
        public Binder<T> simpleNameIfNull() {
            this.customServiceName = null;
            this.simpleNameIfNull = true;
            return this;
        }

        public ServiceRegistry toInstance(T instance) {
            String serviceName;
            if (customServiceName != null) {
                serviceName = customServiceName;
            } else if (simpleNameIfNull) {
                serviceName = interfaceClass.getSimpleName();
            } else {
                serviceName = interfaceClass.getName();
            }
            return register(serviceName, interfaceClass, instance);
        }

    }

}
