/*
 * Copyright (C) 2016, apexes.net. All rights reserved.
 * 
 *        http://www.apexes.net
 * 
 */
package net.apexes.wsonrpc.core;

import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Method;
import java.lang.reflect.Proxy;

/**
 * @author <a href="mailto:hedyn@foxmail.com">HeDYn</a>
 *
 */
public final class RemoteInvoker {
    
    public static RemoteInvoker create(Remote remote) {
        return new RemoteInvoker(remote);
    }

    private final Remote remote;
    private String customServiceName;
    private boolean simpleNameIfNull = false;
    private ClassLoader classLoader;
    private int timeout;
    
    private RemoteInvoker(Remote remote) {
        this.remote = remote;
    }
    
    /**
     * 自定义服务名称
     * @param serviceName 服务名称
     * @return
     */
    public RemoteInvoker serviceName(String serviceName) {
        this.customServiceName = serviceName;
        this.simpleNameIfNull = false;
        return this;
    }

    /**
     * 如果未设置服务名称就使用serviceClass的simpleName
     * @return
     */
    public RemoteInvoker simpleNameIfNull() {
        this.customServiceName = null;
        this.simpleNameIfNull = true;
        return this;
    }

    /**
     * 
     * @param classLoader
     * @return
     */
    public RemoteInvoker classLoader(ClassLoader classLoader) {
        this.classLoader = classLoader;
        return this;
    }

    /**
     * 设置超时时间，0表示永不超时。单位为TimeUnit.MILLISECONDS
     * 
     * @param timeout
     * @return
     */
    public RemoteInvoker timeout(int timeout) {
        this.timeout = timeout;
        return this;
    }

    /**
     * 获取指定类型的对象
     * @param serviceClass
     * @return
     */
    @SuppressWarnings("unchecked")
    public <T> T get(final Class<T> serviceClass) {
        String serviceName;
        if (customServiceName != null) {
            serviceName = customServiceName;
        } else if (simpleNameIfNull) {
            serviceName = serviceClass.getSimpleName();
        } else {
            serviceName = serviceClass.getName();
        }
        if (classLoader == null) {
            classLoader = serviceClass.getClassLoader();
        }
        InvocationHandler handler = new InvocationHandlerImpl(remote, serviceName, timeout);
        return (T) Proxy.newProxyInstance(classLoader, new Class<?>[] { serviceClass }, handler);
    }

    /**
     * @author <a href="mailto:hedyn@foxmail.com">HeDYn</a>
     *
     */
    private static class InvocationHandlerImpl implements InvocationHandler {

        private final Remote remote;
        private final String serviceName;
        private final int timeout;

        InvocationHandlerImpl(Remote remote, String serviceName, int timeout) {
            this.remote = remote;
            this.serviceName = serviceName;
            this.timeout = timeout;
        }

        @Override
        public Object invoke(Object proxy, Method method, Object[] args) throws Throwable {
            if (method.getDeclaringClass() == Object.class) {
                return proxyObjectMethods(method, proxy, args);
            }
            Class<?> returnType = method.getReturnType();
            if (returnType == void.class) {
                remote.notify(serviceName, method.getName(), args);
                return null;
            }
            return remote.request(serviceName, method.getName(), args, returnType, timeout);
        }

    }

    private static Object proxyObjectMethods(Method method, Object proxyObject, Object[] args) {
        String name = method.getName();
        if (name.equals("toString")) {
            return proxyObject.getClass().getName() + "@" + System.identityHashCode(proxyObject);
        }
        if (name.equals("hashCode")) {
            return System.identityHashCode(proxyObject);
        }
        if (name.equals("equals")) {
            return proxyObject == args[0];
        }
        throw new RuntimeException(method.getName() + " is not a member of java.lang.Object");
    }

}
