/*
 * Copyright (C) 2016, apexes.net. All rights reserved.
 * 
 *        http://www.apexes.net
 * 
 */
package net.apexes.wsonrpc.core;

import net.apexes.wsonrpc.json.JsonImplementor;
import net.apexes.wsonrpc.json.JsonRpcMessage;
import net.apexes.wsonrpc.json.JsonRpcRequest;
import net.apexes.wsonrpc.json.JsonRpcResponse;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.util.concurrent.Future;

/**
 * 
 * @author <a href="mailto:hedyn@foxmail.com">HeDYn</a>
 *
 */
public class WsonrpcEngine {

    private static final Logger LOG = LoggerFactory.getLogger(WsonrpcEngine.class);

    protected final WsonrpcConfig config;
    protected final JsonRpcEngine jsonRpcEngine;
    protected final WsonrpcLogger wsonrpcLogger;
    private final WsonrpcCallbackCache callbackCache;

    public WsonrpcEngine(WsonrpcConfig config) {
        this.config = config;
        this.jsonRpcEngine = new JsonRpcEngine(config.getJsonImplementor(), config.getBinaryWrapper());
        this.wsonrpcLogger = config.getWsonrpcLogger();
        this.callbackCache = new WsonrpcCallbackCache();
    }

    public final WsonrpcConfig getConfig() {
        return config;
    }

    public ServiceRegistry getServiceRegistry() {
        return jsonRpcEngine.getServiceRegistry();
    }

    public void notify(WebSocketSession session, String serviceName, String methodName, Object[] args)
            throws IOException, WsonrpcException {
        if (session == null) {
            throw new NullPointerException("session");
        }
        jsonRpcEngine.invoke(serviceName, methodName, args, null, session);
    }

    public <T> Future<T> request(WebSocketSession session, String serviceName, String methodName, Object[] args,
                                 Class<T> returnType) throws IOException, WsonrpcException {
        String id = config.getIdGenerater().next();
        WsonrpcFuture future = new WsonrpcFuture(this, returnType);
        request(session, id, serviceName, methodName, args, future);
        return future;
    }

    public void request(WebSocketSession session, String serviceName, String methodName, Object[] args,
                        WsonrpcCallback callback) throws IOException, WsonrpcException {
        String id = config.getIdGenerater().next();
        request(session, id, serviceName, methodName, args, callback);
    }

    private void request(WebSocketSession session, String id, String serviceName, String methodName, Object[] args,
                           WsonrpcCallback callback) throws IOException, WsonrpcException {
        if (session == null) {
            throw new NullPointerException("session is null");
        }
        callbackCache.put(id, callback);
        try {
            JsonRpcRequest request = jsonRpcEngine.createRequest(serviceName, methodName, args, id);
            if (wsonrpcLogger != null) {
                jsonRpcEngine.transmit(session, request, new JsonRpcLoggerAdapter(session, wsonrpcLogger));
            } else {
                jsonRpcEngine.transmit(session, request);
            }
        } catch (Throwable t) {
            callbackCache.out(id);
            if (t instanceof IOException) {
                throw (IOException) t;
            } else if (t instanceof WsonrpcException) {
                throw (WsonrpcException) t;
            }
            throw new WsonrpcException(t);
        }
    }

    /**
     * 处理收到的JSON数据
     * 
     * @param session session
     * @param bytes 接收到的数据
     */
    public void handle(WebSocketSession session, byte[] bytes) {
        if (session == null) {
            throw new NullPointerException("session");
        }
        try {
            JsonRpcMessage msg;
            if (wsonrpcLogger != null) {
                msg = jsonRpcEngine.receive(bytes, new JsonRpcLoggerAdapter(session, wsonrpcLogger));
            } else {
                msg = jsonRpcEngine.receive(bytes);
            }
            if (msg.isRequest()) {
                handleRequest(session, (JsonRpcRequest) msg);
            } else if (msg instanceof JsonRpcResponse) {
                handleResponse((JsonRpcResponse) msg);
            }
        } catch (Exception e) {
            if (wsonrpcLogger != null) {
                wsonrpcLogger.onError(session.getId(), e);
            }
        }
    }

    protected void handleRequest(WebSocketSession session, JsonRpcRequest request) {
        transmit(session, execute(session, request));
    }

    protected void handleResponse(JsonRpcResponse response) {
        String id = response.getId();
        if (id == null) {
            return;
        }
        WsonrpcCallback callback = callbackCache.out(id);
        if (callback == null) {
            LOG.warn("callback is null. id={}", id);
        } else {
            try {
                callback.setValue(response);
            } catch (Throwable t) {
                callback.setError(t);
            }
        }
    }

    protected JsonRpcResponse execute(WebSocketSession session, JsonRpcRequest request) {
        return jsonRpcEngine.execute(request);
    }

    protected void transmit(WebSocketSession session, JsonRpcResponse resp) {
        if (resp != null) {
            try {
                if (wsonrpcLogger != null) {
                    jsonRpcEngine.transmit(session, resp, new JsonRpcLoggerAdapter(session, wsonrpcLogger));
                } else {
                    jsonRpcEngine.transmit(session, resp);
                }
            } catch (Exception e) {
                if (wsonrpcLogger != null) {
                    wsonrpcLogger.onError(session.getId(), e);
                }
            }
        }
    }

    protected JsonImplementor getJsonImplementor() {
        return jsonRpcEngine.getJsonImplementor();
    }

    protected JsonRpcResponse toResponse(String id, Object result) {
        return getJsonImplementor().createResponse(id, result);
    }

    protected <T> T fromResponse(JsonRpcResponse resp, Class<T> returnType) throws JsonException, RemoteException {
        return jsonRpcEngine.fromResponse(resp, returnType);
    }

    /**
     *
     * @author <a href="mailto:hedyn@foxmail.com">HeDYn</a>
     */
    private static class JsonRpcLoggerAdapter implements JsonRpcLogger {

        private final WebSocketSession session;
        private final WsonrpcLogger logger;

        private JsonRpcLoggerAdapter(WebSocketSession session, WsonrpcLogger logger) {
            this.session = session;
            this.logger = logger;
        }

        @Override
        public void onReceive(String json) {
            logger.onReceive(session.getId(), json);
        }

        @Override
        public void onTransmit(String json) {
            logger.onTransmit(session.getId(), json);
        }
    }
}
