/*
 * Decompiled with CFR 0.152.
 */
package com.azure.core.http;

import com.azure.core.util.Configuration;
import com.azure.core.util.CoreUtils;
import com.azure.core.util.logging.ClientLogger;
import java.io.UnsupportedEncodingException;
import java.net.InetSocketAddress;
import java.net.MalformedURLException;
import java.net.Proxy;
import java.net.URL;
import java.net.URLDecoder;
import java.nio.charset.StandardCharsets;
import java.util.Objects;
import java.util.regex.Pattern;

public class ProxyOptions {
    private static final ClientLogger LOGGER = new ClientLogger(ProxyOptions.class);
    private static final String INVALID_CONFIGURATION_MESSAGE = "'configuration' cannot be 'Configuration.NONE'.";
    private static final String INVALID_AZURE_PROXY_URL = "Configuration {} is an invalid URL and is being ignored.";
    private static final String JAVA_SYSTEM_PROXY_PREREQUISITE = "java.net.useSystemProxies";
    private static final String JAVA_PROXY_HOST = "proxyHost";
    private static final String JAVA_PROXY_PORT = "proxyPort";
    private static final String JAVA_PROXY_USER = "proxyUser";
    private static final String JAVA_PROXY_PASSWORD = "proxyPassword";
    private static final String JAVA_NON_PROXY_HOSTS = "http.nonProxyHosts";
    private static final String HTTPS = "https";
    private static final int DEFAULT_HTTPS_PORT = 443;
    private static final String HTTP = "http";
    private static final int DEFAULT_HTTP_PORT = 80;
    private final InetSocketAddress address;
    private final Type type;
    private String username;
    private String password;
    private String nonProxyHosts;

    public ProxyOptions(Type type, InetSocketAddress address) {
        this.type = type;
        this.address = address;
    }

    public ProxyOptions setCredentials(String username, String password) {
        this.username = Objects.requireNonNull(username, "'username' cannot be null.");
        this.password = Objects.requireNonNull(password, "'password' cannot be null.");
        return this;
    }

    public ProxyOptions setNonProxyHosts(String nonProxyHosts) {
        this.nonProxyHosts = ProxyOptions.sanitizeJavaHttpNonProxyHosts(nonProxyHosts);
        return this;
    }

    public InetSocketAddress getAddress() {
        return this.address;
    }

    public Type getType() {
        return this.type;
    }

    public String getUsername() {
        return this.username;
    }

    public String getPassword() {
        return this.password;
    }

    public String getNonProxyHosts() {
        return this.nonProxyHosts;
    }

    public static ProxyOptions fromConfiguration(Configuration configuration) {
        return ProxyOptions.fromConfiguration(configuration, false);
    }

    public static ProxyOptions fromConfiguration(Configuration configuration, boolean createUnresolved) {
        if (configuration == Configuration.NONE) {
            throw LOGGER.logExceptionAsWarning(new IllegalArgumentException(INVALID_CONFIGURATION_MESSAGE));
        }
        Configuration proxyConfiguration = configuration == null ? Configuration.getGlobalConfiguration() : configuration;
        return ProxyOptions.attemptToLoadProxy(proxyConfiguration, createUnresolved);
    }

    private static ProxyOptions attemptToLoadProxy(Configuration configuration, boolean createUnresolved) {
        ProxyOptions proxyOptions;
        if (Boolean.parseBoolean(configuration.get(JAVA_SYSTEM_PROXY_PREREQUISITE))) {
            proxyOptions = ProxyOptions.attemptToLoadSystemProxy(configuration, createUnresolved, "HTTPS_PROXY");
            if (proxyOptions != null) {
                LOGGER.verbose("Using proxy created from HTTPS_PROXY environment variable.");
                return proxyOptions;
            }
            proxyOptions = ProxyOptions.attemptToLoadSystemProxy(configuration, createUnresolved, "HTTP_PROXY");
            if (proxyOptions != null) {
                LOGGER.verbose("Using proxy created from HTTP_PROXY environment variable.");
                return proxyOptions;
            }
        }
        if ((proxyOptions = ProxyOptions.attemptToLoadJavaProxy(configuration, createUnresolved, HTTPS)) != null) {
            LOGGER.verbose("Using proxy created from JVM HTTPS system properties.");
            return proxyOptions;
        }
        proxyOptions = ProxyOptions.attemptToLoadJavaProxy(configuration, createUnresolved, HTTP);
        if (proxyOptions != null) {
            LOGGER.verbose("Using proxy created from JVM HTTP system properties.");
            return proxyOptions;
        }
        return null;
    }

    private static ProxyOptions attemptToLoadSystemProxy(Configuration configuration, boolean createUnresolved, String proxyProperty) {
        String proxyConfiguration = configuration.get(proxyProperty);
        if (CoreUtils.isNullOrEmpty(proxyConfiguration)) {
            return null;
        }
        try {
            String[] usernamePassword;
            String userInfo;
            URL proxyUrl = new URL(proxyConfiguration);
            int port = proxyUrl.getPort() == -1 ? proxyUrl.getDefaultPort() : proxyUrl.getPort();
            InetSocketAddress socketAddress = createUnresolved ? InetSocketAddress.createUnresolved(proxyUrl.getHost(), port) : new InetSocketAddress(proxyUrl.getHost(), port);
            ProxyOptions proxyOptions = new ProxyOptions(Type.HTTP, socketAddress);
            String nonProxyHostsString = configuration.get("NO_PROXY");
            if (!CoreUtils.isNullOrEmpty(nonProxyHostsString)) {
                proxyOptions.nonProxyHosts = ProxyOptions.sanitizeNoProxy(nonProxyHostsString);
            }
            if ((userInfo = proxyUrl.getUserInfo()) != null && (usernamePassword = userInfo.split(":", 2)).length == 2) {
                try {
                    proxyOptions.setCredentials(URLDecoder.decode(usernamePassword[0], StandardCharsets.UTF_8.toString()), URLDecoder.decode(usernamePassword[1], StandardCharsets.UTF_8.toString()));
                }
                catch (UnsupportedEncodingException e) {
                    return null;
                }
            }
            return proxyOptions;
        }
        catch (MalformedURLException ex) {
            LOGGER.warning(INVALID_AZURE_PROXY_URL, proxyProperty);
            return null;
        }
    }

    private static String sanitizeNoProxy(String noProxyString) {
        CharSequence[] nonProxyHosts = noProxyString.split(",");
        for (int i = 0; i < nonProxyHosts.length; ++i) {
            String prefixWildcard = "";
            String suffixWildcard = "";
            CharSequence body = nonProxyHosts[i];
            if (((String)body).startsWith(".*")) {
                prefixWildcard = ".*";
                body = ((String)body).substring(2);
            } else if (((String)body).startsWith("*") || ((String)body).startsWith(".")) {
                prefixWildcard = ".*";
                body = ((String)body).substring(1);
            }
            if (((String)body).endsWith(".*")) {
                suffixWildcard = ".*";
                body = ((String)body).substring(0, ((String)body).length() - 2);
            } else if (((String)body).endsWith("*") || ((String)body).endsWith(".")) {
                suffixWildcard = ".*";
                body = ((String)body).substring(0, ((String)body).length() - 1);
            }
            nonProxyHosts[i] = prefixWildcard + Pattern.quote((String)body) + suffixWildcard;
        }
        return String.join((CharSequence)"|", nonProxyHosts);
    }

    private static ProxyOptions attemptToLoadJavaProxy(Configuration configuration, boolean createUnresolved, String type) {
        int port;
        String host = configuration.get(type + "." + JAVA_PROXY_HOST);
        if (CoreUtils.isNullOrEmpty(host)) {
            return null;
        }
        try {
            port = Integer.parseInt(configuration.get(type + "." + JAVA_PROXY_PORT));
        }
        catch (NumberFormatException ex) {
            port = HTTPS.equals(type) ? 443 : 80;
        }
        InetSocketAddress socketAddress = createUnresolved ? InetSocketAddress.createUnresolved(host, port) : new InetSocketAddress(host, port);
        ProxyOptions proxyOptions = new ProxyOptions(Type.HTTP, socketAddress);
        String nonProxyHostsString = configuration.get(JAVA_NON_PROXY_HOSTS);
        if (!CoreUtils.isNullOrEmpty(nonProxyHostsString)) {
            proxyOptions.setNonProxyHosts(nonProxyHostsString);
        }
        String username = configuration.get(type + "." + JAVA_PROXY_USER);
        String password = configuration.get(type + "." + JAVA_PROXY_PASSWORD);
        if (username != null && password != null) {
            proxyOptions.setCredentials(username, password);
        }
        return proxyOptions;
    }

    private static String sanitizeJavaHttpNonProxyHosts(String nonProxyHostsString) {
        CharSequence[] nonProxyHosts = nonProxyHostsString.split("\\|");
        for (int i = 0; i < nonProxyHosts.length; ++i) {
            String prefixWildcard = "";
            String suffixWildcard = "";
            CharSequence body = nonProxyHosts[i];
            if (((String)body).startsWith("*")) {
                prefixWildcard = ".*";
                body = ((String)body).substring(1);
            }
            if (((String)body).endsWith("*")) {
                suffixWildcard = ".*";
                body = ((String)body).substring(0, ((String)body).length() - 1);
            }
            nonProxyHosts[i] = prefixWildcard + Pattern.quote((String)body) + suffixWildcard;
        }
        return String.join((CharSequence)"|", nonProxyHosts);
    }

    public static enum Type {
        HTTP(Proxy.Type.HTTP),
        SOCKS4(Proxy.Type.SOCKS),
        SOCKS5(Proxy.Type.SOCKS);

        private final Proxy.Type proxyType;

        private Type(Proxy.Type proxyType) {
            this.proxyType = proxyType;
        }

        public Proxy.Type toProxyType() {
            return this.proxyType;
        }
    }
}

