package net.java.ao.test.jdbc;

import com.google.common.io.Files;
import org.h2.tools.Server;

import java.io.File;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

import static org.h2.engine.Constants.BUILD_ID;
import static org.h2.engine.Constants.VERSION_MAJOR;
import static org.h2.engine.Constants.VERSION_MINOR;

public class H2Server extends AbstractJdbcConfiguration {
    private static final File TEMP_DIR = Files.createTempDir();

    private static final String DEFAULT_URL = "jdbc:h2:tcp://localhost/" + TEMP_DIR.getPath() + "/ao-test;MVCC=TRUE";
    private static final String DEFAULT_USER = "";
    private static final String DEFAULT_PASSWORD = "";
    private static final String DEFAULT_SCHEMA = "PUBLIC";

    private static Server h2Server;

    private static Lock h2ServerLock = new ReentrantLock();

    public H2Server() {
        this(DEFAULT_URL, DEFAULT_USER, DEFAULT_PASSWORD, DEFAULT_SCHEMA);
    }

    public H2Server(String url, String username, String password, String schema) {
        super(url, username, password, schema);
    }

    @Override
    protected String getDefaultUsername() {
        return DEFAULT_USER;
    }

    @Override
    protected String getDefaultPassword() {
        return DEFAULT_PASSWORD;
    }

    @Override
    protected String getDefaultSchema() {
        return DEFAULT_SCHEMA;
    }

    @Override
    protected String getDefaultUrl() {
        return DEFAULT_URL;
    }

    @Override
    public void init() {
        h2ServerLock.lock();
        try {
            if (h2Server == null) {
                // launch an H2 server if there isn't one
                try {
                    h2Server = Server.createTcpServer(getTcpServerParams()).start();
                } catch (SQLException e) {
                    throw new RuntimeException(e);
                }
            }
        } finally {
            h2ServerLock.unlock();
        }
    }

    private static String[] getTcpServerParams() {
        List<String> params = new ArrayList<>();
        params.add("-baseDir");
        params.add(TEMP_DIR.getPath());
        if (isH2VersionGreaterThanOrEqualTo(1, 4, 198)) {
            // 1.4.198+ requires -ifNotExists to allow clients to auto-create a DB
            params.add("-ifNotExists");
        }
        return params.toArray(new String[0]);
    }

    private static boolean isH2VersionGreaterThanOrEqualTo(int major, int minor, int buildId) {
        if (VERSION_MAJOR > major) return true;
        if (VERSION_MAJOR < major) return false;
        if (VERSION_MINOR > minor) return true;
        if (VERSION_MINOR < minor) return false;
        return BUILD_ID >= buildId;
    }
}
