/*
* The MIT License (MIT)	 
*
* Copyright (c) 2015 LeanIX GmbH
* 
* Permission is hereby granted, free of charge, to any person obtaining a copy of
* this software and associated documentation files (the "Software"), to deal in
* the Software without restriction, including without limitation the rights to
* use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of
* the Software, and to permit persons to whom the Software is furnished to do so,
* subject to the following conditions:
* 
* The above copyright notice and this permission notice shall be included in all
* copies or substantial portions of the Software.
* 
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
* IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS
* FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR
* COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
* IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
* CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
*/

package net.leanix.synclog.api;

import net.leanix.dropkit.api.Client;
import net.leanix.dropkit.api.ApiException;
import net.leanix.dropkit.api.ValidationException;
import net.leanix.dropkit.api.Response;

import javax.ws.rs.core.MultivaluedHashMap;
import javax.ws.rs.core.MediaType;
import com.sun.jersey.api.client.UniformInterfaceException;
import net.leanix.synclog.api.models.SyncItemBatch;
import net.leanix.synclog.api.models.SynchronizationListResponse;
import net.leanix.synclog.api.models.Synchronization;
import net.leanix.synclog.api.models.SynchronizationResponse;
import net.leanix.synclog.api.models.SyncItemListResponse;
import java.util.*;

public class SynchronizationsApi
{
	private Client apiClient;
	
	public SynchronizationsApi(Client client)
	{
            if (client == null)
                throw new IllegalArgumentException("client is null");
            
            this.apiClient = client;
	}
	
	public Client getClient()
	{
		return this.apiClient;
	}

	/**
         * getSynchronization.
         * Endpoint to get a specific synchronization.
         *
         * @throws ApiException
         */
	public SynchronizationResponse getSynchronization (String id) throws ApiException
	{
            // verify required params are set
            if(id == null )
            {
                throw new ApiException(400, "missing required params");
            }
            // create path and map variables
            String path = "/synchronizations/{id}".replaceAll("\\{format\\}","json").replaceAll("\\{" + "id" + "\\}", id.toString());

            // query params
            Map<String, String> queryParams = new HashMap<String, String>();
            Map<String, String> headerParams = new HashMap<String, String>();
            try
            {
                return apiClient.resource(path)
                    .queryParams(new MultivaluedHashMap(queryParams))
                    .method("GET", SynchronizationResponse.class);
            }
            catch (UniformInterfaceException ex)
            {
                if(ex.getResponse().getStatus() == 404)
                {
                    return null;
                }
                else if(ex.getResponse().getStatus() == 422)
                {
                    throw new ValidationException(ex.getResponse().getEntity(Response.class));
                }
                else
                {
                    throw new ApiException(ex.getResponse().getStatus(), ex.getResponse().toString());
                }
            }
	}
        
	/**
         * updateSynchronization.
         * Endpoint to update an synchronization.
         *
         * @throws ApiException
         */
	public SynchronizationResponse updateSynchronization (String id, Synchronization body) throws ApiException
	{
            // verify required params are set
            if(id == null )
            {
                throw new ApiException(400, "missing required params");
            }
            // create path and map variables
            String path = "/synchronizations/{id}".replaceAll("\\{format\\}","json").replaceAll("\\{" + "id" + "\\}", id.toString());

            // query params
            Map<String, String> queryParams = new HashMap<String, String>();
            Map<String, String> headerParams = new HashMap<String, String>();
            try
            {
                return apiClient.resource(path)
                    .queryParams(new MultivaluedHashMap(queryParams))
                    .type(MediaType.APPLICATION_JSON).method("PUT", SynchronizationResponse.class, body);
            }
            catch (UniformInterfaceException ex)
            {
                if(ex.getResponse().getStatus() == 404)
                {
                    return null;
                }
                else if(ex.getResponse().getStatus() == 422)
                {
                    throw new ValidationException(ex.getResponse().getEntity(Response.class));
                }
                else
                {
                    throw new ApiException(ex.getResponse().getStatus(), ex.getResponse().toString());
                }
            }
	}
        
	/**
         * getSyncItems.
         * Endpoint to list the items
         *
         * @throws ApiException
         */
	public SyncItemListResponse getSyncItems (String id, Integer page, Integer size, String sort) throws ApiException
	{
            // verify required params are set
            if(id == null )
            {
                throw new ApiException(400, "missing required params");
            }
            // create path and map variables
            String path = "/synchronizations/{id}/syncItems".replaceAll("\\{format\\}","json").replaceAll("\\{" + "id" + "\\}", id.toString());

            // query params
            Map<String, String> queryParams = new HashMap<String, String>();
            if(!"null".equals(String.valueOf(page)))
            queryParams.put("page", String.valueOf(page));
            if(!"null".equals(String.valueOf(size)))
            queryParams.put("size", String.valueOf(size));
            if(!"null".equals(String.valueOf(sort)))
            queryParams.put("sort", String.valueOf(sort));
            Map<String, String> headerParams = new HashMap<String, String>();
            try
            {
                return apiClient.resource(path)
                    .queryParams(new MultivaluedHashMap(queryParams))
                    .method("GET", SyncItemListResponse.class);
            }
            catch (UniformInterfaceException ex)
            {
                if(ex.getResponse().getStatus() == 404)
                {
                    return null;
                }
                else if(ex.getResponse().getStatus() == 422)
                {
                    throw new ValidationException(ex.getResponse().getEntity(Response.class));
                }
                else
                {
                    throw new ApiException(ex.getResponse().getStatus(), ex.getResponse().toString());
                }
            }
	}
        
	/**
         * addSyncItemBatch.
         * Endpoint to add new items
         *
         * @throws ApiException
         */
	public SynchronizationResponse addSyncItemBatch (String id, SyncItemBatch body) throws ApiException
	{
            // verify required params are set
            if(id == null )
            {
                throw new ApiException(400, "missing required params");
            }
            // create path and map variables
            String path = "/synchronizations/{id}/syncItemBatch".replaceAll("\\{format\\}","json").replaceAll("\\{" + "id" + "\\}", id.toString());

            // query params
            Map<String, String> queryParams = new HashMap<String, String>();
            Map<String, String> headerParams = new HashMap<String, String>();
            try
            {
                return apiClient.resource(path)
                    .queryParams(new MultivaluedHashMap(queryParams))
                    .type(MediaType.APPLICATION_JSON).method("POST", SynchronizationResponse.class, body);
            }
            catch (UniformInterfaceException ex)
            {
                if(ex.getResponse().getStatus() == 404)
                {
                    return null;
                }
                else if(ex.getResponse().getStatus() == 422)
                {
                    throw new ValidationException(ex.getResponse().getEntity(Response.class));
                }
                else
                {
                    throw new ApiException(ex.getResponse().getStatus(), ex.getResponse().toString());
                }
            }
	}
        
	/**
         * createSynchronization.
         * Endpoint to create an synchronization.
         *
         * @throws ApiException
         */
	public SynchronizationResponse createSynchronization (Synchronization body) throws ApiException
	{
            // create path and map variables
            String path = "/synchronizations".replaceAll("\\{format\\}","json");

            // query params
            Map<String, String> queryParams = new HashMap<String, String>();
            Map<String, String> headerParams = new HashMap<String, String>();
            try
            {
                return apiClient.resource(path)
                    .queryParams(new MultivaluedHashMap(queryParams))
                    .type(MediaType.APPLICATION_JSON).method("POST", SynchronizationResponse.class, body);
            }
            catch (UniformInterfaceException ex)
            {
                if(ex.getResponse().getStatus() == 404)
                {
                    return null;
                }
                else if(ex.getResponse().getStatus() == 422)
                {
                    throw new ValidationException(ex.getResponse().getEntity(Response.class));
                }
                else
                {
                    throw new ApiException(ex.getResponse().getStatus(), ex.getResponse().toString());
                }
            }
	}
        
	/**
         * getSynchronizations.
         * Endpoint to list the synchronizations.
         *
         * @throws ApiException
         */
	public SynchronizationListResponse getSynchronizations (String workspaceId, String userId, String topic, Integer page, Integer size, String sort) throws ApiException
	{
            // create path and map variables
            String path = "/synchronizations".replaceAll("\\{format\\}","json");

            // query params
            Map<String, String> queryParams = new HashMap<String, String>();
            if(!"null".equals(String.valueOf(workspaceId)))
            queryParams.put("workspaceId", String.valueOf(workspaceId));
            if(!"null".equals(String.valueOf(userId)))
            queryParams.put("userId", String.valueOf(userId));
            if(!"null".equals(String.valueOf(topic)))
            queryParams.put("topic", String.valueOf(topic));
            if(!"null".equals(String.valueOf(page)))
            queryParams.put("page", String.valueOf(page));
            if(!"null".equals(String.valueOf(size)))
            queryParams.put("size", String.valueOf(size));
            if(!"null".equals(String.valueOf(sort)))
            queryParams.put("sort", String.valueOf(sort));
            Map<String, String> headerParams = new HashMap<String, String>();
            try
            {
                return apiClient.resource(path)
                    .queryParams(new MultivaluedHashMap(queryParams))
                    .method("GET", SynchronizationListResponse.class);
            }
            catch (UniformInterfaceException ex)
            {
                if(ex.getResponse().getStatus() == 404)
                {
                    return null;
                }
                else if(ex.getResponse().getStatus() == 422)
                {
                    throw new ValidationException(ex.getResponse().getEntity(Response.class));
                }
                else
                {
                    throw new ApiException(ex.getResponse().getStatus(), ex.getResponse().toString());
                }
            }
	}
        
	}

