/*
* The MIT License (MIT)	 
*
* Copyright (c) 2015 LeanIX GmbH
* 
* Permission is hereby granted, free of charge, to any person obtaining a copy of
* this software and associated documentation files (the "Software"), to deal in
* the Software without restriction, including without limitation the rights to
* use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of
* the Software, and to permit persons to whom the Software is furnished to do so,
* subject to the following conditions:
* 
* The above copyright notice and this permission notice shall be included in all
* copies or substantial portions of the Software.
* 
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
* IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS
* FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR
* COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
* IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
* CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
*/

package net.leanix.webhooks.api;

import net.leanix.dropkit.api.Client;
import net.leanix.dropkit.api.ApiException;
import net.leanix.dropkit.api.ValidationException;
import net.leanix.dropkit.api.Response;
import net.leanix.webhooks.api.models.DeliveryListResponse;
import net.leanix.webhooks.api.models.Subscription;
import net.leanix.webhooks.api.models.SubscriptionResponse;
import net.leanix.webhooks.api.models.SubscriptionListResponse;
import com.sun.jersey.api.client.UniformInterfaceException;
import com.sun.jersey.core.util.MultivaluedMapImpl;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.MultivaluedMap;

import java.util.HashMap;
import java.util.Map;
import java.util.List;

public class SubscriptionsApi
{
	private Client apiClient;
	
	public SubscriptionsApi(Client client)
	{
            if (client == null)
                throw new IllegalArgumentException("client is null");
            
            this.apiClient = client;
	}
	
	public Client getClient()
	{
		return this.apiClient;
	}

    private static MultivaluedMap<String, String> buildmvm(Map<String, String> map) {
        MultivaluedMapImpl mvMap = new MultivaluedMapImpl();
        for (Map.Entry<String, String> entry: map.entrySet()) {
            mvMap.add(entry.getKey(), entry.getValue());
        }
        return mvMap;
    }

	/**
         * getSubscription.
         * returns a subscription with a given id
         *
         * @throws ApiException
         */
	public SubscriptionResponse getSubscription (String id) throws ApiException
	{
            // verify required params are set
            if(id == null )
            {
                throw new ApiException(400, "missing required params");
            }
            // create path and map variables
            String path = "/subscriptions/{id}".replaceAll("\\{format\\}","json").replaceAll("\\{" + "id" + "\\}", id.toString());

            // query params
            Map<String, String> queryParams = new HashMap<String, String>();
            Map<String, String> headerParams = new HashMap<String, String>();
            try
            {
                return apiClient.resource(path)
                    .queryParams(buildmvm(queryParams))
                    .method("GET", SubscriptionResponse.class);
            }
            catch (UniformInterfaceException ex)
            {
                if(ex.getResponse().getStatus() == 404)
                {
                    return null;
                }
                else if(ex.getResponse().getStatus() == 422)
                {
                    throw new ValidationException(ex.getResponse().getEntity(Response.class));
                }
                else
                {
                    throw new ApiException(ex.getResponse().getStatus(), ex.getResponse().toString());
                }
            }
	}
        
	/**
         * updateSubscription.
         * Endpoint to update a subscription.
         *
         * @throws ApiException
         */
	public SubscriptionResponse updateSubscription (String id, Subscription body) throws ApiException
	{
            // verify required params are set
            if(id == null )
            {
                throw new ApiException(400, "missing required params");
            }
            // create path and map variables
            String path = "/subscriptions/{id}".replaceAll("\\{format\\}","json").replaceAll("\\{" + "id" + "\\}", id.toString());

            // query params
            Map<String, String> queryParams = new HashMap<String, String>();
            Map<String, String> headerParams = new HashMap<String, String>();
            try
            {
                return apiClient.resource(path)
                    .queryParams(buildmvm(queryParams))
                    .type(MediaType.APPLICATION_JSON).method("PUT", SubscriptionResponse.class, body);
            }
            catch (UniformInterfaceException ex)
            {
                if(ex.getResponse().getStatus() == 404)
                {
                    return null;
                }
                else if(ex.getResponse().getStatus() == 422)
                {
                    throw new ValidationException(ex.getResponse().getEntity(Response.class));
                }
                else
                {
                    throw new ApiException(ex.getResponse().getStatus(), ex.getResponse().toString());
                }
            }
	}
        
	/**
         * deleteSubscription.
         * Endpoint to delete a subscription.
         *
         * @throws ApiException
         */
	public SubscriptionResponse deleteSubscription (String id) throws ApiException
	{
            // verify required params are set
            if(id == null )
            {
                throw new ApiException(400, "missing required params");
            }
            // create path and map variables
            String path = "/subscriptions/{id}".replaceAll("\\{format\\}","json").replaceAll("\\{" + "id" + "\\}", id.toString());

            // query params
            Map<String, String> queryParams = new HashMap<String, String>();
            Map<String, String> headerParams = new HashMap<String, String>();
            try
            {
                return apiClient.resource(path)
                    .queryParams(buildmvm(queryParams))
                    .method("DELETE", SubscriptionResponse.class);
            }
            catch (UniformInterfaceException ex)
            {
                if(ex.getResponse().getStatus() == 404)
                {
                    return null;
                }
                else if(ex.getResponse().getStatus() == 422)
                {
                    throw new ValidationException(ex.getResponse().getEntity(Response.class));
                }
                else
                {
                    throw new ApiException(ex.getResponse().getStatus(), ex.getResponse().toString());
                }
            }
	}
        
	/**
         * createSubscription.
         * Endpoint to create a subscription.
         *
         * @throws ApiException
         */
	public SubscriptionResponse createSubscription (Subscription body) throws ApiException
	{
            // create path and map variables
            String path = "/subscriptions".replaceAll("\\{format\\}","json");

            // query params
            Map<String, String> queryParams = new HashMap<String, String>();
            Map<String, String> headerParams = new HashMap<String, String>();
            try
            {
                return apiClient.resource(path)
                    .queryParams(buildmvm(queryParams))
                    .type(MediaType.APPLICATION_JSON).method("POST", SubscriptionResponse.class, body);
            }
            catch (UniformInterfaceException ex)
            {
                if(ex.getResponse().getStatus() == 404)
                {
                    return null;
                }
                else if(ex.getResponse().getStatus() == 422)
                {
                    throw new ValidationException(ex.getResponse().getEntity(Response.class));
                }
                else
                {
                    throw new ApiException(ex.getResponse().getStatus(), ex.getResponse().toString());
                }
            }
	}
        
	/**
         * getSubscriptions.
         * Endpoint to list the subscriptions.
         *
         * @throws ApiException
         */
	public SubscriptionListResponse getSubscriptions (String workspaceId, String identifier) throws ApiException
	{
            // create path and map variables
            String path = "/subscriptions".replaceAll("\\{format\\}","json");

            // query params
            Map<String, String> queryParams = new HashMap<String, String>();
            if(!"null".equals(String.valueOf(workspaceId)))
            queryParams.put("workspaceId", String.valueOf(workspaceId));
            if(!"null".equals(String.valueOf(identifier)))
            queryParams.put("identifier", String.valueOf(identifier));
            Map<String, String> headerParams = new HashMap<String, String>();
            try
            {
                return apiClient.resource(path)
                    .queryParams(buildmvm(queryParams))
                    .method("GET", SubscriptionListResponse.class);
            }
            catch (UniformInterfaceException ex)
            {
                if(ex.getResponse().getStatus() == 404)
                {
                    return null;
                }
                else if(ex.getResponse().getStatus() == 422)
                {
                    throw new ValidationException(ex.getResponse().getEntity(Response.class));
                }
                else
                {
                    throw new ApiException(ex.getResponse().getStatus(), ex.getResponse().toString());
                }
            }
	}
        
	/**
         * getSubscriptionDeliveries.
         * Endpoint to list the deliveries
         *
         * @throws ApiException
         */
	public DeliveryListResponse getSubscriptionDeliveries (String id, Integer page, Integer size, String sort) throws ApiException
	{
            // verify required params are set
            if(id == null )
            {
                throw new ApiException(400, "missing required params");
            }
            // create path and map variables
            String path = "/subscriptions/{id}/deliveries".replaceAll("\\{format\\}","json").replaceAll("\\{" + "id" + "\\}", id.toString());

            // query params
            Map<String, String> queryParams = new HashMap<String, String>();
            if(!"null".equals(String.valueOf(page)))
            queryParams.put("page", String.valueOf(page));
            if(!"null".equals(String.valueOf(size)))
            queryParams.put("size", String.valueOf(size));
            if(!"null".equals(String.valueOf(sort)))
            queryParams.put("sort", String.valueOf(sort));
            Map<String, String> headerParams = new HashMap<String, String>();
            try
            {
                return apiClient.resource(path)
                    .queryParams(buildmvm(queryParams))
                    .method("GET", DeliveryListResponse.class);
            }
            catch (UniformInterfaceException ex)
            {
                if(ex.getResponse().getStatus() == 404)
                {
                    return null;
                }
                else if(ex.getResponse().getStatus() == 422)
                {
                    throw new ValidationException(ex.getResponse().getEntity(Response.class));
                }
                else
                {
                    throw new ApiException(ex.getResponse().getStatus(), ex.getResponse().toString());
                }
            }
	}
        
	}

