package net.leanix.webhooks.api;

import net.leanix.dropkit.apiclient.ApiException;
import net.leanix.webhooks.api.models.Event;
import net.leanix.webhooks.api.models.EventBatch;
import net.leanix.webhooks.api.models.EventListResponse;
import net.leanix.webhooks.api.models.EventResponse;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * The EventSubmitter wraps the EventApi class, to provide a retry logic for sending events
 */
public class EventSubmitter {

    private EventsApi eventsApi;
    private long backoffTime;
    private int maxRetries;
    private final static Logger LOG = LoggerFactory.getLogger(EventsApi.class);

    public EventSubmitter(EventsApi eventsApi, long backoffTime, int maxRetries) {
        this.eventsApi = eventsApi;
        this.backoffTime = backoffTime;
        this.maxRetries = maxRetries;
    }

    public EventsApi getEventsApi() {
        return eventsApi;
    }

    public void setEventsApi(EventsApi eventsApi) {
        this.eventsApi = eventsApi;
    }

    /**
     * Send an event to webhooks and retry if the post fails.
     * This is the default class for sending events.
     */
    public EventResponse submitEvent(Event body) throws ApiException {
        return submitEvent(body, 1);
    }

    private EventResponse submitEvent(Event body, int attempt) throws ApiException {
        try {
            return eventsApi.createEvent(body);
        } catch (ApiException e) {
            retry(attempt, e);
            return submitEvent(body, ++attempt);
        }
    }

    /**
     * Send an event batch to webhooks and retry if the post fails.
     * This is the default class for sending events.
     */
    public EventListResponse submitEventBatch(EventBatch batch) throws ApiException {
        return submitEventBatch(batch, 1);
    }

    private EventListResponse submitEventBatch(EventBatch batch, int attempt) throws ApiException {
        try {
            return eventsApi.createEventBatch(batch);
        } catch (ApiException e) {
            retry(attempt, e);
            return submitEventBatch(batch, ++attempt);
        }
    }

    private void retry(int attempt, ApiException e) throws ApiException {
        if (attempt == maxRetries) {
            throw e;
        }
        long backoff = (backoffTime * attempt);
        LOG.warn(String.format("Failed to send event to webhooks: %s\nRetrying in %d ms.", e, backoff));
        try {
            Thread.sleep(backoff);
        } catch (InterruptedException ignored) {
        }
    }
}
