package pragma.protoc.plugin.custom

import com.google.protobuf.DescriptorProtos
import com.google.protobuf.compiler.PluginProtos.CodeGeneratorRequest
import com.google.protobuf.compiler.PluginProtos.CodeGeneratorResponse

class CoreSdkGenerator : SdkGenerator() {
    override fun generate(
        target: String,
        request: CodeGeneratorRequest
    ): Iterable<CodeGeneratorResponse.File> {
        // This generator creates the type map necessary to map /v1/types info to native C++ types.
        //
        // Note: SDK Core C++ type definitions are currently generated via the normal protobuf flow. See "proto-cpp" in
        // 4-demo/demo-protos/pom.xml or equivalent pom.xml for your project.
        val filesWithApiTypes = filterSdkApiFiles(request)

        if (filesWithApiTypes.isEmpty()) return listOf()

        val templateContext = mapOf(
            "backendTypes" to filesWithApiTypes.flatMap { file ->
                file.messageTypeList.filter { isSdkApiType(it) }.map { message ->
                    BackendTypeTemplateContext(backendTypeName(file, message.name), nativeTypeName(file, message.name))
                }
            }.sortedBy { it.backendName },
            "dependencies" to filesWithApiTypes.mapNotNull { it.name.replace(".proto", "") }
        )

        return listOf(
            CodeGeneratorResponse.File.newBuilder()
                .setName("$target/generated_type_specializations.tpp")
                .setContent(compileTemplate("cpp/generated_type_specializations.tpp.mustache", templateContext))
                .build(),
            CodeGeneratorResponse.File.newBuilder()
                .setName("$target/rpc_type_traits.cpp")
                .setContent(compileTemplate("cpp/rpc_type_traits.cpp.mustache", templateContext))
                .build(),
        )
    }

    private fun nativeTypeName(file: DescriptorProtos.FileDescriptorProto, name: String): String {
        val nameSpace = file.`package`.replace(".", "::")
        return "${nameSpace}::${name}"
    }
}
