package net.roseboy.framework.core;

import com.jfinal.aop.Duang;
import com.jfinal.config.Routes;
import com.jfinal.core.Controller;
import com.jfinal.kit.StrKit;
import com.jfinal.plugin.activerecord.ActiveRecordPlugin;
import com.jfinal.plugin.activerecord.Model;
import com.jfinal.plugin.activerecord.Table;
import com.jfinal.plugin.activerecord.TableMapping;
import net.roseboy.framework.annotation.ControllerBind;
import net.roseboy.framework.annotation.Service;
import net.roseboy.framework.annotation.TableBind;
import net.roseboy.framework.util.ClassSearcher;
import net.roseboy.framework.util.ExceptionUtils;
import net.roseboy.framework.util.Log;

import java.lang.reflect.Field;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Jfinal工具类
 * 
 * @author roseboy
 *
 */
public class JFinalUtils {
	private static Map<String, String> MODELMAPPER = new HashMap<String, String>();// model名称-对应的class全名
	private static Map<String, Object> serviceMap = new HashMap<String, Object>();// services实例

	public static void addModel(String name, String cksName) {
		if (!MODELMAPPER.containsKey(name)) {
			MODELMAPPER.put(name, cksName);
		} else {
			ExceptionUtils.ThrowProjectException("Model名称:\"" + name + "\"已经存在");
		}
	}

	/**
	 * 根据model名称获取类的全名
	 * 
	 * @param name
	 * @return
	 */
	public static String getClassName(String name) {
		return MODELMAPPER.get(name);
	}

	/**
	 * 根据类名获取service实例
	 * 
	 * @param name
	 * @return
	 */
	public static Object getService(String name) {
		return serviceMap.get(name);
	}

	/**
	 * 根据类获取对应的名
	 * 
	 * @param class1
	 * @return
	 */
	@SuppressWarnings("rawtypes")
	public static Table getTable(Class<? extends Model> cls) {
		Table table = TableMapping.me().getTable(cls);
		if (table == null) {
			ExceptionUtils.ThrowProjectException("获取表失败");
		}
		return table;
	}

	/**
	 * 根据model名创建实例
	 * 
	 * @param name
	 * @return
	 * @throws ClassNotFoundException
	 * @throws InstantiationException
	 * @throws IllegalAccessException
	 */
	public static Model<?> getObjByModelName(String name) throws ClassNotFoundException, InstantiationException, IllegalAccessException {
		Class<?> clazz = Class.forName(JFinalUtils.getClassName(name));// 根据model类全名实例化
		Model<?> obj = (Model<?>) clazz.newInstance();
		return obj;
	}

	/**
	 * 自动绑定Controller
	 * 
	 * @param r
	 */
	public static void AutoBindController(Routes r) {
		List<Class<? extends Controller>> controllerClasses = ClassSearcher.of(Controller.class).search();
		ControllerBind cb = null;
		for (Class<? extends Controller> c : controllerClasses) {
			if (c.isAnnotationPresent(ControllerBind.class)) {
				cb = c.getAnnotation(ControllerBind.class);
				if ("".equals(cb.viewPath())) {
					r.add(cb.controllerKey(), c);
				} else {
					r.add(cb.controllerKey(), c, cb.viewPath());
				}
				Log.info("ControllerBinding: " + cb.controllerKey() + "==>" + c.getName());
			}
		}
	}

	/**
	 * 自动绑定model
	 * 
	 * @param arpDb
	 */
	public static void AutoBindTable(ActiveRecordPlugin arpDb) {
		List<Class<? extends Model<?>>> modelClasses = ClassSearcher.of(Model.class).search();
		TableBind tb = null;
		for (Class<? extends Model<?>> c : modelClasses) {
			if (c.isAnnotationPresent(TableBind.class)) {
				tb = c.getAnnotation(TableBind.class);
				if ("".equals(tb.primaryKey())) {
					arpDb.addMapping(tb.tabName(), c);
				} else {
					arpDb.addMapping(tb.tabName(), tb.primaryKey(), c);
				}
				// model名称与类名对应
				String modelName = tb.name();
				if ("".equals(modelName)) {
					modelName = c.getSimpleName();
				}
				modelName = modelName.toLowerCase();
				addModel(modelName, c.getName());
				Log.info("TableBinding: " + modelName + "==>" + c.getName() + "==>" + tb.tabName());
			}
		}
	}

	/**
	 * 自动绑定service，扫描所有service，并给service中的service注入
	 */
	public static void AutoBindService() {
		// 扫描所有的service，并把增强的实例放在serviceMap
		List<Class<? extends Object>> allClasses = ClassSearcher.of(Object.class).search();
		for (Class<? extends Object> c : allClasses) {
			if (c.isAnnotationPresent(Service.class)) {
				try {
					serviceMap.put(c.getName(), Duang.duang(c.newInstance().getClass()));
					Log.info("Service: " + c.getName());
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}

		// 给service的service注入service
		for (Map.Entry<String, Object> entry : serviceMap.entrySet()) {
			Field[] fields = entry.getValue().getClass().getSuperclass().getDeclaredFields();
			for (Field f : fields) {
				Object service = JFinalUtils.getService(f.getType().getName());
				if (service != null) {
					try {
						f.setAccessible(true);
						f.set(entry.getValue(), service);
					} catch (Exception e) {
						e.printStackTrace();
					}
				}
			}
		}

	}

	/**
	 * 自动注入service，给当前类的service自动注入，一般放在构造方法中
	 *
	 */
	public static void AutoBindService(Object obj) {
		Field[] fields = obj.getClass().getDeclaredFields();
		Field field;
		for (int i = 0; i < fields.length; i++) {
			field = fields[i];
			Object service = JFinalUtils.getService(field.getType().getName());
			if (service != null) {
				try {
					field.setAccessible(true);
					field.set(obj, service);
					//Log.info("ServiceBinding: " + obj.getClass().getSimpleName() + "." + field.getName() + "==>" + field.getType().getName());
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}
	}


	/**
	 * 转换成java的命名规范
	 * 
	 * @param n
	 * @return
	 */
	public static String toJavaName(String n, String q) {
		n = n.toLowerCase();
		String[] ss = n.split("_");
		n = "";
		for (int i = 0; i < ss.length; i++) {
			String s = ss[i];
			n += (s.substring(0, 1).toUpperCase()) + s.substring(1);
		}
		if (StrKit.isBlank(q)) {
			n = (n.substring(0, 1).toLowerCase()) + n.substring(1);
		} else {
			n = q + n;
		}
		return n;
	}

	/**
	 * 转换数据库命名规范
	 * 
	 * @param n
	 * @return
	 */
	public static String toDbName(String n) {
		StringBuffer sb = new StringBuffer();
		for (int i = 0; i < n.length(); i++) {
			if (Character.isUpperCase(n.charAt(i))) {
				sb.append("_").append(Character.toLowerCase(n.charAt(i)));
			} else {
				sb.append(n.charAt(i));
			}
		}
		return sb.toString();
	}
}
