package net.roseboy.framework.util;

import org.apache.poi.hssf.usermodel.*;
import org.apache.poi.ss.usermodel.*;

import java.io.*;
import java.math.BigDecimal;
import java.text.DecimalFormat;
import java.text.ParseException;
import java.util.*;

/**
 * excel生成工具
 * 
 * @author roseboy.net
 *
 */
public class ExcelUtils {

	public static File exportXls(String head, List<String> titles, List<String> fields, List<Map<String, Object>> records) {
		HSSFWorkbook workbook = new HSSFWorkbook();
		HSSFSheet sheet = workbook.createSheet();
		workbook.setSheetName(0, head);// 工作簿 标题

		HSSFRow row;
		HSSFCell cell;
		int index = 0;
		row = sheet.createRow(index);// 第一行，创建列名
		for (int i = 0; i < titles.size(); i++) {
			cell = row.createCell(i);
			cell.setCellValue(titles.get(i));
			cell.setCellStyle(getTitleStyle(workbook));
		}

		index++;
		for (Map<String, Object> record : records) {
			row = sheet.createRow(index);
			String field;
			String clsname;
			for (int i = 0; i < fields.size(); i++) {
				field = fields.get(i);// 字段名
				Object value = record.get(field);// 字段值
				value = (value == null ? "" : value);
				clsname = value.getClass().getSimpleName();// 字段类型名
				cell = row.createCell(i);
				if ("String".equals(clsname)) {
					cell.setCellValue(value.toString());
				} else if ("Double".equals(clsname)) {
					cell.setCellValue(Double.parseDouble(value.toString()));
				} else if ("Timestamp".equals(clsname)) {
					try {
						Date date = DateUtils.parseDate(value.toString(), "yyyy-MM-dd HH:mm:ss.0");
						cell.setCellValue(DateUtils.parseDate(date, "yyyy-MM-dd HH:mm:ss"));
					} catch (ParseException e) {
						cell.setCellValue(value.toString());
					}
				} else if ("Integer".equals(clsname)) {
					cell.setCellValue(Integer.parseInt(value.toString()));
				} else if ("Long".equals(clsname)) {
					cell.setCellValue(Long.parseLong(value.toString()));
				} else if ("Boolean".equals(clsname)) {
					cell.setCellValue(Boolean.parseBoolean(value.toString()));
				} else if ("BigDecimal".equals(clsname)) {
					cell.setCellValue(value.toString());
				} else {
					cell.setCellValue(value.toString());
				}
			}
			index++;
		}

		// =======================================

		String uploadPath = ExcelUtils.class.getResource("/").getPath() + "../../upload/xls_temp/";
		File uploadFile = new File(uploadPath);
		if (!uploadFile.exists()) {
			uploadFile.mkdirs();
		} else {
			String[] paths = uploadFile.list();
			if (paths != null && paths.length >= 30) {// 临时文件超过30个删除
				File file;
				for (String p : paths) {
					file = new File(uploadFile, p);
					file.delete();
				}
			}
		}
		String fname = uploadPath + head + "-" + DateUtils.parseDate(new Date(), "yyyyMMddhhmmss") + ".xls";

		FileOutputStream fOut;
		try {
			fOut = new FileOutputStream(fname);
			workbook.write(fOut);
			fOut.flush();
			fOut.close();
		} catch (IOException e) {
			e.printStackTrace();
		}
		File f = new File(fname);
		if (f.exists())
			return f;
		else
			return null;
	}

	private static HSSFCellStyle getTitleStyle(HSSFWorkbook workbook) {
		HSSFCellStyle cellStyle = workbook.createCellStyle();
		cellStyle.setAlignment(HSSFCellStyle.ALIGN_CENTER);
		cellStyle.setVerticalAlignment((short) 1);
		HSSFFont font = workbook.createFont();
		font.setBoldweight(HSSFFont.BOLDWEIGHT_BOLD);
		font.setFontHeightInPoints((short) 11);

		cellStyle.setFont(font);
		return cellStyle;
	}

	/**
	 * 读取excel的标题
	 * 
	 * @param xlsFile
	 * @return
	 */
	public static List<Map<String, Object>> readerExcel(File xlsFile) {
		return readerExcel(xlsFile, 0);
	}

	public static List<Map<String, Object>> readerExcel(File xlsFile, int line) {
		List<Map<String, Object>> list = new ArrayList<Map<String, Object>>();
		try {
			long columnNum = 0;// 列数
			InputStream inp = new FileInputStream(xlsFile);
			Workbook wb = new HSSFWorkbook(inp);
			Sheet sheet = wb.getSheetAt(0);
			if (sheet.getRow(0) != null) {
				columnNum = sheet.getRow(0).getLastCellNum() - sheet.getRow(0).getFirstCellNum();
			}
			List<String> titles = new ArrayList<String>();
			if (columnNum > 0) {
				for (Row row : sheet) {// 遍历每一行
					Map<String, Object> map = new LinkedHashMap<String, Object>();
					for (int i = 0; i < columnNum; i++) {
						Cell cell = row.getCell(i, Row.CREATE_NULL_AS_BLANK);
						if (row.getRowNum() == line) {// 第一行读取标题
							titles.add(cell.getStringCellValue());
						} else if (row.getRowNum() > line) {// 其他行是数据
							Object value = getCellValue(cell);
							map.put(titles.get(i), value);
						}
					}
					if (map.size() > 0) {
						list.add(map);
					}
				}
			}
		} catch (Exception e) {
			e.printStackTrace();
		}
		return list;
	}

	/**
	 * 获取单元格内的值
	 * 
	 * @param cell
	 * @return
	 */
	private static Object getCellValue(Cell cell) {
		switch (cell.getCellType()) {
		case Cell.CELL_TYPE_BLANK:
			return "";
		case Cell.CELL_TYPE_BOOLEAN:
			return cell.getBooleanCellValue();
		case Cell.CELL_TYPE_NUMERIC:
			if (DateUtil.isCellDateFormatted(cell)) {
				return cell.getDateCellValue();
			} else {
				DecimalFormat df = new DecimalFormat("0.00");
				String v = df.format(cell.getNumericCellValue());// 防止变成科学计数法
				return new BigDecimal(v);
			}
		case Cell.CELL_TYPE_STRING:
			return cell.getStringCellValue().trim();
		case Cell.CELL_TYPE_ERROR:
			return "ERROR";
		case Cell.CELL_TYPE_FORMULA:
		default:
		}
		return "";
	}

	public static void main(String[] args) {
		File file = new File("C:\\Users\\admin\\Desktop\\CEA得分20170519110024.xls");
		List<Map<String, Object>> list = readerExcel(file, 0);
		for (Map<String, Object> map : list) {
			System.out.println(map);
		}
	}
}
