package net.sf.gluebooster.demos.pojo.flashcards;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Random;

/**
 * Flashcards are 'cards' (with front side and backside) with data (CardElement). See https://en.wikipedia.org/wiki/Flashcard
 * Vocabulary-Example: for a multilingual vocabulary (english, german, french) each card has three data parts. Each part is the word in the corresponding language. Part 0 is the english word, Part 1 is the german translation, Part 2 is the french translation.  
 * 
 * The cards are grouped by the level of knowledge one has about the data.
 * Vocabulary-Example: Unknown words may have the level of knowledge 0, words that have been memorized for the first time may have level 1 and known words have level 2. 
 * 
 * One card may be the currently used card.
 * 
 * @param <CardElement> the class of the parts of the card.   
 * 
 */
public class Flashcards<CardElement> {

	/**
	 * used to generate random numbers to select the next card.
	 */
	private Random randomGenerator = new Random();

	/**
	 * The cards are a list, because they may be sorted according
	 * to the level of knowledge. One card is a List<?>, with the front and back
	 * entries.
	 * 
	 * <pre>
	 * Cards of all Levels = List< All cards of one level = List<
	 * OneCard=List<CardElement>>
	 * 
	 * Example: 
	 * Vocabulary (english german)    go = gehen
	 * 
	 * CardElement = String (either 'go' or 'gehen')
	 * 
	 * One Card = List<String>:  ["go", "gehen"]
	 * 
	 * Cards of one level = List<List<String>> 
	 * 
	 * cards = all levels = List<List<List<String>>>
	 * </pre>
	 */
	private List/*all cards of one level*/<List/* of card*/<List/*front side, back side*/<CardElement>>> cards;

	/**
	 * Mapping of the index of the card element to its name.
	 * Vocabulary-Example: 0 = English, 1 = German, 2 = French 
	 * 
	 */
	private Map<Integer, String> names;

	/**
	 * The indices of the innermost list, that indicate which card elements are to be seen on the front side.
	 * Vocabulary-Example: the front side may be the English word, so the indices are [0]. 
	 */
	private int[] frontSideIndices;

	/**
	 * The indices of the innermost list, that indicate which card elements are to be seen on the back side. 
	 * Vocabulary-Example: the back side may be the German and French translation, so the indices are [1,2]. 
	 */
	private int[] backSideIndices;

	/**
	 * The flashcard that is currently used.
	 */
	private List<CardElement> currentCard;

	public final List<List<List<CardElement>>> getCards() {
		return cards;
	}
	
	/**
	 * Are there a minimum of cards in this instance.
	 * 
	 * @param numberOfCardsNeededAtLeast
	 *            the minimum number of cards that are needed.
	 * @return true iff there are enough cards.
	 */
	public boolean areCardsSet(int numberOfCardsNeededAtLeast){
		int numberOfCards = 0;
		for (int level = 0; level < cards.size(); level++) {
			List<? extends List<CardElement>> cardsOfSameLevel = cards
					.get(level);
			numberOfCards += cardsOfSameLevel.size();
			if (numberOfCards >= numberOfCardsNeededAtLeast){
				return true;
			}
		}
		return false;
			
	}

	public final void setCards(List<List<List<CardElement>>> cardsWithLevel) {
		// duplicate cards to avoid problems with the list (remove not
		// implemented, etc.)
		cards = new ArrayList<List<List<CardElement>>>(cardsWithLevel.size());
		for (List<List<CardElement>> cardsOfOneLevel : cardsWithLevel) {
			ArrayList<List<CardElement>> list = new ArrayList<List<CardElement>>(
					cardsOfOneLevel);
			cards.add(list);
		}
		currentCard = null;
	}

	/**
	 * Sets the cards of the lowest level.
	 * @param cards
	 */
	public void setCardsAtLevel0(List<List<CardElement>> cards) {
		ArrayList<List<List<CardElement>>> cardsWithLevel = new ArrayList<List<List<CardElement>>>();
		cardsWithLevel.add(cards);
		setCards(cardsWithLevel);
	}

	public final Map<Integer, String> getNames() {
		return names;
	}

	public final void setNames(Map<Integer, String> names) {
		this.names = names;
	}

	public final int[] getFrontSideIndices() {
		return frontSideIndices;
	}

	public final void setFrontSideIndices(int[] frontSideIndices) {
		this.frontSideIndices = frontSideIndices;
	}

	public final int[] getBackSideIndices() {
		return backSideIndices;
	}

	public final void setBackSideIndices(int[] backSideIndices) {
		this.backSideIndices = backSideIndices;
	}

	/**
	 * The current card is moved to the next level (if known) or to the first
	 * level (if not known).
	 * 
	 * @param currentCardKnown
	 *            is the current card known
	 */
	private void adaptCurrentCardLevel(Boolean currentCardKnown){
		if (currentCardKnown != null && currentCard != null){
			//move the current card to its new place
			for (int level = 0; level < cards.size(); level++) {
				List<? extends List<CardElement>> cardsOfSameLevel = cards
						.get(level);
			    if (cardsOfSameLevel.contains(currentCard)){
			    	if (Boolean.FALSE.equals(currentCardKnown)){
			    		if (level > 0){
			    			//put the card into the first level
			    			cards.get(0).add(0, currentCard);
							cardsOfSameLevel.remove(currentCard);
							break;
						}
					} else { // Card known
						// put the card into the next level
						if (cards.size() == level + 1) {
							cards.add(new ArrayList());
			    		}

						cards.get(level + 1).add(currentCard);
						cardsOfSameLevel.remove(currentCard);
						break;
					}
			    }
			}
		}
		
	}
	
	/**
	 * Return the frontside string of the next card. The current card is moved
	 * to the next level (if known) or to the first level (if not known).
	 * 
	 * @param currentCardKnown
	 *            whether the current card is known (true), unknown (false) or
	 *            no specification (null)
	 * @return the next card to be displayed
	 */
	public List<CardElement> getNextCard(Boolean currentCardKnown) {
		
		adaptCurrentCardLevel(currentCardKnown);

		//you need at least one card more than the current card.
		int minimumOfCardsNeeded = currentCard == null? 1: 2;
		
		
		List<CardElement> oldCard = currentCard;
		currentCard = null;

		int levels = cards.size();
		
		if (! areCardsSet(minimumOfCardsNeeded)){
			throw new IllegalStateException("not enough cards available. Needed " + minimumOfCardsNeeded + " got " + cards);
		}

		while (currentCard == null) {
				// get random number
				long random = randomGenerator.nextInt((int) Math.pow(10, levels));
				if (random > 0) {
					//choose the level
					int level = levels - (int) Math.log10(random) - 1;
					List<List<CardElement>> cardsOfSameLevel = cards.get(level);
					if (cardsOfSameLevel.size() > 0) {
					// choose one card of the level, but only from the first 20 cards at level 0.
					int maxPosition = (level == 0) ? 20 : Integer.MAX_VALUE;
						int position = randomGenerator.nextInt(Math.min(
								maxPosition, cardsOfSameLevel.size()));
						currentCard = cardsOfSameLevel.get(position);
						if (currentCard.equals(oldCard)){
							currentCard = null;
						}
					}
				}
	
			}

		return currentCard;
	}

	public List<CardElement> getCurrentCard() {
		return currentCard;
	}

	/**
	 * Gets the backside of the current card as string.
	 * 
	 * @return the text of the backside of the current card.
	 */
	public String getCurrentBacksideString() {
		return getBackSideString(getCurrentCard());
	}

	/**
	 * Gets the text of the front side of the card
	 * 
	 * @param card
	 *            the card which front side is to be inspected
	 * @return the front side as string
	 */
	public String getFrontSideString(List<CardElement> card) {

		return getSide(card, frontSideIndices);

	}

	/**
	 * Gets the frontside of the current card as string.
	 * 
	 * @return the text of the frontside of the current card.
	 */
	public String getCurrentFrontsideString() {
		return getFrontSideString(getCurrentCard());
	}

	/**
	 * Gets the text of the back side of the card
	 * 
	 * @param card
	 *            the card which back side is to be inspected
	 * @return the back side as string
	 */
	public String getBackSideString(List<CardElement> card) {

		return getSide(card, backSideIndices);

	}

	/**
	 * Computes one side of the card. The side contains all card elements that
	 * are at the position of given indices.
	 * 
	 * @param card
	 *            the card which side is to be inspected
	 * @param indices
	 *            the indices of the side
	 * @return All wanted card elements as concatenated string.
	 */
	private String getSide(List<CardElement> card, int[] indices) {
		StringBuilder result = new StringBuilder();
		for (int index : indices) {
			result.append(card.get(index));
			result.append(" \n");
		}

		return result.toString().trim();

	}

	/**
	 * Computes the number of cards of the corresponding level.
	 * 
	 * @return the computed number
	 */
	public List<Integer> getSizeInfo() {
		ArrayList<Integer> result = new ArrayList<Integer>(cards.size());
		for (List level : cards) {
			result.add(level.size());
		}

		return result;
	}

}
