package net.sf.gluebooster.demos.pojo.languages.chinese;

import java.awt.BorderLayout;
import java.awt.Frame;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.FileReader;
import java.io.Reader;
import java.io.StringReader;
import java.io.StringWriter;
import java.io.Writer;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;

import javax.swing.JButton;
import javax.swing.JPanel;

import org.apache.commons.lang3.tuple.Pair;

import net.sf.gluebooster.java.booster.basic.gui.swing.SwingBoostUtils;
import net.sf.gluebooster.java.booster.basic.text.general.ToCharacterIterator;
import net.sf.gluebooster.java.booster.basic.text.general.ToWordIterator;
import net.sf.gluebooster.java.booster.essentials.container.IteratorWithIterable;
import net.sf.gluebooster.java.booster.essentials.meta.objects.DisplayConfiguration;
import net.sf.gluebooster.java.booster.essentials.meta.objects.ObjectHolder;
import net.sf.gluebooster.java.booster.essentials.utils.ContainerBoostUtils;
import net.sf.gluebooster.java.booster.essentials.utils.IoBoostUtils;
import net.sf.gluebooster.java.booster.essentials.utils.ThrowableBoostUtils;

/**
 * Helper for the german wikibook <a href="https://de.wikibooks.org/wiki/Vokabeltexte_Chinesisch">"Vokabeltexte Chinesisch"</a>.
 * 
 * @author cbauer
 *
 */
public class VokabeltexteChinesischHelper implements ActionListener {

	private Frame frame;
	private ChineseHelper helper;

	private VokabeltexteChinesischHelper() {
	}

	public VokabeltexteChinesischHelper(Reader hanDeDictReader, boolean useGui, boolean onlyOneCharacterWords) throws Exception {

		Collection<ChineseVocabularyEntry> dictionary = CeDictParser.parseDictionary(hanDeDictReader, onlyOneCharacterWords);
		helper = new ChineseHelper();
		helper.setDictionary(dictionary);

		if (useGui) {
			DisplayConfiguration config = new DisplayConfiguration("Vokabeltexte Chinesisch Helper");
			config.setDisposeFrameAfterClosing(true);
			JButton run = new JButton("add pinyin");
			run.addActionListener(this);
			JPanel panel = new JPanel();
			panel.setLayout(new BorderLayout());
			panel.add(run, BorderLayout.SOUTH);
			frame = SwingBoostUtils.createFrame(config, panel);
			frame.setVisible(true);
		}

	}

	public static void main(String[] ignored) throws Exception {
		new VokabeltexteChinesischHelper(new FileReader(SwingBoostUtils.chooseFile("handedict", true)), true, false);
	}

	@Override
	public void actionPerformed(ActionEvent e) {
		try {
			String text = IoBoostUtils.getClipboardString();
			StringWriter result = new StringWriter();
			modify(new StringReader(text), result);
			IoBoostUtils.setClipboard(result.toString());

		} catch (Exception ex) {
			throw ThrowableBoostUtils.toRuntimeException(ex);
		}

	}

	public void modify(Reader text, Writer result) throws Exception {

		ToWordIterator toWords = new ToWordIterator();
		toWords.setWordSeparators(toWords.getWordSeparators().replace(":", ""));
		// remove : because the modifier is looking for Vorlage:Chinesisch

		IteratorWithIterable[] iterators = IteratorWithIterable.link(new ToCharacterIterator(text), toWords, new Modifier(helper));
		Iterator<String> lastIterator = iterators[iterators.length - 1];
		while (lastIterator.hasNext()) {
			result.write(lastIterator.next());
		}
		result.close();

	}

	static private class Modifier extends IteratorWithIterable<String> {

		private static final int POSITION_SOMEWHERE = 0;
		private static final int POSITION_VORLAGE = 1;
		private static final int POSITION_BEFORE_CHINESE = 2;
		private static final int POSITION_AFTER_CHINESE = 3;
		private static final int POSITION_PINYIN = 4;

		private int position = POSITION_SOMEWHERE;
		private StringBuilder chineseCharacters = new StringBuilder();

		private ChineseHelper helper;

		private Modifier(ChineseHelper helper) {
			this.helper = helper;
		}

		@Override
		protected ObjectHolder computeNextObject() throws Exception {
			Iterator<String> words = getIterable();
			if (!words.hasNext()) {
				return null;
			} else {
				String word = words.next();

				switch (position) {
				case POSITION_SOMEWHERE:
					if ("Vorlage:Chinesisch".equals(word)) {
						position = POSITION_VORLAGE;
					}
					break;
				case POSITION_VORLAGE:
					if ("|".equals(word)) {
						position = POSITION_BEFORE_CHINESE;
					}
					break;
				case POSITION_BEFORE_CHINESE:
					if (!word.startsWith("}}")) {
						chineseCharacters.append(word);
					} else {
						if (word.contains("||")) {
							position = POSITION_PINYIN;
						} else {
							position = POSITION_AFTER_CHINESE;
						}
					}
					break;
				case POSITION_AFTER_CHINESE:
					if (word.contains("||")) {
						position = POSITION_PINYIN;
					}
					break;
				case POSITION_PINYIN:
					if (word.trim().isEmpty()) {
						// just continue
					} else if (word.contains("||")) {
						StringBuilder pinyin = new StringBuilder();
						for (char c : chineseCharacters.toString().toCharArray()) {
							ChineseVocabularyEntry entry = helper.getEntry("" + c);
							if (entry == null) {
								pinyin.append(c);
							} else {
								for (Pair<String, List<String>> pinyinTranslation : entry.getPinyinTranslations()) {
									pinyin.append(pinyinTranslation.getLeft()).append("/");
								}
								pinyin.setLength(pinyin.length() - 1); // remove last slash

							}
							pinyin.append(" ");
						}
						word = pinyin.toString() + " " + word;
						chineseCharacters.setLength(0);
						position = POSITION_SOMEWHERE;
					} else {
						// pinyin already present
						chineseCharacters.setLength(0);
						position = POSITION_SOMEWHERE;
					}
					break;
				}


				// modify if necessary
				return ObjectHolder.createObjectHolder(word);
			}
		}

	}

	public List<Pair<String, ChineseVocabularyEntry>> getSubdictionary(Collection<String> chineseCharacters) throws Exception {
		return helper.getSubdictionary(chineseCharacters);
	}

}
