package net.sf.gluebooster.demos.pojo.languages.sanskrit;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.apache.commons.lang3.tuple.ImmutablePair;
import org.apache.commons.lang3.tuple.Pair;

import net.sf.gluebooster.demos.pojo.languages.LanguageEntry;

public class Sutra {

	/**
	 * The sutra in devanagari
	 */
	private String devanagari;

	/**
	 * Variants of the devanagari
	 */
	private String devanagariVariants;

	/**
	 * transcript with sandhi
	 */
	private String withSandhi;

	/**
	 * Transcript of the variants
	 */
	private String withSandhiVariants;

	/**
	 * transcript without sandhi
	 */
	private String withoutSandhi;

	/**
	 * transcript of the variants without sandhi
	 */
	private String withoutSandhiVariants;

	/**
	 * word by word translation
	 */
	private String syntaxTranslationDE;

	/**
	 * better translation
	 */
	private String semanticTranslationDE;

	/**
	 * A list of the used words. The ordering is for the display. The first element of the pair is the word as in the dictionary, the second is the inflected
	 * word. The words are the unicode (??) representation.
	 */
	private List<Pair<String, String>> usedWordAndInflectionUnicode = new ArrayList<Pair<String, String>>();;

	/**
	 * Example: Sutra 1 (yogaḥ cittavṛtti-nirodhaḥ) defines yoga
	 */
	private List<String> definedTerms = new ArrayList();

	/**
	 * Example: Sutra 1 (yogaḥ cittavṛtti-nirodhaḥ) uses cittavṛtti and nirodha
	 */
	private List<String> usedTerms = new ArrayList();

	public Sutra(String devanagari, String withSandhi, String withoutSandhi) {
		this.devanagari = devanagari;
		this.withSandhi = withSandhi;
		this.withoutSandhi = withoutSandhi;
	}

	public Sutra(String devanagari, String devanagariVariants, String withSandhi, String withSandhiVariants, String withoutSandhi,
			String withoutSandhiVariants) {
		this.devanagari = devanagari;
		this.devanagariVariants = devanagariVariants;
		this.withSandhi = withSandhi;
		this.withSandhiVariants = withSandhiVariants;
		this.withoutSandhi = withoutSandhi;
		this.withoutSandhiVariants = withoutSandhiVariants;
	}


	public Sutra(String devanagari, String withSandhi, String withoutSandhi, String syntaxTranslationDE, String semanticTranslationDE, String commentContentDE,
			String commentStructureDE) {
		this(devanagari, withSandhi, withoutSandhi);
		this.syntaxTranslationDE = syntaxTranslationDE;
		this.semanticTranslationDE = semanticTranslationDE;
		this.commentContentDE = commentContentDE;
		this.commentStructureDE = commentStructureDE;

	}

	/**
	 * comment of the sutra
	 */
	private String commentContentDE;

	/**
	 * comments on the interdependencies with other sutras
	 */
	private String commentStructureDE;

	public String getDevanagari() {
		return devanagari;
	}

	public void setDevanagari(String devanagari) {
		this.devanagari = devanagari;
	}

	public String getWithSandhi() {
		return withSandhi;
	}

	public void setWithSandhi(String withSandhi) {
		this.withSandhi = withSandhi;
	}

	public String getWithoutSandhi() {
		return withoutSandhi;
	}

	public void setWithoutSandhi(String withoutSandhi) {
		this.withoutSandhi = withoutSandhi;
	}


	public String getSyntaxTranslationDE() {
		return syntaxTranslationDE;
	}

	public void setSyntaxTranslationDE(String syntaxTranslationDE) {
		this.syntaxTranslationDE = syntaxTranslationDE;
	}

	public String getSemanticTranslationDE() {
		return semanticTranslationDE;
	}

	public void setSemanticTranslationDE(String semanticTranslationDE) {
		this.semanticTranslationDE = semanticTranslationDE;
	}

	public String getCommentContentDE() {
		return commentContentDE;
	}

	public void setCommentContentDE(String commentContentDE) {
		this.commentContentDE = commentContentDE;
	}

	public String getCommentStructureDE() {
		return commentStructureDE;
	}

	public void setCommentStructureDE(String commentStructureDE) {
		this.commentStructureDE = commentStructureDE;
	}

	public List<Pair<String, String>> getUsedWordAndInflectionUnicode() {

		// sort the words according to their position in the sutra (the inflection of the word is in the sutra without sandhi)
		Collections.sort(usedWordAndInflectionUnicode, new Comparator<Pair<String, String>>() {

			@Override
			public int compare(Pair<String, String> wordInflection1, Pair<String, String> wordInflection2) {

				return Integer.compare(withoutSandhi.indexOf(wordInflection1.getRight()), withoutSandhi.indexOf(wordInflection2.getRight()));
			}

		});

		return usedWordAndInflectionUnicode;

	}

	public List<SanskritWord> getUsedWords(boolean includingSubwords, String errorHint) throws Exception {
		List<Pair<String, String>> wordInflections = getUsedWordAndInflectionUnicode();

		ArrayList<SanskritWord> result = new ArrayList<>(wordInflections.size());
		Set<String> alreadyUsed = new HashSet<String>();

		ArrayList<String> words = new ArrayList<String>(wordInflections.size());
		for (Pair<String, String> wordInflection : wordInflections) {
			words.add(wordInflection.getLeft());
		}

		Sanskrit.SINGLETON.getWordsForUnicode(words, errorHint, includingSubwords, result, alreadyUsed, true);

		for (int i = 0; i < result.size(); i++) {
			SanskritWord word = result.get(i);
			if (words.contains(word.getUnicode())) {
				// make it bold
				word = word.cloneMe();
				word.setEmphasized(true);
				result.set(i, word);
			}
		}

		return result;
	}

	public void setUsedWordAndInflectionUnicode(List<Pair<String, String>> usedWordAndInflectionUnicode) {
		this.usedWordAndInflectionUnicode = usedWordAndInflectionUnicode;
	}

	public void addInflectedWord(String word, String inflection) {
		if (usedWordAndInflectionUnicode == null) {
			usedWordAndInflectionUnicode = new ArrayList<Pair<String, String>>();
		}

		usedWordAndInflectionUnicode.add(new ImmutablePair<String, String>(word, inflection));
	}

	public void setUsedWordAndInflectionUnicode(String... usedWordAndInflectionUnicode) {
		ArrayList<Pair<String, String>> list = new ArrayList();
		String word = null;
		for (String part : usedWordAndInflectionUnicode) {
			if (part == null) {
				throw new IllegalStateException("part must not be null");
			}

			if (word == null) {
				word = part;
			} else {
				list.add(new ImmutablePair<String, String>(word, part));
				word = null;
			}

		}
		if (word != null) {
			throw new IllegalStateException("words do not end with inflected word");
		}
		setUsedWordAndInflectionUnicode(list);
	}

	public List<String> getDefinedTerms() {
		return definedTerms;
	}

	public void setDefinedTerms(List<String> definedTerms) {
		this.definedTerms = definedTerms;
	}

	public List<String> getUsedTerms() {
		return usedTerms;
	}

	public void setUsedTerms(List<String> usedTerms) {
		this.usedTerms = usedTerms;
	}

	public void addDefinedTerm(String term) {
		definedTerms.add(term);
	}

	public void addUsedTerm(String term) {
		usedTerms.add(term);
	}

	@Override
	public String toString() {

		return getWithSandhi();
	}
}
