package net.sf.gluebooster.demos.pojo.math;

import java.lang.reflect.Constructor;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.ResourceBundle;
import java.util.Set;

import javax.naming.Name;

import net.sf.gluebooster.demos.pojo.math.library.Basics;
import net.sf.gluebooster.demos.pojo.math.library.General;
import net.sf.gluebooster.demos.pojo.math.library.algebra.Algebra;
import net.sf.gluebooster.demos.pojo.math.library.algebra.Group;
import net.sf.gluebooster.demos.pojo.math.library.algebra.Identity;
import net.sf.gluebooster.demos.pojo.math.library.algebra.Inverse;
import net.sf.gluebooster.demos.pojo.math.library.algebra.Magma;
import net.sf.gluebooster.demos.pojo.math.library.algebra.Monoid;
import net.sf.gluebooster.demos.pojo.math.library.algebra.SemiGroup;
import net.sf.gluebooster.demos.pojo.math.library.categoryTheory.CategoryTheory;
import net.sf.gluebooster.demos.pojo.math.library.logic.Bool;
import net.sf.gluebooster.demos.pojo.math.library.logic.Logic;
import net.sf.gluebooster.demos.pojo.math.library.numberTheory.InductionExamples;
import net.sf.gluebooster.demos.pojo.math.library.numberTheory.NaturalNumbers;
import net.sf.gluebooster.demos.pojo.math.library.numberTheory.Numbers;
import net.sf.gluebooster.demos.pojo.math.library.probabilityTheory.DescriptiveStatistics;
import net.sf.gluebooster.demos.pojo.math.library.setTheory.ClassesSets;
import net.sf.gluebooster.demos.pojo.math.library.setTheory.EmptySet;
import net.sf.gluebooster.demos.pojo.math.library.setTheory.SetOperations;
import net.sf.gluebooster.demos.pojo.math.library.setTheory.Subset;
import net.sf.gluebooster.demos.pojo.math.library.setTheory.Tuples;
import net.sf.gluebooster.demos.pojo.math.library.setTheory.functions.Associative;
import net.sf.gluebooster.demos.pojo.math.library.setTheory.functions.Commutative;
import net.sf.gluebooster.demos.pojo.math.library.setTheory.functions.CompositionOfMappings;
import net.sf.gluebooster.demos.pojo.math.library.setTheory.functions.IdentityFunction;
import net.sf.gluebooster.demos.pojo.math.library.setTheory.functions.JectiveFunctions;
import net.sf.gluebooster.demos.pojo.math.library.setTheory.functions.Mappings;
import net.sf.gluebooster.demos.pojo.math.library.setTheory.operations.CartesianProduct;
import net.sf.gluebooster.demos.pojo.math.library.setTheory.operations.Difference;
import net.sf.gluebooster.demos.pojo.math.library.setTheory.operations.Intersection;
import net.sf.gluebooster.demos.pojo.math.library.setTheory.operations.Powerset;
import net.sf.gluebooster.demos.pojo.math.library.setTheory.operations.Union;
import net.sf.gluebooster.demos.pojo.math.library.setTheory.relations.Composition;
import net.sf.gluebooster.demos.pojo.math.library.setTheory.relations.InverseRelation;
import net.sf.gluebooster.demos.pojo.math.library.setTheory.relations.Relation;
import net.sf.gluebooster.demos.pojo.math.library.setTheory.relations.RelationBinary;
import net.sf.gluebooster.demos.pojo.math.library.topology.Topology;
import net.sf.gluebooster.demos.pojo.math.library.topology.TopologyBase;
import net.sf.gluebooster.demos.pojo.math.studies.StudyResourceBundle;
import net.sf.gluebooster.demos.pojo.math.studies.StudyUnit;
import net.sf.gluebooster.demos.pojo.math.studies.TermsDefinitionsBasics;
import net.sf.gluebooster.java.booster.basic.container.BoostedNode;
import net.sf.gluebooster.java.booster.basic.container.ResourceBundleFactory;
import net.sf.gluebooster.java.booster.basic.meta.DocumentationContext;
import net.sf.gluebooster.java.booster.basic.text.general.BookBoostUtils;
import net.sf.gluebooster.java.booster.essentials.eventsCommands.CallableByReflection;
import net.sf.gluebooster.java.booster.essentials.utils.Check;
import net.sf.gluebooster.java.booster.essentials.utils.TextBoostUtils;
import net.sf.gluebooster.java.booster.essentials.utils.ThrowableBoostUtils;

/**
 * Create math books and excercises.
 * 
 * @author cbauer
 *
 */
public class MathStudies {

	public static final String MATH = "mathematics";
	public static final String LITERATURE = "literature";
	public static final String CONTENT = "content";
	public static final String GLOSSARY = "glossary";
	public static final String INDEX = "index";

	private static final int TYPE_NORMAL = 0;
	private static final int TYPE_GLOSSARY = 1;
	private static final int TYPE_INDEX = 2;

	private static Constructor<StudyResourceBundle> studyResourceBundleConstructor() {
		try {
			return StudyResourceBundle.class.getConstructor(ResourceBundle.class);
		} catch (Exception ex) {
			throw ThrowableBoostUtils.toRuntimeException(ex);
		}

	}

	private static final CallableByReflection hierarchyResourceBundleFactory = new CallableByReflection("hierarchy resource bundle maker",
			studyResourceBundleConstructor());
	/**
	 * Translations of terms and definitions.
	 */
	public static final ResourceBundleFactory termsDefinitionsTranslator = new ResourceBundleFactory("termsDefinitionsTranslator",
			MathStudies.class.getPackage().getName() + ".studies.TermsDefinitions", hierarchyResourceBundleFactory);
	// private Map<Locale, ResourceBundle> termsDefinitionsTranslator = new HashMap();

	/**
	 * Translations of the description of terms and definitions.
	 */
	public static final ResourceBundleFactory termsDefinitionsTextTranslator = new ResourceBundleFactory("termsDefinitionsTextTranslator",
			MathStudies.class.getPackage().getName() + ".studies.TermsDefinitionsText", hierarchyResourceBundleFactory);
	// private Map<Locale, ResourceBundle> termsDefinitionsTextTranslator = new HashMap();


	/**
	 * explicitely set units
	 */
	private StudyUnit[] units;

	/**
	 * All used units
	 */
	private static StudyUnit[] allUnits;

	public static StudyUnit[] all() throws Exception {

		if (allUnits == null) {
			allUnits = new StudyUnit[] { Basics.createStudyUnit(), Numbers.createStudyUnit(), RelationBinary.createStudyUnit1(), Logic.createStudyUnit1(), //
					ClassesSets.createStudyUnit0(), Bool.createStudyUnit1(), Logic.createStudyUnit2(), //
					ClassesSets.createStudyUnit1(), //
					Subset.createStudyUnit1(), NaturalNumbers.createStudyUnit1(), InductionExamples.createStudyUnit1(), EmptySet.createStudyUnit1(), //
					Intersection.createStudyUnit1(), Union.createStudyUnit1(), Difference.createStudyUnit1(), SetOperations.createStudyUnit1(),
					/* Powerset.createStudyUnit1() , */ Tuples.createStudyUnit1(), CartesianProduct.createStudyUnit1(), IdentityFunction.createStudyUnit0(),
					Relation.createStudyUnit1(),
					RelationBinary.createStudyUnit2(),
					IdentityFunction.createStudyUnit1(), //
					Intersection.createStudyUnit2(), Union.createStudyUnit2(), Difference.createStudyUnit2(), //
					Mappings.createStudyUnit1(), Associative.createStudyUnit1(), Commutative.createStudyUnit1(), 
					SetOperations.createStudyUnit2(),
					General.createStudyUnit1(),
					Algebra.createStudyUnit1(), Magma.createStudyUnit1(), Identity.createStudyUnit1(), SemiGroup.createStudyUnit1(),
					CategoryTheory.createStudyUnit1(), Monoid.createStudyUnit1(), Powerset.createStudyUnit1(), Topology.createStudyUnit1(),
					TopologyBase.createStudyUnit1(), InverseRelation.createStudyUnit1(), Composition.createStudyUnit1(),
					DescriptiveStatistics.createStudyUnit1(), JectiveFunctions.createStudyUnit1(), Inverse.createStudyUnit1(), Group.createStudyUnit1(),
					CompositionOfMappings.createStudyUnit1() };
			Bool.addProofs(); // add the proofs in that way to prevent endless recursions when the proofs want to use all statements
		}

		return allUnits;

	}

	public static void checkIntegrity(StudyUnit... studyUnits) throws Exception {
		HashSet<Name> alreadyDefined = new HashSet<>();
		for (StudyUnit unit : studyUnits) {
			checkIntegrity(unit, alreadyDefined);
		}

	}

	private static void checkIntegrity(StudyUnit toBeTested, Set<Name> alreadyDefined) throws Exception {
		for (Statement statement : toBeTested.getUnimportantStatements()) {
			alreadyDefined.add(statement.getTypeIdentifier());
			if (statement.getNameOfInstance() != null) {
				alreadyDefined.add(statement.getNameOfInstance().getIdentifier());
			}

		}
		for (Statement statement : toBeTested.getStatements()) {
			alreadyDefined.add(statement.getTypeIdentifier());
			if (statement.getNameOfInstance() != null) {
				alreadyDefined.add(statement.getNameOfInstance().getIdentifier());
			}
			checkIntegrity("", alreadyDefined, new HashSet(), statement);
		}

	}

	private static void checkIntegrity(String origin, Set<Name> alreadyDefined, Set<Statement> alreadyTested, Statement... toBeTested) throws Exception {
		for (Statement statement : toBeTested) {
			if (statement != null && !alreadyTested.contains(statement)) {

				alreadyTested.add(statement);

				Name identifier = statement.getTypeIdentifier();
				if (!alreadyDefined.contains(identifier)) {
					throw new IllegalStateException("identifier not found " + identifier + " for " + origin);
				}

				checkIntegrity(alreadyDefined, alreadyTested, statement.getAllBe(), "be from " + identifier + " with history " + origin);
				checkIntegrity(alreadyDefined, alreadyTested, statement.getExamples(), "examples from " + identifier + " with history " + origin);
				checkIntegrity(alreadyDefined, alreadyTested, statement.getInformalMain(), "informal main from " + identifier + " with history " + origin);
				checkIntegrity(alreadyDefined, alreadyTested, statement.getMain(), "main from " + identifier + " with history " + origin);
				checkIntegrity(alreadyDefined, alreadyTested, statement.getProofs(), "proofs from " + identifier + " with history " + origin);
				checkIntegrity(alreadyDefined, alreadyTested, statement.getVariables(), "variables from " + identifier + " with history " + origin);
			}
		}

	}

	private static void checkIntegrity(Set<Name> alreadyDefined, Set<Statement> alreadyTested, Collection<Statement> toBeTested, String origin)
			throws Exception {
		if (toBeTested != null) {
			checkIntegrity(origin, alreadyDefined, alreadyTested, toBeTested.toArray(new Statement[] {}));
		}
	}

	public static List<Statement> allStatements() throws Exception {
		ArrayList<Statement> result = new ArrayList();
		for (StudyUnit unit : all()) {
			result.addAll(unit.getUnimportantStatements());
			result.addAll(unit.getStatements());
		}

		return result;
	}

	/**
	 * @return the root node of the book
	 * @throws Exception
	 */
	public BoostedNode createStudyBook(DocumentationContext context, List<Locale> additionalStatementNameTranslation) throws Exception {
		Check.notNull(context.getLocale(), "context.locale");

		BoostedNode book = BookBoostUtils.createBook(null, context, translateTermDefinition(context, MATH), null);
		BookBoostUtils.createDiv(book, translateTermDefinitionText(context, MATH));// Introduction

		BoostedNode literature = BookBoostUtils.createChapter(book, translateTermDefinition(context, LITERATURE));
		// TODO collect references

		BoostedNode index = BookBoostUtils.createChapter(book, translateTermDefinition(context, INDEX));
		for (StudyUnit studyUnit : units) {
			addChapter(index, context, studyUnit, additionalStatementNameTranslation, TYPE_INDEX);
		}

		BoostedNode glossary = BookBoostUtils.createChapter(book, translateTermDefinition(context, GLOSSARY));
		for (StudyUnit studyUnit : units) {
			addChapter(glossary, context, studyUnit, additionalStatementNameTranslation, TYPE_GLOSSARY);
		}

		BoostedNode content = BookBoostUtils.createChapter(book, translateTermDefinition(context, CONTENT));
		for (StudyUnit studyUnit : units) {
			addChapter(content, context, studyUnit, additionalStatementNameTranslation, TYPE_NORMAL);
		}

		return book;
	}

	private void addChapter(BoostedNode parent, DocumentationContext context, StudyUnit unit, List<Locale> additionalStatementNameTranslation, int type)
			throws Exception {
		if (!unit.getStatements().isEmpty()) {

			BoostedNode chapter = BookBoostUtils.createChapter(parent, getChapterHeading(context, unit)/* translateTermDefinition(context, unit.getName()) */);
			if (TYPE_INDEX == type) {
				return; // we need just the heading (yet)
			}

			Object introduction = translateTermDefinitionText(context, unit.getName());
			if (TYPE_NORMAL == type && introduction != null) {
				BookBoostUtils.createText(chapter, introduction);
			}
			for (Statement statement : unit.getStatements()) {
				addStatement(chapter, context, statement, additionalStatementNameTranslation, type);
			}

			if (TYPE_NORMAL == type) {
				// Just some space to separate the chapters
				BookBoostUtils.createDivWithRawContent(chapter, "<br/><br/><br/><br/><br/><br/><br/>");
			}
		}
	}

	private void addStatement(BoostedNode parent, DocumentationContext context, Statement statement, List<Locale> additionalStatementNameTranslation,
			int typeOfText)
			throws Exception {

		Check.notNull(statement, "statement");
		Object type = statement.getType();
		if (type != null) {
			if (termsDefinitionsTranslator.containsKey(context, type)) {
				type = translateTermDefinition(context, type);
			}
		}

		if (type == null) {
			type = "";
		} else {
			type = TextBoostUtils.capitalizeFirstCharacter(type.toString().trim()) + ": ";
		}

		if (TYPE_GLOSSARY == typeOfText) {
			type = "";
		}

		List<Object> chapterName = new ArrayList(
				Arrays.asList(type.toString(),
						TextBoostUtils.capitalizeFirstCharacter(translateTermDefinition(context, statement.getIdentifyingName()).toString())));

		Statement nameOfInstance = statement.getNameOfInstance();
		if (statement.isDefinition() && nameOfInstance != null) {
			chapterName.add(" ");
			chapterName.add(nameOfInstance);
		}

		BoostedNode statementChapter;

		if (TYPE_NORMAL == typeOfText) {
			statementChapter = BookBoostUtils.createChapter(parent, chapterName);
			statementChapter.getAttributes().setId(statement.getIdentifyingName());
		} else {// Glossary
			chapterName.add(": ");
			BookBoostUtils.createText(parent, chapterName);
			statementChapter = parent;
		}

		if (TYPE_NORMAL == typeOfText) {
			// translations of the name
			if (statement.isWorthwhileToTranslate()) {
				for (Locale locale : additionalStatementNameTranslation) {
					DocumentationContext subcontext = new DocumentationContext(locale);
					Object key = statement.getIdentifyingName();
					if (hasTermDefinitionTranslation(subcontext, key)) {
						BookBoostUtils.createDiv(statementChapter,
								translateTermDefinition(context, locale.getLanguage()) + ": " + translateTermDefinition(subcontext, key));
					}
				}
			}
		}

		// Precondition
		List<Statement> preconditions = statement.getBe();
		if (preconditions != null && !preconditions.isEmpty()) {
			Object preconditionText = translateTermDefinition(context, TermsDefinitionsBasics.PRECONDITION) + ": ";

			BoostedNode be = null;

			if (TYPE_NORMAL == typeOfText) {
				be = BookBoostUtils.createDiv(statementChapter, preconditionText);
			} else { // glossary
				BookBoostUtils.createText(statementChapter, preconditionText);
				be = statementChapter;
			}
			for (Statement pre : preconditions) {
				BookBoostUtils.createText(be, pre);
				if (pre.isDisplayIdentifyingNameInPrecondition()) {
					BookBoostUtils.createText(be, " " + translateTermDefinition(context, pre.getIdentifier()));
				}
				BookBoostUtils.createText(be, "; ");
			}
		}
		List<Statement> informalMain = null;
		List<Statement> main = null;

		// informal main
		if (statement.isWorthwileMain()) {

			if (TYPE_NORMAL == typeOfText) {
				BookBoostUtils.createText(statementChapter, type.toString());
			}

			informalMain = statement.getInformalMain();
			if (informalMain != null && !informalMain.isEmpty()) {
				for (Statement part : informalMain) {
					BookBoostUtils.createDiv(statementChapter, part);
				}
			}

			// main
			main = statement.getMain();
			if (main != null && !main.isEmpty()) {
				for (Statement part : main) {
					if (TYPE_NORMAL == typeOfText) {
						BookBoostUtils.createDiv(statementChapter, part);
					} else {
						// glossary
						BookBoostUtils.createText(statementChapter, part);
					}
				}
			}
		}

		// introduction text only in normal text or if both main and informal-main are empty
		if (TYPE_NORMAL == typeOfText || ((informalMain == null || informalMain.isEmpty()) && (main == null || main.isEmpty()))) {

			Object introduction = translateTermDefinitionText(context, statement.getIdentifyingName());
			if (introduction != null) {
				BookBoostUtils.createText(statementChapter, introduction);
			}
		}

		if (TYPE_NORMAL == typeOfText) {
			// proofs
			List<Statement> proofs = statement.getProofs();
			if (proofs != null && !proofs.isEmpty()) {
				BookBoostUtils.createDiv(statementChapter, translateTermDefinition(context, TermsDefinitionsBasics.PROOF) + ":");
				for (Statement part : proofs) {
					BookBoostUtils.createDiv(statementChapter, /* part */ Basics.multiline(part, Basics.PROOF_END));
				}
			}

			// examples
			List<Statement> examplesOnlyName = statement.getExamplesOnlyName();
			List<Statement> examples = statement.getExamples();
			boolean hasExamplesOnlyName = examplesOnlyName != null && !examplesOnlyName.isEmpty();
			boolean hasExamples = examples != null && !examples.isEmpty();
			if (hasExamplesOnlyName || hasExamples) {
				BoostedNode subChapter = BookBoostUtils.createChapter(statementChapter,
						translateTermDefinition(context, TermsDefinitionsBasics.EXAMPLES) + ":");
				if (hasExamplesOnlyName) {
					for (Statement part : examplesOnlyName) {
						BookBoostUtils.createDiv(subChapter, translateTermDefinition(context, part.getIdentifyingName()));// TODO make list instead of div
					}
				}

				if (hasExamples) {
					for (Statement part : examples) {
						BookBoostUtils.createDiv(subChapter, part);// TODO make list instead of div
					}
				}
			}

			// references
			List<Statement> references = statement.getReferences();
			if (references != null && !references.isEmpty()) {
				BoostedNode subChapter = BookBoostUtils.createChapter(statementChapter, translateTermDefinition(context, LITERATURE));
				for (Statement reference : references) {
					BookBoostUtils.createDiv(subChapter, reference);
				}
			}
		}

		if (TYPE_GLOSSARY == typeOfText) {
			BookBoostUtils.createDiv(statementChapter);// linebreak
		}

	}



	private boolean hasTermDefinitionTranslation(DocumentationContext context, Object toTranslate) throws Exception {
		return termsDefinitionsTranslator.containsKey(context, toTranslate);
	}


	public static Object translateTermDefinition(DocumentationContext context, Object toTranslate) throws Exception {
		return termsDefinitionsTranslator.getObject(context, toTranslate, true);

	}

	private Object translateTermDefinitionText(DocumentationContext context, Object toTranslate) throws Exception {
		return termsDefinitionsTextTranslator.getObject(context, toTranslate, false);

	}

	public StudyUnit[] getUnits() {
		return units;
	}

	public void setUnits(StudyUnit[] units) {
		this.units = units;
	}

	private String getChapterHeading(DocumentationContext context, StudyUnit unit) throws Exception {
		String result = "" + translateTermDefinition(context, unit.getName());
		Statements origin = unit.getOrigin();// category of the unit
		if (origin != null) {
			origin = origin.getParentCategory();
		}
		while (origin != null) {
			result = "" + translateTermDefinition(context, origin.getCategory()) + ": " + result;
			origin = origin.getParentCategory();
		}
		return result;
	}

}
