package net.sf.gluebooster.demos.pojo.math;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang3.tuple.ImmutablePair;
import org.apache.commons.lang3.tuple.Pair;

import net.sf.gluebooster.demos.pojo.math.library.Basics;
import net.sf.gluebooster.demos.pojo.math.library.logic.Bool;
import net.sf.gluebooster.demos.pojo.math.library.logic.Logic;
import net.sf.gluebooster.demos.pojo.math.library.setTheory.ClassesSets;
import net.sf.gluebooster.demos.pojo.math.library.setTheory.EmptySet;
import net.sf.gluebooster.demos.pojo.math.library.setTheory.relations.Relation;
import net.sf.gluebooster.demos.pojo.math.library.setTheory.relations.RelationSpecial;
import net.sf.gluebooster.demos.pojo.prolog.AbstractProlog;
import net.sf.gluebooster.demos.pojo.prolog.Prolog;
import net.sf.gluebooster.java.booster.basic.container.Tuple;
import net.sf.gluebooster.java.booster.essentials.objects.BoostedObject;
import net.sf.gluebooster.java.booster.essentials.utils.TextBoostUtils;

/**
 * Generate proofs using prolog
 * 
 * @author cbauer
 *
 */
public class PrologProofGenerator extends BoostedObject {

	/**
	 * Counter to create names for unnamed variables
	 */
	private int unnamedVarCounter = 0;

	public String getFunctor(Statement statement) {
		return statement.getIdentifyingName().toString();
	}

	public Prolog createPrologWithStatements(Collection<Statement> statements) throws Exception {


		Prolog prolog = AbstractProlog.getDefaultProlog();

		HashMap<Statement, Object> prologAtomsMapping = new HashMap<>();

		Collection<Statement> addedStatements;

		do { // the ordering of the statements may not be as needed for the import into prolog. So do multiple runs
			ArrayList<Statement> toAdd = new ArrayList<>();
			addedStatements = new ArrayList<>();
			for (Statement statement : statements) {
				try {
					addStatement(prolog, statement, prologAtomsMapping);
					addedStatements.add(statement);
				} catch (Exception ex) {
					toAdd.add(statement);
				}
			}
			statements = toAdd;
		} while ((!statements.isEmpty()) && !addedStatements.isEmpty());

		if (!statements.isEmpty()) {
			// debugging:
			for (Statement statement : statements) {
				try {
					addStatement(prolog, statement, prologAtomsMapping);
				} catch (Exception ex) {
					getLog().warn("could not add ", statement, ex);
				}
			}

			throw new IllegalStateException("could not add " + statements);
		}

		System.out.println(prolog.getInitContent());

		prolog.initEnd();

		return prolog;

	}

	/**
	 * Tries to add a statement
	 * 
	 * @param prolog
	 * @param statement
	 * @param prologAtomsMapping
	 * @throws Exception
	 *             for example if depending statements are not yet present
	 */
	private void addStatement(Prolog prolog, Statement statement, HashMap<Statement, Object> prologAtomsMapping) throws Exception {
		List<Statement> ignoreStatements = Arrays.asList(/* Classes.ELEMENT_OF */);

		boolean skipStatement = false;
		for (Statement ignore : ignoreStatements) {
			if (ignore.is(statement)) {
				skipStatement = true;
				break;
			}
		}
		if (skipStatement) {
			return; // next statement
		}

		List<Statement> variables = statement.getVariables();
		List<Statement> main = statement.getMain();
		List<Statement> be = statement.getAllBe();

		String name = getFunctor(statement);

		if (main == null || main.isEmpty()) {
			if (Logic.EQUALS.is(statement)) {
				// binary equals
				// Object varX = prolog.var();
				// Object varY = prolog.var();
				// prolog.addRule(prolog.compound(name, varX, varY), prolog.equals(varX, varY));

				// lists with equal elements
				Object head = prolog.var("H");
				Object head2 = prolog.var("H2");
				Object tail = prolog.var("T");

				prolog.addFact(prolog.compound(name, prolog.list(head))); // list with one element is ok
				prolog.addRule(prolog.compound(name, prolog.listWith(head, prolog.listWith(head, tail))), prolog.compound(name, prolog.listWith(head, tail))); // [H,
																																								// [H|T]]
																																								// wenn
																																								// [H|T]
																																								// gleiche
																																								// Elemente
																																								// hat

			} else if (variables.isEmpty()) {
				Object rawData = statement.getRawData();
				if (rawData == null) {
					// example verum, falsum
					Object fact = prolog.addFact(prolog.atom(name));
					prologAtomsMapping.put(statement, fact);
				} else {
					// example explicit relation (NOT_FUNCTION)
					if (RelationSpecial.EXPLICIT_RELATION.is(statement)) {
						for (Object row : (Object[]) rawData) {
							Object[] rowData = (Object[]) row;
							Object[] mappedData = new Object[rowData.length];
							for (int i = 0; i < mappedData.length; i++) {
								mappedData[i] = prologAtomsMapping.get(rowData[i]);
								if (mappedData[i] == null) {
									throw new IllegalStateException("no mapping found for" + rowData[i]);
								}
							}
							prolog.addFact(prolog.compound(name, mappedData));
						}
					} else {
						throw new IllegalStateException("raw data not yet implemented");
					}
				}

			} else { // variables present
				if (ClassesSets.EXPLICIT_SET.is(statement)) {
					// the variables are the elements

					for (Statement variable : variables) {
						Object atom = prolog.atom(getFunctor(variable));
						// Maybe it must not be an atom but be a compound
						prolog.addFact(prolog.compound(name, atom));
					}

				} else if (ClassesSets.ELEMENT_OF.is(statement)) {
					Statement element = variables.get(0);
					Object elem;
					if (Logic.VARIABLE.is(element)) {
						elem = prolog.var(element.getIdentifier().toString());
					} else {
						throw new IllegalStateException("other args not yet implemented ");
					}
					String clasz = getFunctor(variables.get(1));
					// Maybe it must not be an atom but be a compound
					// prolog.addFact(prolog.compound(clasz, elem));
					getLog().warn("'element of' not yet correctly implemented");

				} else {
					getLog().warn("Ignoring a statement: statment not yet processed: " + statement);
					// let's hope it is not needed
					// throw new IllegalStateException("statment not yet processed: " + statement);
				}
				// make the statement to a fact
			}

		} else {
			addRule(prolog, statement);
			// HashSet<Statement> statementsAlreadyInProlog = new HashSet<>();
			// HashMap<Statement, Object> prologObjects = new HashMap<>();
			//
			// putStatement(prolog, statementsAlreadyInProlog, prologObjects, statement);
			// throw new IllegalStateException("statment not yet processed: " + statement);

		}

	}


	private void addRule(Prolog prolog, Statement statement) throws Exception {
		// Example 2:
		// FUNCTION_VALUE_AT = functionValueAt(f, x);
		// FUNCTION_VALUE_AT.be(mapping(f, A, F, B), Classes.isElementOf(x, A));
		// FUNCTION_VALUE_AT.main(Boolean.biconditional(Logic.definedAs(y, functionValueAt(f, x)), Classes.isElementOf(Tuples.tuple(x, y), F)));

		String statementFunctor = getFunctor(statement);
		List<Statement> variables = statement.getVariables();
		List<Statement> main = statement.getMain();
		List<Statement> be = statement.getAllBe();

		if (main.size() == 1 && Basics.MATH_TABLE.is(main.get(0))) {
			// Example Boolean@not@naive with main basics@mathTable@naive
			Statement mathTable = main.get(0);

			int varCount = variables.size();
			Statement function = mathTable.getMain().get(0);
			String tableFunctor = getFunctor(function);
			Object[] vars = new Object[varCount + 1];
			for (int i = 0; i <= varCount; i++) {
				vars[i] = prolog.var();
			}
			Object rule = prolog.addRule(prolog.compound(statementFunctor, vars), prolog.compound(tableFunctor, vars));
			System.out.println("added rule");

		} else {
			System.out.println("add rule not yet implemented fully");
			// throw new IllegalStateException("add rule not yet implemented");
		}
	}

	/**
	 * Creates a prolog term of a statement. Substatements (be, main) are not considered). Precondition is that all variables are alredy in the prologObjects
	 * map.
	 * 
	 * @param prolog
	 * @param prologObjects
	 * @param statement
	 * @throws Exception
	 */
	@Deprecated
	private void putStatementWithoutSubstatements(Prolog prolog, Collection<Statement> statementsAlreadyInProlog, Map<Statement, Object> prologObjects,
			Statement statement) throws Exception {

		if (!prologObjects.containsKey(statement)) {
			if (Logic.VARIABLE.is(statement)) {
				prologObjects.put(statement, prolog.var());
			} else { // create a compound
				List<Statement> variables = statement.getVariables();
				int varSize = variables.size();
				Object[] terms = new Object[varSize];
				boolean allTermsFound = true;
				for (int i = 0; i < varSize; i++) {
					Statement variable = variables.get(i);
					terms[i] = prologObjects.get(variable);
					// terms[i] may be null
					if (terms[i] == null) {
						allTermsFound = false;
					}
				}

				if (ClassesSets.ELEMENT_OF.is(statement)) {
					// Example Statement classes@isElementOf@naive with variables [Statement logic@variable@naive@A, Statement setTheory@explicitSet@naive
					// Boolean@boolean set@naive]
					// logic@variable@naive@A is mapped to Var1

					prologObjects.put(statement, prolog.compound(getFunctor(variables.get(1)), terms[0]));
					// Example Boolean@boolean set@naive[Var1]

				} else if (allTermsFound) {
					// example Statement Boolean@not@naive Variable [Statement logic@variable@naive@A]
					prologObjects.put(statement, prolog.compound(getFunctor(statement), terms));
					// throw new IllegalStateException("not yet correct");
				} else {
					throw new IllegalStateException("not yet implemented");
				}

			}
		}

	}


	/**
	 * 
	 * @param prolog
	 * @param prologVariables
	 *            the variables corresponding to the value of the statements
	 * @param statement
	 * @return the prolog object/variable corresponding to the result of the statement
	 * @throws Exception
	 */
	private Object putStatement(Prolog prolog, Map<Statement, Object> prologVariables, Collection<Object> compounds, Statement statement) throws Exception {

		if (prologVariables.containsKey(statement)) {
			return prologVariables.get(statement); // nothing to do
		}

		List<Statement> variables = statement.getVariables();

		Object theResultVariable = null;//

		Statement compoundFunctor = null;
		Object[] terms = null;

		if (Logic.VARIABLE.is(statement)) {
			if (prologVariables.containsKey(statement)) {
				return prologVariables.get(statement);
			} else {
				theResultVariable = prolog.var(); // new variable
			}

		} else if (Logic.BRACKET.is(statement) && variables.size() == 1) {
			// ignore the bracket because it has no semantic meaning
			return putStatement(prolog, prologVariables, compounds, variables.get(0));
		} else if (ClassesSets.ELEMENT_OF.is(statement)) {
			compoundFunctor = variables.get(1);// the class/set
			terms = new Object[] { putStatement(prolog, prologVariables, compounds, variables.get(0)) }; // the element
			// getLog().info("'element of' not yet correctly implemented");
			// StringBuilder text = new StringBuilder();
			// prolog.humanReadable(compounds, text);
			// getLog().info(text);
		} else { // function with the result as last variable
			int varSize = variables.size();
			compoundFunctor = statement;
			terms = new Object[varSize];

			for (int i = 0; i < varSize; i++) {
				Statement variable = variables.get(i);
				if (variable == null) {
					// create variable (for example for the result of the function) that has not been needed before
					theResultVariable = prolog.var(); // new variable
					terms[i] = theResultVariable;

				} else {
					terms[i] = putStatement(prolog, prologVariables, compounds, variable);
				}
			}

			// // create a compound
			// Object compound = prolog.compound(getFunctor(statement), terms);
			// compounds.add(compound);
			// if (varSize == 0) {
			// // return compound; // ???
			// throw new IllegalStateException("not handled yet");
			// } else {
			// Object lastVariable = terms[varSize - 1];
			// prologVariables.put(statement, lastVariable);
			// return lastVariable;
			// }


		}

		// create a compound if defined
		if (compoundFunctor != null) {
			Object compound = prolog.compound(getFunctor(compoundFunctor), terms);
			compounds.add(compound);
		}

		if (theResultVariable == null) {
			// throw new IllegalStateException("not handled yet");
			return null;
		} else {
			prologVariables.put(statement, theResultVariable);
			return theResultVariable;
		}

	}

	public Tuple<Prolog, Object, Object[], Map<Statement, Object>> createPrologAndQuery(Collection<Statement> availableStatements, Statement[] conditions,
			Statement... displayed) throws Exception {
		Prolog prolog = createPrologWithStatements(availableStatements);
		StringBuilder text = new StringBuilder();
		prolog.humanReadable(prolog.getAddedObjects(), text);

		getLog().info(text);
		// add functions and solve
		Map<Statement, Object> prologObjects = new HashMap();
		ArrayList<Object> queryStatements = new ArrayList<>();

		for (Statement st : conditions) {
			putStatement(prolog, prologObjects, queryStatements, st);
		}
		for (Statement st : displayed) {
			putStatement(prolog, prologObjects, queryStatements, st);
		}

		Object[] interesting = new Object[displayed.length];
		for (int i = 0; i < interesting.length; i++) {
			interesting[i] = prologObjects.get(displayed[i]);
			if (interesting[i] == null) {
				throw new IllegalStateException("statement not in map: " + displayed[i] + " " + prologObjects);
			}
		}
		Object query = prolog.conjunction(queryStatements.toArray());

		return new Tuple<Prolog, Object, Object[], Map<Statement, Object>>(prolog, query, interesting, prologObjects);
	}

	public Statement displayValueTable(Collection<Statement> availableStatements, Statement[] conditions, Statement... displayed) throws Exception {

		Tuple<Prolog, Object, Object[], Map<Statement, Object>> init = createPrologAndQuery(availableStatements, conditions, displayed);
		Prolog prolog = init.getFirst();
		Object query = init.getSecond();
		Object[] interesting = init.getThird();
		Map<Statement, Object> prologObjects = init.getFourth();

		Collection<Map<String, Object>> allSolutions = prolog.getAllSolutions(query, interesting);
		if (allSolutions.size() == 0) {
			throw new IllegalStateException("no solutions found");
		}
		Object[] table = new Object[allSolutions.size() + 1];
		Object[] header = new Object[displayed.length];
		System.arraycopy(displayed, 0, header, 0, displayed.length);
		table[0] = header;
		int i = 0;
		Iterator<Map<String, Object>> solutionIterator = allSolutions.iterator();
		while (solutionIterator.hasNext()) {
			i++;
			Map<String, Object> rowValues = solutionIterator.next();

			Object[] row = new Object[displayed.length];
			for (int j = 0; j < displayed.length; j++) {
				Object prologValue = rowValues.get(prolog.getVarname(prologObjects.get(displayed[j])));
				row[j] = transformPrologToStatement(prolog, prologValue);
			}
			table[i] = row;

		}

		return Basics.table(table);

	}

	/**
	 * 
	 * @param prologValue
	 *            for example the struct ':-'('Boolean@falsum@naive',true)
	 * @return
	 */
	private Statement transformPrologToStatement(Prolog prolog, Object prologValue) throws Exception {
		Object[] predicateArgs = prolog.unCompound(prologValue);

		if (predicateArgs.length == 1) { // only the name like "Boolean@falsum@naive" ("true" or something similar should not occur)
			return new Statement((String) predicateArgs[0]);

		} else if (":-".equals(predicateArgs[0])) {

			if ("true".equals(prolog.unCompound(predicateArgs[2])[0])) {
				// a factum
				return transformPrologToStatement(prolog, predicateArgs[1]);
			} else {
				throw new IllegalStateException("non facts not supported yet");
			}

		} else {
			throw new IllegalStateException("predicate " + predicateArgs[0] + " not yet supported");
		}

	}

	private void createProlog(Statement statement, StringBuilder result) throws Exception {

		StatementProlog prolog = new StatementProlog(statement);

		if (prolog.isConvertable()) {
			result.append(prolog.toProlog(true));
		}


	}

	/**
	 * Creates Prolog for the data and the query.
	 * 
	 * Not finished yet. Too complex
	 * 
	 * @param availableStatements
	 * @param toBeProven
	 * @return
	 * @throws Exception
	 */
	public Pair<String, String> createPrologForProof(Collection<Statement> availableStatements, Statement toBeProven) throws Exception {

		StringBuilder data = new StringBuilder("\n");

		data.append("% Knowledge base \n");
		for (Statement statement : availableStatements) {
			if (Bool.BICONDITIONAL_FROM_OTHERS.is(statement)) {
				System.out.println("breakpoint test");
			}

			createProlog(statement, data);
		}

		data.append("\n\n% to be proven: be \n");
		for (Statement statement : toBeProven.getAllBe()) {
			createProlog(statement, data);
		}

		data.append("\n");
		StringBuilder query = new StringBuilder();
		for (Statement statement : toBeProven.getMain()) {
			createProlog(statement, query);
		}

		return new ImmutablePair<String, String>(data.toString(), query.toString());

		// Prolog prolog = createPrologWithStatementsAndProof(availableStatements);

		// // add functions and solve
		// Map<Statement, Object> prologObjects = new HashMap();
		// ArrayList<Object> queryStatements = new ArrayList<>();
		//
		// for (Statement st : conditions) {
		// putStatement(prolog, prologObjects, queryStatements, st);
		// }
		// for (Statement st : displayed) {
		// putStatement(prolog, prologObjects, queryStatements, st);
		// }
		//
		// Object[] interesting = new Object[displayed.length];
		// for (int i = 0; i < interesting.length; i++) {
		// interesting[i] = prologObjects.get(displayed[i]);
		// if (interesting[i] == null) {
		// throw new IllegalStateException("statement not in map: " + displayed[i] + " " + prologObjects);
		// }
		// }
		// Object query = prolog.conjunction(queryStatements.toArray());
		// Collection<Map<String, Object>> allSolutions = prolog.getAllSolutions(query, interesting);
		// if (allSolutions.size() == 0) {
		// throw new IllegalStateException("no solutions found");
		// }
		// Object[] table = new Object[allSolutions.size() + 1];
		// Object[] header = new Object[displayed.length];
		// System.arraycopy(displayed, 0, header, 0, displayed.length);
		// table[0] = header;
		// int i = 0;
		// Iterator<Map<String, Object>> solutionIterator = allSolutions.iterator();
		// while (solutionIterator.hasNext()) {
		// i++;
		// Map<String, Object> rowValues = solutionIterator.next();
		//
		// Object[] row = new Object[displayed.length];
		// for (int j = 0; j < displayed.length; j++) {
		// Object prologValue = rowValues.get(prolog.getVarname(prologObjects.get(displayed[j])));
		// row[j] = transformPrologToStatement(prolog, prologValue);
		// }
		// table[i] = row;
		//
		// }
		//
		// return Basics.table(table);

	}


}
