package net.sf.gluebooster.demos.pojo.math;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Properties;

import javax.naming.CompoundName;
import javax.naming.InvalidNameException;
import javax.naming.Name;
import javax.swing.JOptionPane;

import net.sf.gluebooster.demos.pojo.math.studies.WriteExtended;
import net.sf.gluebooster.java.booster.essentials.meta.BoostedCloneable;
import net.sf.gluebooster.java.booster.essentials.meta.objects.TextDescription;
import net.sf.gluebooster.java.booster.essentials.utils.Check;
import net.sf.gluebooster.java.booster.essentials.utils.ThrowableBoostUtils;

/**
 * Mathematical statements. In this context statements can be used as predicates but also as axioms, postulates, lemma, theorems, etc.
 * 
 * @author cbauer
 *
 */
public class Statement implements BoostedCloneable {

	public static final String TYPE_DEFINITION = "definition";
	public static final String TYPE_LEMMA = "lemma";
	public static final String TYPE_THEOREM = "theorem";
	public static final String TYPE_EXAMPLE = "example";

	/**
	 * Separator of the parts of the identifier
	 */
	public static final String INDENTIFIER_SEPARATOR = "@";
	/**
	 * The first part is the category of the statement Example: "linear algebra", "category theory", "analysis", "group theory"
	 * 
	 * The second part is the name of the statement: Example: a mathematical group
	 * 
	 * The third part may be used to make distinctions: Example: if there are several definitions of a group the names may differ only in this part.
	 * 
	 * More parts may have a special meaning. They are are optional in comparisions.
	 */
	private CompoundName identifier;

	/**
	 * Informal description of the statement.
	 */
	private String description;

	/**
	 * The name of an instance of the category defined by this statement. Only usable if this statment is a definition of a set/category (example a group)
	 */
	private Statement nameOfInstance;

	/**
	 * Variables used in the statement. Example: a group consists of a set of elements G and an operation O. Variables are G and O It is a list because the
	 * ordering is important for computations.
	 */
	private List<Statement> variables;
	
	/**
	 * Additional infos about the variables.
	 * May be used for explaining something.
	 * Example Statement equals with variables (4, 3+1, 1+3)  and variableInfos  (null, 4 is defined as successor of 3, addition is commutative)
	 */
	private List<Statement> variableInfos;

	/**
	 * Simple conditions that must be fulfilled by the variables. Example: G must be set, O must bei Operation It is a list because the ordering is important
	 * when displaying the statement. More general Statements should be first.
	 */
	private List<Statement> be;

	/**
	 * Other conditions or definitions that are technically necessary, but not important/to be displayed.
	 * 
	 */
	private List<Statement> unimportant;

	/**
	 * Main conditions that must be fulfilled by the variables. Example: the conditions that the operation O must fulfill It is a list because the ordering is
	 * important when displaying the statement. If "main" is null then the statement is defined only informally.
	 */
	private List<Statement> main;

	/**
	 * Main condition, but not formalized. Example: a table instead of a formal definition.
	 */
	private List<Statement> informalMain;

	private List<Statement> proofs;

	/**
	 * Link to other sources
	 */
	private List<Statement> references;

	/**
	 * A list of examples of which only the name is displayed/interesting.
	 */
	private List<Statement> examplesOnlyName;

	private List<Statement> examples;

	/**
	 * Date used to display the content of this statement that can not be modeled by statements.
	 */
	private Object rawData;

	private String type;
	
	/**
	 * Should the identifying name be displayed if the statement is used in a precondition of another statement? Example 1: Be x a variable. Be the statement
	 * the definition of a class. The precondition should then display: x is a class
	 * 
	 * Example 2: Be x,y a variable. Be the statement that x > y . The precondition should then display: x > y (but not that > is the greater relation)
	 */
	private boolean displayIdentifyingNameInPrecondition;
	/**
	 * Always display the identifying name
	 */
	private boolean displayIdentifyingName;

	/**
	 * Should a more detailed variant be displayed. See the constants in WriteExtended.
	 */
	private int displayLevel = Statements.DEFAULT;


	/**
	 * Is it worthwhile to display translations of the name. Default is true.
	 */
	private boolean worthwhileToTranslate = true;

	/**
	 * Are the main statements worthwile to display. Default is true.
	 */
	private boolean worthwileMain = true;
	/**
	 * States that a definition is not formal, but a 'naive' definition.
	 */
	public static final String NAIVE = "naive";

	public static final String DEFAULT = "default";

	private TextDescription displayVariant;

	/**
	 * Used when the statement acts as axiom, lemma, etc.
	 * 
	 * @param name
	 * @param description
	 * @param variables
	 * @param be
	 * @param main
	 * @param links
	 */
	public Statement(CompoundName name, String description, List<Statement> variables, List<Statement> be, List<Statement> main, List<Statement> references) {
		this.identifier = name;
		this.description = description;
		this.variables = variables;
		this.be = be;
		this.main = main;
		this.references = references;
	}

	/**
	 * Used when the statement acts as predicate.
	 * @param name
	 * @param description
	 * @param variables
	 */
	public Statement(CompoundName name, List<Statement> variables) {
		this(name, null, variables, null, null, null);
	}

	public Statement(String category, String name, String nameDetail, String description, List<Statement> variables, List<Statement> be, List<Statement> main,
			List<Statement> references)
	{
		this(compound(category, name, nameDetail), description, variables, be, main, references);
	}

	public Statement(String category, String name, String nameDetail, List<Statement> variables) {
		this(compound(category, name, nameDetail), variables);
	}

	public Statement(String category, String name, String nameDetail) {
		this(compound(category, name, nameDetail), (List<Statement>) null);
	}

	public Statement(CompoundName name, Statement nameOfInstance) {
		this.identifier = name;
		this.nameOfInstance = nameOfInstance;
	}

	public Statement(String category, String name, String nameDetail, Statement nameOfInstance) {
		this(compound(category, name, nameDetail), nameOfInstance);
	}

	public Statement(String compoundName) {
		String[] parts = compoundName.split(INDENTIFIER_SEPARATOR);
		this.identifier = compound(parts);
	}

	private static CompoundName compound(String... parts/* category, String name, String nameDetail */) {
		for (String part : parts) {
			Check.notNull(part, "part");
		}
		// Check.notNull(category, "category");
		// Check.notNull(name, "name");
		// Check.notNull(nameDetail, "nameDetail");
		Properties syntax = new Properties();
		syntax.setProperty("jndi.syntax.direction", "left_to_right");
		syntax.setProperty("jndi.syntax.separator", INDENTIFIER_SEPARATOR);
		syntax.setProperty("jndi.syntax.escape", "\\");
		try {
			CompoundName result = new CompoundName("", syntax);
			for (String part : parts) {
				result.add(part);
			}
			// result.add(category);
			// result.add(name);
			// result.add(nameDetail);
		return result;
		} catch (InvalidNameException e) {

			JOptionPane.showMessageDialog(null, "Statement constructor exception");

			throw ThrowableBoostUtils.toRuntimeException(e);
		}
	}

	/**
	 * changes the name detail into NAIVE.
	 * 
	 * @throws Exception
	 */
	public void makeNaive() throws Exception {
		identifier.remove(2);
		identifier.add(2, NAIVE);
	}

	public String getCategory() {
		return identifier.get(0);
	}

	public String getName() {
		return identifier.get(1);
	}

	public String getNameDetail() {
		if (identifier.size() < 2) {
			return null;
		} else {
			return identifier.get(2);
		}
	}


	public CompoundName getIdentifier() {
		return identifier;
	}

	/**
	 * Returns the identifier of the type (category, name, nameDetail)
	 */
	public Name getTypeIdentifier() {
		return identifier.getPrefix(3);
	}

	public CompoundName getIdentifyingName(){
			if (nameOfInstance != null) {
				return nameOfInstance.getIdentifier();
			} else {
			return getIdentifier();
		}
		
	}

	public void setIdentifier(CompoundName identifier) {
		this.identifier = identifier;
	}

	public String getDescription() {
		return description;
	}

	public void setDescription(String description) {
		this.description = description;
	}

	/**
	 * 
	 * @return never null
	 */
	public List<Statement> getVariables() {
		if (variables == null) {
			variables = new ArrayList();
		}
		return variables;
	}

	public boolean hasVariables() {
		return variables != null && !variables.isEmpty();
	}

	public void setVariables(List<Statement> variables) {
		this.variables = variables;
	}

	public List<Statement> getBe() {
		if (be == null) {
			be = new ArrayList();
		}
		return be;
	}

	public List<Statement> getAllBe() {
		ArrayList<Statement> result = new ArrayList();
		if (be != null) {
			result.addAll(be);
		}
		if (unimportant != null) {
			result.addAll(unimportant);
		}

		return result;

	}

	public void setBe(List<Statement> be) {
		this.be = be;
	}

	public void be(Statement... statements) {
		if (be == null) {
			be = new ArrayList();
		}
		for (Statement statement : statements) {
			be.add(statement);
		}
	}

	public List<Statement> getMain() {
		if (main == null) {
			main = new ArrayList();
		}
		return main;
	}

	public void setMain(List<Statement> main) {
		this.main = main;
	}

	public void setMain(Statement... main) {
		setMain(Arrays.asList(main));
	}

	public void main(Statement... statements) {
		if (main == null) {
			main = new ArrayList();
		}
		for (Statement statement : statements) {
			main.add(statement);
		}
	}

	public List<Statement> getReferences() {
		return references;
	}

	public void setReferences(List<Statement> links) {
		this.references = links;
	}

	public void setReferences(Statement... references) {
		this.references = Arrays.asList(references);
	}

	@Override
	public String toString() {
		return /* getClass().getSimpleName() + " " + */ getIdentifier() + (nameOfInstance == null ? "" : " " + nameOfInstance.getIdentifier()) + " ("
				+ variables + ")";

	}

	/**
	 * Just compares the identifier (only the first three are mandatory, the remaining are optional).
	 * 
	 * @param other
	 * @return
	 */
	public boolean is(Statement other) {
		Check.notNull(other, "other");

		CompoundName thisInstanceName = nameOfInstance != null ? nameOfInstance.getIdentifyingName() : null;
		CompoundName otherInstanceName = other.getIdentifyingName();

		CompoundName thisIdentifier = this.identifier;
		CompoundName otherIdentifier = other.identifier;


		if (thisInstanceName != null) {
			return is(thisInstanceName, otherInstanceName);
		} else {
			return is(thisIdentifier, otherIdentifier);
		}

		// boolean result = is(thisIdentifier, otherIdentifier) || is(this.identifier, other.identifier);
		// return result;
		// int thisSize = thisIdentifier.size();// identifier.size();
		// int otherSize = otherIdentifier.size();
		// if (thisSize == otherSize) {
		// return thisIdentifier.equals(otherIdentifier);
		// } else if (thisSize < 3 || otherSize < 3) {
		// return false;
		// } else {
		// for (int i = 0; i < Math.min(thisSize, otherSize); i++) {
		// if (!thisIdentifier.get(i).equals(otherIdentifier.get(i))) {
		// return false;
		// }
		// }
		// return true;
		// }
	}

	private boolean is(CompoundName thisIdentifier, CompoundName otherIdentifier) {

		if (thisIdentifier == null || otherIdentifier == null) {
			return false;
		}

		int thisSize = thisIdentifier.size();// identifier.size();
		int otherSize = otherIdentifier.size();
		if (thisSize == otherSize) {
			return thisIdentifier.equals(otherIdentifier);
		} else if (thisSize < 3 || otherSize < 3) {
			return false;
		} else {
			for (int i = 0; i < Math.min(thisSize, otherSize); i++) {
				if (!thisIdentifier.get(i).equals(otherIdentifier.get(i))) {
					return false;
				}
			}
			return true;
		}
	}


	public List<Statement> getVariableInfos() {
		return variableInfos;
	}

	public void setVariableInfos(List<Statement> variableInfos) {
		this.variableInfos = variableInfos;
	}

	public <Result> Result getRawData() {
		return (Result) rawData;
	}

	public void setRawData(Object rawData) {
		this.rawData = rawData;
	}

	public Statement getNameOfInstance() {
		return nameOfInstance;
	}

	public void setNameOfInstance(Statement nameOfInstance) {
		this.nameOfInstance = nameOfInstance;
	}

	public void setLemma() {
		type = TYPE_LEMMA;
	}

	public void setDefinition() {
		type = TYPE_DEFINITION;
	}

	public void setTheorem() {
		type = TYPE_THEOREM;
	}

	public void setExample() {
		type = TYPE_EXAMPLE;
	}

	public boolean isLemma() {
		return TYPE_LEMMA.equals(type);
	}

	public boolean isDefinition() {
		return TYPE_DEFINITION.equals(type);
	}

	public boolean isTheorem() {
		return TYPE_THEOREM.equals(type);
	}

	public String getType() {
		return type;
	}

	public void setType(String type) {
		this.type = type;
	}


	@Override
	public <Implementation> Implementation cloneMe() throws Exception {

		return (Implementation) clone();
	}

	public <Implementation extends Statement> Implementation copy() {

		try {
			return cloneMe();
		} catch (Exception e) {
			throw ThrowableBoostUtils.toRuntimeException(e);
		}
	}

	public <Implementation extends Statement> Implementation cloneWithDisplayIdentifyingName() {

		Implementation result = copy();
		result.setDisplayIdentifyingName(true);
		return result;
	}

	public boolean isDisplayIdentifyingNameInPrecondition() {
		return displayIdentifyingNameInPrecondition;
	}

	public Statement setDisplayIdentifyingNameInPrecondition(boolean displayIdentifyingNameInPrecondition) {
		this.displayIdentifyingNameInPrecondition = displayIdentifyingNameInPrecondition;
		return this;
	}

	public boolean isWorthwhileToTranslate() {
		return worthwhileToTranslate;
	}

	public void setWorthwhileToTranslate(boolean worthwhileToTranslate) {
		this.worthwhileToTranslate = worthwhileToTranslate;
	}

	public List<Statement> getProofs() {
		return proofs;
	}

	public void setProofs(List<Statement> proofs) {
		this.proofs = proofs;
	}

	public void setProofs(Statement... proofs) {
		this.proofs = Arrays.asList(proofs);
	}

	public List<Statement> getUnimportant() {
		if (unimportant == null) {
			unimportant = new ArrayList();
		}

		return unimportant;
	}

	public void setUnimportant(List<Statement> unimportantBe) {
		this.unimportant = unimportantBe;
	}

	public void setUnimportant(Statement... unimportantBe) {
		this.unimportant = Arrays.asList(unimportantBe);
	}

	public List<Statement> getInformalMain() {
		return informalMain;
	}

	public void setInformalMain(List<Statement> informalMain) {
		this.informalMain = informalMain;
	}

	public void setInformalMain(Statement... informalMain) {
		this.informalMain = Arrays.asList(informalMain);
	}

	public List<Statement> getExamplesOnlyName() {
		return examplesOnlyName;
	}

	public void setExamplesOnlyName(List<Statement> examplesOnlyName) {
		this.examplesOnlyName = examplesOnlyName;
	}

	public void setExamplesOnlyName(Statement... examplesOnlyName) {
		this.examplesOnlyName = Arrays.asList(examplesOnlyName);
	}

	public boolean isWorthwileMain() {
		return worthwileMain;
	}

	public void setWorthwileMain(boolean worthwileMain) {
		this.worthwileMain = worthwileMain;
	}


	public boolean isDisplayIdentifyingName() {
		return displayIdentifyingName;
	}

	public Statement setDisplayIdentifyingName(boolean displayIdentifyingName) {
		this.displayIdentifyingName = displayIdentifyingName;
		return this;
	}

	public List<Statement> getExamples() {
		return examples;
	}

	public void setExamples(List<Statement> examples) {
		this.examples = examples;
	}

	public void setExamples(Statement... examples) {
		this.examples = Arrays.asList(examples);
	}

	public int getDisplayLevel() {
		return displayLevel;
	}

	public Statement setDisplayLevel(int displayLevel) {
		this.displayLevel = displayLevel;
		return this;
	}

	public TextDescription getDisplayVariant() {
		return displayVariant;
	}

	public void setDisplayVariant(TextDescription displayVariant) {
		this.displayVariant = displayVariant;
	}

	public Statement setDisplayVariant(Object... variants) {
		displayVariant = new TextDescription();
		displayVariant.setVariant(variants);
		return this;
	}
}
