package net.sf.gluebooster.demos.pojo.math;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Locale;

import org.w3c.dom.Node;

import net.sf.gluebooster.demos.pojo.math.library.Basics;
import net.sf.gluebooster.demos.pojo.math.library.References;
import net.sf.gluebooster.demos.pojo.math.library.VariableExamples;
import net.sf.gluebooster.demos.pojo.math.library.logic.Logic;
import net.sf.gluebooster.demos.pojo.math.studies.RuleContext;
import net.sf.gluebooster.demos.pojo.math.studies.StudyResourceBundle;
import net.sf.gluebooster.demos.pojo.math.studies.Write;
import net.sf.gluebooster.demos.pojo.math.studies.WriteAfterStatementTransformation;
import net.sf.gluebooster.demos.pojo.math.studies.WriteOperation;
import net.sf.gluebooster.java.booster.essentials.eventsCommands.CallableAbstraction;
import net.sf.gluebooster.java.booster.essentials.logging.LogBooster;
import net.sf.gluebooster.java.booster.essentials.meta.objects.GraphElementDescription;
import net.sf.gluebooster.java.booster.essentials.utils.ThrowableBoostUtils;
import net.sourceforge.jeuclid.elements.presentation.table.Mtable;
import net.sourceforge.jeuclid.elements.presentation.table.Mtd;
import net.sourceforge.jeuclid.elements.presentation.table.Mtr;

/**
 * Utilities for classes containing statements.
 * 
 * @author cbauer
 *
 */
public class Statements implements VariableExamples {

	private static final LogBooster LOG = new LogBooster(Statements.class);

	public static final int SHORT = 0;
	public static final int DEFAULT = 1;
	public static final int EXTENDED = 2;
	public static final int VERY_EXTENDED = 3;

	/**
	 * The category the statements belong to.
	 */
	private String category;

	private Statements parentCategory;



	public String getCategory() {
		return category;
	}

	public void setCategory(String category) {
		this.category = category;
	}

	/**
	 * Default constructor for tests.
	 */
	private Statements() {

	}

	public Statements(String category, Statements parentCategory) {
		this.category = category;
		this.parentCategory = parentCategory;
	}

	public Statement nameDetails(String name, String nameDetail) {
		return new Statement(category, name, nameDetail);
	}

	public Statement nameDetailsVar(String name, String nameDetail, List<Statement> variables) {
		return new Statement(category, name, nameDetail, variables);
	}

	public Statement naive(String name) {
		return nameDetails(name, Statement.NAIVE);
	}

	public Statement naive(String name, Statement... s1) {
		return naive(name, nonNullList(s1));
	}

	public Statement naive(String name, List<Statement> variables) {
		return nameDetailsVar(name, Statement.NAIVE, variables);
	}

	public Statement naive(String name, List<Statement> variables, List<Statement> main) {
		Statement result = naive(name, variables);
		result.setMain(main);
		return result;
	}

	public static Statement naiveWithIdentifyingName(Statements factory, String statementName, String name, Statement... indices) {
		try {
			Statement result = factory.naive(statementName, Arrays.asList(indices));
			result.setDisplayIdentifyingNameInPrecondition(true);
			if (name != null) {
				result.getIdentifier().add(name);
				result.setDescription(name);
			}
			return result;
		} catch (Exception ex) {
			throw ThrowableBoostUtils.toRuntimeException(ex);
		}
	}

	public Statement naiveBinary(String name, Statement s1, Statement s2) {
		return naive(name, nonNullList(s1, s2));
	}

	public Statement normal(String name) {
		return nameDetails(name, Statement.DEFAULT);
	}

	public Statement normal(String name, Statement... statement) {
		return normal(name, nonNullList(statement));
	}

	public Statement normal(Statement nameOfInstance, String name, Statement... statement) {
		Statement result = normal(name, statement);
		result.setNameOfInstance(nameOfInstance);
		return result;
	}

	public Statement normal(String name, List<Statement> variables) {
		return nameDetailsVar(name, Statement.DEFAULT, variables);
	}

	public Statement normal(String name, List<Statement> variables, List<Statement> main) {
		Statement result = normal(name, variables);
		result.setMain(main);
		return result;
	}

	/**
	 * A binary operation
	 * 
	 * @param name
	 * @param s1
	 * @param s2
	 * @return
	 */
	public Statement binary(String name, Statement s1, Statement s2) {
		return normal(name, Arrays.asList(s1, s2, null));// the null for the result; nonNullList is wrong
	}

	public Statement statement(String name, String nameDetail) {
		return new Statement(category, name, nameDetail);
	}

	public Statement statement(String name, String nameDetail, List<Statement> variables) {
		return new Statement(category, name, nameDetail, variables);
	}

	/**
	 * If all elements are null return an empty list
	 * 
	 * @param statements
	 * @return
	 */
	protected static List<Statement> nonNullList(Statement... statements) {

		boolean added = false;
		boolean ignored = false;
		// ArrayList<Statement> result = new ArrayList();
		for (Statement statement : statements) {
			if (statement != null) {
				// result.add(statement);
				added = true;
			} else {
				ignored = true;
				// LOG.trace("ignored one null element");
			}
		}

		if (!added) {
			return Collections.EMPTY_LIST;
		} else {
			return Arrays.asList(statements);
		}
	}

	protected static void titleGerman(Object key, Object value) {
		MathStudies.termsDefinitionsTranslator.putDefaultValue(Locale.GERMAN, transformKey(key), value);
	}

	protected static void titleEnglish(Object key, Object value) {
		MathStudies.termsDefinitionsTranslator.putDefaultValue(Locale.ENGLISH, transformKey(key), value);
	}


	private static String transformKey(Object key){
		if (key instanceof Statement){
			key = StudyResourceBundle.id2((Statement) key);
		}

		return key.toString();
	}

	protected static void textGerman(Object key, Object value) {
		if (value != null) {
			MathStudies.termsDefinitionsTextTranslator.putDefaultValue(Locale.GERMAN, transformKey(key), value);
		}
	}


	protected static void textEnglish(Object key, Object value) {
		if (value != null) {
			MathStudies.termsDefinitionsTextTranslator.putDefaultValue(Locale.ENGLISH, transformKey(key), value);
		}
	}

	protected static void titleText(Object key, String titleEnglish, Object textEnglish, String titleGerman, Object... textGerman) {
		titleEnglish(key, titleEnglish);
		textEnglish(key, textEnglish);
		titleGerman(key, titleGerman);
		if (textGerman != null && textGerman.length > 0) {
			textGerman(key, Arrays.asList(textGerman));
		}

	}

	protected static void lemma(Statement key, String titleEnglish, Object textEnglish, String titleGerman, Object... textGerman) {
		titleText(key, titleEnglish, textEnglish, titleGerman, textGerman);
		key.setLemma();
	}

	protected static void example(Statement key, String titleEnglish, Object textEnglish, String titleGerman, Object... textGerman) {
		titleText(key, titleEnglish, textEnglish, titleGerman, textGerman);
		key.setExample();
	}

	protected static void definition(Statement key, String titleEnglish, Object textEnglish, String titleGerman, Object... textGerman) {
		titleText(key, titleEnglish, textEnglish, titleGerman, textGerman);
		key.setDefinition();
	}

	protected static void theorem(Statement key, String titleEnglish, Object textEnglish, String titleGerman, Object... textGerman) {
		titleText(key, titleEnglish, textEnglish, titleGerman, textGerman);
		key.setTheorem();
	}

	public Statement lemma(String name) {
		Statement result = normal(name);
		result.setLemma();
		return result;
	}

	public Statement theorem(String name) {
		Statement result = normal(name);
		result.setTheorem();
		return result;
	}

	public String unit(int number) {
		return getCategory() + "." + number;
	}

	public Statements getParentCategory() {
		return parentCategory;
	}

	public void setParentCategory(Statements parentCategory) {
		this.parentCategory = parentCategory;
	}

	protected static Write writeMantissaIndexExponent(Object mantissa, Object index, Object exponent) {
		if (index == null) {
			index = "";
		}

		if (exponent == null) {
			exponent = "";
		}

		return WriteAfterStatementTransformation.ruleTransformation(Basics.MANTISSA_INDEX_EXPONENT, mantissa, index, exponent);
	}

	protected static Write writeIndexed(Object mantissa, Object index) {
		return writeMantissaIndexExponent(mantissa, index, null);
	}

	protected static Write writeExponent(Object mantissa, Object exponent) {
		return writeMantissaIndexExponent(mantissa, null, exponent);
	}

	protected static Write writeIndexedFromTo(String operator) {
		return WriteAfterStatementTransformation.ruleTransformation(Basics.NOT_SEPARATED,
				new Object[] { Basics.UNDEROVER, operator, new Object[] { Logic.EQUALS, FIRST_VARIABLE, SECOND_VARIABLE }, THIRD_VARIABLE }, FOURTH_VARIABLE);
	}

	public static CallableAbstraction<RuleContext<Statement>, Node> writeMultiline(Statement operator) {

		return new WriteOperation(//
				GraphElementDescription.createNestedParentElementsWithIndex(0, Mtable.ELEMENT, Mtr.ELEMENT, Mtd.ELEMENT), //
				GraphElementDescription.createNestedParentElementsWithIndex(1, Mtr.ELEMENT, Mtd.ELEMENT, operator), //
				null);
	}

	public static CallableAbstraction<RuleContext<Statement>, Node> writeMultilineOperatorSeparate(Statement operator) {

		return new WriteOperation(//
				GraphElementDescription.createNestedParentElementsWithIndex(0, Mtable.ELEMENT, Mtr.ELEMENT, Mtd.ELEMENT), //
				new Object[] { GraphElementDescription.createNestedParentElementsWithIndex(1, Mtr.ELEMENT, Mtd.ELEMENT, operator), // operator separate
						GraphElementDescription.createNestedParentElementsWithIndex(1, Mtr.ELEMENT, Mtd.ELEMENT),// a new line
				}, //
				null);
	}

	protected static Statement kat1(String name) throws Exception {
		return References.kat1(name);
	}

	protected static Statement ml1(String name) throws Exception {
		return References.ml1(name);
	}

	protected static Statement ml2(String name) throws Exception {
		return References.ml2(name);
	}

	protected static Statement wt1(String name) throws Exception {
		return References.wt1(name);
	}

	protected static Statement stat2(String name) throws Exception {
		return References.stat2(name);
	}

	protected static Statement an1_1(String name) throws Exception {
		return References.an1_1(name);
	}

	protected static Statement wikiDe(String name) throws Exception {
		return References.wikiDe(name);
	}

	protected static Statement wikiEn(String name) throws Exception {
		return References.wikiEn(name);
	}
}
