package net.sf.gluebooster.demos.pojo.math.library.logic;

import java.util.Arrays;

import net.sf.gluebooster.demos.pojo.math.MathStudiesProofs;
import net.sf.gluebooster.demos.pojo.math.PrologProofGenerator;
import net.sf.gluebooster.demos.pojo.math.Statement;
import net.sf.gluebooster.demos.pojo.math.Statements;
import net.sf.gluebooster.demos.pojo.math.library.Basics;
import net.sf.gluebooster.demos.pojo.math.library.References;
import net.sf.gluebooster.demos.pojo.math.library.setTheory.ClassesSets;
import net.sf.gluebooster.demos.pojo.math.library.setTheory.TuplesFactory;
import net.sf.gluebooster.demos.pojo.math.library.setTheory.relations.RelationSpecial;
import net.sf.gluebooster.demos.pojo.math.studies.MathMLGenerator;
import net.sf.gluebooster.demos.pojo.math.studies.RuleSelect;
import net.sf.gluebooster.demos.pojo.math.studies.StudyUnit;
import net.sf.gluebooster.demos.pojo.math.studies.WriteAfterStatementTransformation;
import net.sf.gluebooster.demos.pojo.math.studies.WriteOperation;
import net.sf.gluebooster.java.booster.essentials.meta.objects.GraphElementDescription;
import net.sf.gluebooster.java.booster.essentials.utils.ThrowableBoostUtils;
import net.sourceforge.jeuclid.elements.presentation.table.Mtable;
import net.sourceforge.jeuclid.elements.presentation.table.Mtd;
import net.sourceforge.jeuclid.elements.presentation.table.Mtr;

/**
 * Boolean domain with two elements.
 *
 * @see https://en.wikipedia.org/wiki/Boolean_algebra
 * @see http://www.zweigmedia.com/RealWorld/logic/logic5.html
 * @author cbauer
 *
 */
public class Bool extends BoolFactory {

	/**
	 * The boolean value TRUE
	 */
	public static Statement VERUM;

	/**
	 * The boolean value FALSE
	 */
	public static Statement FALSUM;

	public static Statement BOOLEAN_SET;

	/**
	 * The negation of a formula/truth value
	 */
	public static Statement NOT_RELATION;

	/**
	 * The conjunction of two formulas/truth values
	 */
	public static Statement AND_RELATION;

	/**
	 * Multiple conjunctions
	 */
	public static Statement AND_MULTIPLE;

	public static Statement AND_MULTILINE;

	public static Statement AND_COMMUTATIVE;

	public static Statement AND_IDEMPOTENT;

	public static Statement AND_ASSOCIATIVE;


	/**
	 * The disjunction of two formulas/truth values
	 */
	public static Statement OR_RELATION;

	public static Statement OR_COMMUTATIVE;

	public static Statement OR_IDEMPOTENT;

	public static Statement OR_ASSOCIATIVE;

	/**
	 * The implication A -> B between two formulas/truth values
	 */
	public static Statement CONDITIONAL_RELATION;

	/**
	 * Multiple implications.
	 */
	public static Statement CONDITIONAL_MULTI;
	public static Statement CONDITIONAL_MULTILINE;
	public static Statement BICONDITIONAL_MULTILINE;

	/**
	 * B <- A is another notation of the implication A -> B between two formulas/truth values
	 */
	public static Statement IMPLIED_BY_NAIVE;

	/**
	 * Both the implication A -> B and B -> A between two formulas/truth values
	 */
	public static Statement BICONDITIONAL_RELATION;

	public static Statement DEFINED_AS_EQUAL_TO;
	public static Statement DEFINED_AS_EQUAL_TO_MULTILINE;

	public static Statement XOR_RELATION;

	public static Statement NAND_RELATION;

	public static Statement NOR_RELATION;

	public static Statement TRANSITIVE_CONDITIONAL;

	/**
	 * not a = a nand a ...
	 */
	public static Statement NOT_FROM_OTHERS;

	public static Statement BICONDITIONAL_FROM_OTHERS;

	public static Statement BICONDITONAL_PROOF_BY_IMPLICATIONS;

	public static Statement XOR_FROM_OTHERS;

	public static Statement CONDITIONAL_FROM_OTHERS;

	public static Statement NAND_FROM_OTHERS;

	public static Statement AND_FROM_OTHERS;

	public static Statement OR_FROM_OTHERS;

	public static Statement MODUS_PONENS;

	public static Statement MODUS_TOLLENS;

	public static Statement CONTRAPOSITION;

	public static Statement DISJUNCTIVE_SYLLOGISM;

	public static Statement DOUBLE_NEGATION;

	public static Statement DE_MORGAN_1;

	public static Statement DE_MORGAN_2;

	// TODO translate
	public static Statement RINGSCHLUSS;

	public static Statement DISTRIBUTIVE_1;

	public static Statement DISTRIBUTIVE_2;

	public static Statement IDEMPOTENT_AND;

	public static Statement IDEMPOTENT_OR;

	public static Statement TERTIUM_NON_DATUR;

	public static Statement NONCONTRADICTION;

	public static Statement SPECIALIZATION;

	public static Statement GENERALIZATION;

	/**
	 * The space of Boolean with XOR as funktion
	 */
	public static Statement BOOL_XOR;
	public static Statement BOOL_AND;

	public static final WriteOperation WRITE_OPERATION_AND = new WriteOperation("∧");

	static {
		try {
			titleText(SINGLETON.unit(1), "Boolean domain (1)", null, "Boolesche Algebra (1)", null);

			VERUM = SINGLETON.naive("verum");
			VERUM.setNameOfInstance(VERUM);
			VERUM.setReferences(References.wikiDe("Wahrheitswert"), References.wikiEn("Truth_value"));
			definition(VERUM, "True", null, "Verum", "Mit Verum wird der Wahrheitswert WAHR bezeichnet. Geschrieben wird er oft als W oder ", Bool.VERUM,
					" (Tautologie)");
			MathMLGenerator.displayRule(VERUM, WriteOperation.prefix("⊤"));

			FALSUM = SINGLETON.naive("falsum");
			FALSUM.setNameOfInstance(FALSUM);
			FALSUM.setReferences(References.wikiDe("Wahrheitswert"), References.wikiEn("Truth_value"));
			definition(FALSUM, "False", null, "Falsum", "Mit Falsum wird der Wahrheitswert FALSCH bezeichnet. Geschrieben wird er oft als F oder ",
					Bool.FALSUM);
			MathMLGenerator.displayRule(FALSUM, WriteOperation.prefix("⊥"));

			BOOLEAN_SET = SINGLETON.normal("boolean set");// ClassesSets.explicitSet(SINGLETON.naive("boolean set"), Arrays.asList(VERUM, FALSUM));
			BOOLEAN_SET.setReferences(References.wikiDe("Wahrheitswert"), References.wikiDe("Boolesche_Algebra"), References.wikiEn("Truth_value"),
					References.wikiEn("Boolean_domain"));
			BOOLEAN_SET.setMain(Logic.definedAs(BOOLEAN_SET, ClassesSets.explicitSet(null, Arrays.asList(VERUM, FALSUM))));
			definition(BOOLEAN_SET, "Boolean domain (truth values)", null, "Boolesche Menge (Wahrheitswerte)",
					"Eine Menge mit zwei Elementen. In der klassischen zweiwertigen Logik bedeuten die Elemente die Wahrheitswerte  WAHR (VERUM) und FALSCH (FALSUM). Bei Schaltungen verwendet man oft die Symbole 0 und 1. Sie kann also als Menge wie folgt geschrieben werden: ",
					ClassesSets.explicitSet(null, Arrays.asList(Bool.FALSUM, Bool.VERUM)));
			MathMLGenerator.displayRule(BOOLEAN_SET, WriteAfterStatementTransformation.ruleTransformation(Basics.MANTISSA_INDEX_EXPONENT, "\u23b9B", "", ""));

			NOT_RELATION = RelationSpecial.explicitRelation(not(null),
					new Object[][] { { Bool.VERUM, Bool.FALSUM }, { Bool.FALSUM, Bool.VERUM } });
			NOT_RELATION.setReferences(References.wikiDe("Aussagenlogik"), References.wikiDe("Prädikatenlogik"),
					References.wikiEn("First-order_logic#Evaluation_of_truth_values"));
			NOT_RELATION.setInformalMain(Basics.mathTable(Arrays.asList(A), Arrays.asList(Bool.NOT_RELATION)));
			definition(NOT_RELATION, "Negation", null, "Negation",
					"Die Negation (Verneinung) eines Wahrheitswerts (Aussage) dreht den Wahrheitswert um.  ");
			MathMLGenerator.displayRule(NOT_RELATION, WriteOperation.prefix("¬"));

			AND_RELATION = RelationSpecial.explicitRelation(and(EMPTY, EMPTY),
					new Object[][] { { Bool.VERUM, Bool.VERUM, Bool.VERUM }, { Bool.VERUM, Bool.FALSUM, Bool.FALSUM },
							{ Bool.FALSUM, Bool.VERUM, Bool.FALSUM }, { Bool.FALSUM, Bool.FALSUM, Bool.FALSUM } });
			AND_RELATION.setReferences(References.wikiDe("Aussagenlogik"), References.wikiDe("Prädikatenlogik"),
					References.wikiEn("First-order_logic#Evaluation_of_truth_values"));
			AND_RELATION.setInformalMain(Basics.mathTable(Arrays.asList(A, B), Arrays.asList(Bool.AND_RELATION)));
			definition(AND_RELATION, "And", null, "Und",
					"Eine Konjunktion (UND-Verknüpfung) zweier Wahrheitswerte (Aussagen) ist genau dann wahr, wenn beide Argumente wahr sind. ");
			MathMLGenerator.displayRule(AND_RELATION, WRITE_OPERATION_AND);

			AND_COMMUTATIVE = SINGLETON.normal("AND_COMMUTATIVE");
			// AND_ASSOCIATIVE.setReferences(References.wikiEn("Union_(set_theory)"));
			AND_COMMUTATIVE.setMain(Logic.equals(and(A, B), and(B, A)));
			lemma(AND_COMMUTATIVE, null, null, "Kommutativgesetz von UND", "Der Beweise ist einfach aus der Wahrheitstabelle abzulesen");

			AND_IDEMPOTENT = SINGLETON.normal("AND_IDEMPOTENT");
			AND_IDEMPOTENT.setReferences(References.wikiDe("Idempotenz"), References.wikiEn("Idempotence"));
			// AND_ASSOCIATIVE.setReferences(References.wikiEn("Union_(set_theory)"));
			AND_IDEMPOTENT.setMain(Logic.equals(and(A, A), A));
			lemma(AND_IDEMPOTENT, null, null, "Idempotenz von UND", "Der Beweise ist einfach aus der Wahrheitstabelle abzulesen");

			AND_ASSOCIATIVE = SINGLETON.normal("AND_ASSOCIATIVE");
			// AND_ASSOCIATIVE.setReferences(References.wikiEn("Union_(set_theory)"));
			AND_ASSOCIATIVE.setMain(Logic.equals(and(Logic.bracket(and(A, B)), C), and(A, Logic.bracket(and(B, C)))));
			lemma(AND_ASSOCIATIVE, null, null, "Assoziativgesetz von UND", null);

			AND_MULTILINE = andMultiline();
			MathMLGenerator.displayRule(AND_MULTILINE, writeMultilineOperatorSeparate(Bool.and(Basics.EMPTY, Basics.EMPTY)));

			OR_RELATION = RelationSpecial.explicitRelation(or(EMPTY, EMPTY),
					new Object[][] { { Bool.VERUM, Bool.VERUM, Bool.VERUM }, { Bool.VERUM, Bool.FALSUM, Bool.VERUM },
							{ Bool.FALSUM, Bool.VERUM, Bool.VERUM }, { Bool.FALSUM, Bool.FALSUM, Bool.FALSUM } });
			OR_RELATION.setReferences(References.wikiDe("Aussagenlogik"), References.wikiDe("Prädikatenlogik"),
					References.wikiEn("First-order_logic#Evaluation_of_truth_values"));
			OR_RELATION.setInformalMain(Basics.mathTable(Arrays.asList(A, B), Arrays.asList(Bool.OR_RELATION)));
			definition(OR_RELATION, "Or", null, "Oder",
					"Eine nichtausschließende ODER-Verknüpfung (Disjunktion, Adjunktion) zweier Wahrheitswerte (Aussagen) ist genau dann wahr, wenn mindestens eines der beiden Argumente wahr ist.  ");
			MathMLGenerator.displayRule(OR_RELATION, new WriteOperation("∨"));

			OR_COMMUTATIVE = SINGLETON.normal("OR_COMMUTATIVE");
			// AND_ASSOCIATIVE.setReferences(References.wikiEn("Union_(set_theory)"));
			OR_COMMUTATIVE.setMain(Logic.equals(or(A, B), or(B, A)));
			lemma(OR_COMMUTATIVE, null, null, "Kommutativgesetz von ODER", "Der Beweise ist einfach aus der Wahrheitstabelle abzulesen");

			OR_IDEMPOTENT = SINGLETON.normal("OR_IDEMPOTENT");
			OR_IDEMPOTENT.setReferences(References.wikiDe("Idempotenz"), References.wikiEn("Idempotence"));
			OR_IDEMPOTENT.setMain(Logic.equals(and(A, A), A));
			lemma(OR_IDEMPOTENT, null, null, "Idempotenz von ODER", "Der Beweise ist einfach aus der Wahrheitstabelle abzulesen");

			OR_ASSOCIATIVE = SINGLETON.normal("OR_ASSOCIATIVE");
			// AND_ASSOCIATIVE.setReferences(References.wikiEn("Union_(set_theory)"));
			OR_ASSOCIATIVE.setMain(Logic.equals(or(Logic.bracket(or(A, B)), C), or(A, Logic.bracket(or(B, C)))));
			lemma(OR_ASSOCIATIVE, null, null, "Assoziativgesetz von ODER", null);

			CONDITIONAL_RELATION = RelationSpecial.explicitRelation(implies(EMPTY, EMPTY),
					new Object[][] { { Bool.VERUM, Bool.VERUM, Bool.VERUM }, { Bool.VERUM, Bool.FALSUM, Bool.FALSUM },
							{ Bool.FALSUM, Bool.VERUM, Bool.VERUM }, { Bool.FALSUM, Bool.FALSUM, Bool.VERUM } });
			CONDITIONAL_RELATION.setReferences(References.wikiDe("Aussagenlogik"), References.wikiDe("Prädikatenlogik"),
					References.wikiEn("First-order_logic#Evaluation_of_truth_values"));
			CONDITIONAL_RELATION.setInformalMain(Basics.mathTable(Arrays.asList(A, B), Arrays.asList(Bool.CONDITIONAL_RELATION)));
			definition(CONDITIONAL_RELATION, "Implication", null, "Implikation",
					"Die (materiale) Implikation (Konditional, Subjunktion) zwischen zwei Wahrheitswerte (Aussagen) bedeutet, dass ein wahrer Wert (Aussage) A hinreichend für die Wahrheit eines zweiten Werts (Aussage) B ist. Beachte, dass eine falsche Aussage jede beliebige andere Aussage impliziert. Sprechweisen: 'Wenn A, dann B', 'A impliziert B', 'A ist hinreichend für B', 'B ist notwendig für A'. ",
					"Schreibweise: ", Bool.impliedBy(B, A), " steht für ", Bool.implies(A, B), ".  ");
			MathMLGenerator.displayRule(CONDITIONAL_RELATION, new WriteOperation("⇒"));


			IMPLIED_BY_NAIVE = impliedBy(A, B);
			IMPLIED_BY_NAIVE.setReferences(References.wikiDe("Aussagenlogik"), References.wikiDe("Prädikatenlogik"),
					References.wikiEn("First-order_logic#Evaluation_of_truth_values"));
			IMPLIED_BY_NAIVE.setDefinition();
			MathMLGenerator.displayRule(IMPLIED_BY_NAIVE, new WriteOperation("⇐"));

			CONDITIONAL_MULTILINE = impliesMultiline();
			MathMLGenerator.displayRule(CONDITIONAL_MULTILINE, writeMultiline(Bool.implies(Basics.EMPTY, Basics.EMPTY)));

			CONDITIONAL_MULTI = impliesMulti();
			MathMLGenerator.displayRule(CONDITIONAL_MULTI, new WriteOperation("⇒"));

			BICONDITIONAL_RELATION = RelationSpecial.explicitRelation(biconditional(EMPTY, EMPTY),
					new Object[][] { { Bool.VERUM, Bool.VERUM, Bool.VERUM }, { Bool.VERUM, Bool.FALSUM, Bool.FALSUM },
							{ Bool.FALSUM, Bool.VERUM, Bool.FALSUM }, { Bool.FALSUM, Bool.FALSUM, Bool.VERUM } });
			MathMLGenerator.displayRule(BICONDITIONAL_RELATION, new WriteOperation("⇔"));

			// BICONDITIONAL_RELATION.setNameOfInstance(biconditional(null, null));
			BICONDITIONAL_RELATION.setReferences(References.wikiDe("Aussagenlogik"), References.wikiDe("Prädikatenlogik"),
					References.wikiEn("First-order_logic#Evaluation_of_truth_values"), References.ml1("1.2.5"));
			BICONDITIONAL_RELATION.setInformalMain(Basics.mathTable(Arrays.asList(A, B), Arrays.asList(Bool.BICONDITIONAL_RELATION)));
			definition(BICONDITIONAL_RELATION, "Biconditional", null, "Bikonditional",
					"Ein Bikonditional ( [materielle] Äquivalenz) zwischen zwei Wahrheitswerten (Aussagen) bedeutet, dass beide Argumente den gleichen Wahrheitswert haben. Ist die erste Aussage A durch die zweite Aussage B definiert, so schreibt man ",
					Bool.definedAsEqualTo(A, B),
					".  Sprechweisen: 'A genau dann, wenn B (A gdw. B)', 'A dann und nur dann, wenn B', 'A ist gleichbedeutend mit B', 'A ist äquivalent zu B', 'A ist gleichwertig zu B', 'A ist hinreichend und notwendig für B', 'A dann und nur dann, wenn B'.");

			BICONDITIONAL_MULTILINE = biconditionalMultiline();
			MathMLGenerator.displayRule(BICONDITIONAL_MULTILINE, writeMultiline(iff(Basics.EMPTY, Basics.EMPTY)));

			DEFINED_AS_EQUAL_TO = definedAsEqualTo(A, B);
			DEFINED_AS_EQUAL_TO.makeNaive();
			MathMLGenerator.displayRule(DEFINED_AS_EQUAL_TO, new WriteOperation(":⇔"));

			DEFINED_AS_EQUAL_TO_MULTILINE = definedAsEqualToMultiline();
			MathMLGenerator.displayRule(DEFINED_AS_EQUAL_TO_MULTILINE,
					writeMultilineOperatorSeparate(definedAsEqualTo(Basics.EMPTY, Basics.EMPTY)));

			XOR_RELATION = RelationSpecial.explicitRelation(xor(EMPTY, EMPTY),
					new Object[][] { { Bool.VERUM, Bool.VERUM, Bool.FALSUM }, { Bool.VERUM, Bool.FALSUM, Bool.VERUM },
							{ Bool.FALSUM, Bool.VERUM, Bool.VERUM }, { Bool.FALSUM, Bool.FALSUM, Bool.FALSUM } });
			XOR_RELATION.setReferences(References.wikiDe("Kontravalenz"), References.wikiEn("Exclusive_or"));
			XOR_RELATION.setInformalMain(Basics.mathTable(Arrays.asList(A, B), Arrays.asList(Bool.XOR_RELATION)));
			definition(XOR_RELATION, "XOR", null, "Kontravalenz",
					"XOR steht für Exclusive OR (ausschließendes ODER). Es ist nur dann wahr, wenn genau eines der Argumente wahr ist.");
			MathMLGenerator.displayRule(XOR_RELATION, new WriteOperation(" XOR "));

			NAND_RELATION = RelationSpecial.explicitRelation(nand(EMPTY, EMPTY),
					new Object[][] { { Bool.VERUM, Bool.VERUM, Bool.FALSUM }, { Bool.VERUM, Bool.FALSUM, Bool.VERUM },
							{ Bool.FALSUM, Bool.VERUM, Bool.VERUM }, { Bool.FALSUM, Bool.FALSUM, Bool.VERUM } });
			NAND_RELATION.setReferences(References.wikiDe("NAND-Gatter"), References.wikiDe("Shefferscher_Strich"), References.wikiEn("NAND_gate"),
					References.wikiEn("Sheffer_stroke"));
			NAND_RELATION.setInformalMain(Basics.mathTable(Arrays.asList(A, B), Arrays.asList(Bool.NAND_RELATION)));
			definition(NAND_RELATION, "NAND", null, "Shefferscher Strich",
					"NAND steht für Not AND, also für Nicht-Und. Mit NAND lassen sich alle anderen Operatoren erzeugen.");
			MathMLGenerator.displayRule(NAND_RELATION, new WriteOperation(" NAND "));

			NOR_RELATION = RelationSpecial.explicitRelation(nor(EMPTY, EMPTY),
					new Object[][] { { Bool.VERUM, Bool.VERUM, Bool.FALSUM }, { Bool.VERUM, Bool.FALSUM, Bool.FALSUM },
							{ Bool.FALSUM, Bool.VERUM, Bool.FALSUM }, { Bool.FALSUM, Bool.FALSUM, Bool.VERUM } });
			NOR_RELATION.setReferences(References.wikiDe("NOR-Gatter"), References.wikiEn("NOR_gate"));
			NOR_RELATION.setInformalMain(
					Basics.mathTable(Arrays.asList(A, B), Arrays.asList(Bool.NOR_RELATION/* , Boolean.NAND_RELATION, Boolean.AND_RELATION */)));
			definition(NOR_RELATION, "NOR", null, "Peirce-Funktion",
					"NOR steht für Not OR, also für Nicht-Oder.  Mit NOR lassen sich alle anderen Operatoren erzeugen.");
			MathMLGenerator.displayRule(NOR_RELATION, new WriteOperation(" NOR "));
			
			TRANSITIVE_CONDITIONAL = SINGLETON.naive("TRANSITIVE_CONDITIONAL");
			TRANSITIVE_CONDITIONAL.setWorthwhileToTranslate(false);
			TRANSITIVE_CONDITIONAL.setUnimportant(ClassesSets.elementOf(A, BOOLEAN_SET), ClassesSets.elementOf(B, BOOLEAN_SET),
					ClassesSets.elementOf(C, BOOLEAN_SET));
			TRANSITIVE_CONDITIONAL.setMain(impliesWithBrackets(andWithBrackets(implies(A, B), implies(B, C)), implies(A, C)));
			// proof separate added below
			lemma(TRANSITIVE_CONDITIONAL, "Conditional is transitive", null, "Implikation ist transitiv",
					"NOR steht für Not OR, also für Nicht-Oder.  Mit NOR lassen sich alle anderen Operatoren erzeugen.");

			NOT_FROM_OTHERS = SINGLETON.naive("NOT_FROM_OTHERS");
			NOT_FROM_OTHERS.setWorthwhileToTranslate(false);
			NOT_FROM_OTHERS.setUnimportant(ClassesSets.elementOf(A, BOOLEAN_SET));
			NOT_FROM_OTHERS.setMain(Logic.equals(Arrays.asList(not(A), nand(A, A))));
			// proof separate added below
			lemma(NOT_FROM_OTHERS, "NOT by other operators", null, "Darstellung von NOT mittels anderer Operatoren",
					null);

			BICONDITIONAL_FROM_OTHERS = SINGLETON.naive("BICONDITIONAL_FROM_OTHERS");
			BICONDITIONAL_FROM_OTHERS.setWorthwhileToTranslate(false);
			BICONDITIONAL_FROM_OTHERS.setUnimportant(ClassesSets.elementOf(A, BOOLEAN_SET), ClassesSets.elementOf(B, BOOLEAN_SET));
			BICONDITIONAL_FROM_OTHERS.setMain(
					Logic.equals(
							Arrays.asList(Logic.bracket(iff(A, B)), Logic.bracket(andWithBrackets(implies(A, B), implies(B, A))), notWithBrackets(xor(A, B)),
							orWithBrackets(and(not(A), not(B)), and(A, B)), nandWithBrackets(nand(A, B), nandWithBrackets(nand(A, A), nand(B, B)))

					)));
			// proof separate added below
			// a equi b = (a -> b) and (b -> a) = not (a xor b) = (not a and not b) or (a and b) = (a nand b) nand ((a nand a)nand(b nand b))
			lemma(BICONDITIONAL_FROM_OTHERS, "Biconditional by other operators", null, "Darstellung der Äquivalenz mittels anderer Operatoren",
					"Die Gleichheit von Äquivalenz und zwei Implikationen verwendet man in vielen Beweisen. Ist ", iff(A, B),
					" zu beweisen, beweist man zuerst (", implies(EMPTY, EMPTY), ") ", implies(A, B), ". In einem zweiten Schritt (", impliedBy(EMPTY, EMPTY),
					") beweist man ", impliedBy(A, B));


			BICONDITONAL_PROOF_BY_IMPLICATIONS = iffByIf(null, null, null, null);
			MathMLGenerator.displayRule(BICONDITONAL_PROOF_BY_IMPLICATIONS, WriteAfterStatementTransformation.ruleTransformation(Basics.MULTILINE, //
					new Object[] { Basics.BLANK_SEPARATED, "'", Bool.implies(Basics.EMPTY, Basics.EMPTY), "' ", Basics.defNames(Basics.TO_PROOF), " : ",
							RuleSelect.FIRST_VARIABLE }, //
					RuleSelect.SECOND_VARIABLE, //
					new Object[] { Basics.BLANK_SEPARATED, "'", Bool.impliedBy(Basics.EMPTY, Basics.EMPTY), "' ", Basics.defNames(Basics.TO_PROOF),
							" : ",
							RuleSelect.THIRD_VARIABLE }, //
					RuleSelect.FOURTH_VARIABLE) //
			);

			XOR_FROM_OTHERS = SINGLETON.naive("XOR_FROM_OTHERS");
			XOR_FROM_OTHERS.setWorthwhileToTranslate(false);
			XOR_FROM_OTHERS.setUnimportant(ClassesSets.elementOf(A, BOOLEAN_SET), ClassesSets.elementOf(B, BOOLEAN_SET));
			XOR_FROM_OTHERS.setMain(Logic.equals(Arrays.asList(xor(A, B), andWithBrackets(or(A, B), notWithBrackets(and(A, B))), notWithBrackets(iff(A, B)),
					nandWithBrackets(nand(A, Logic.bracket(nand(A, B))), nand(B, Logic.bracket(nand(A, B)))))));
			// a xor b = (a or b) and (not (a and b)) = not (a equi b) = ( a nand (a nand b)) nand (b nand (a nand b))
			lemma(XOR_FROM_OTHERS, "XOR  by other operators", null, "Darstellung von XOR mittels anderer Operatoren", null);

			CONDITIONAL_FROM_OTHERS = SINGLETON.naive("CONDITIONAL_FROM_OTHERS");
			CONDITIONAL_FROM_OTHERS.setWorthwhileToTranslate(false);
			CONDITIONAL_FROM_OTHERS.setUnimportant(ClassesSets.elementOf(A, BOOLEAN_SET), ClassesSets.elementOf(B, BOOLEAN_SET));
			CONDITIONAL_FROM_OTHERS.setMain(Logic.equals(Arrays.asList(Logic.bracket(implies(A, B)) //
					, or(not(A), B) //
					, nand(A, Logic.bracket(nand(B, B)))//
					, notWithBrackets(and(A, not(B))) //
					, implies(not(B), not(A)) //
			)));
			// a->b = ~a or b = a nand (b nand b) = a nand (a nand b) = not( a and not b)) = ~b -> ~a
			lemma(CONDITIONAL_FROM_OTHERS, "Implication by other operators", null, "Darstellung der Implikation mittels anderer Operatoren", null);

			NAND_FROM_OTHERS = SINGLETON.naive("NAND_FROM_OTHERS");
			NAND_FROM_OTHERS.setWorthwhileToTranslate(false);
			NAND_FROM_OTHERS.setUnimportant(ClassesSets.elementOf(A, BOOLEAN_SET), ClassesSets.elementOf(B, BOOLEAN_SET));
			NAND_FROM_OTHERS.setMain(Logic.equals(Arrays.asList(nand(A, B), notWithBrackets(and(A, B)), or(not(A), not(B)))));
			// a nand b = not (a and b)
			lemma(NAND_FROM_OTHERS, "NAND by other operators", null, "Darstellung von NAND mittels anderer Operatoren", null);

			
			AND_FROM_OTHERS = SINGLETON.naive("AND_FROM_OTHERS");
			AND_FROM_OTHERS.setWorthwhileToTranslate(false);
			AND_FROM_OTHERS.setUnimportant(ClassesSets.elementOf(A, BOOLEAN_SET), ClassesSets.elementOf(B, BOOLEAN_SET));
			AND_FROM_OTHERS.setMain(Logic.equals(Arrays.asList(and(A, B), nandWithBrackets(nand(A, B), nand(A, B)))));
			// a and b = (a nand b) nand (a nand b)
			lemma(AND_FROM_OTHERS, "AND by other operators", null, "Darstellung von AND mittels anderer Operatoren", null);

			OR_FROM_OTHERS = SINGLETON.naive("OR_FROM_OTHERS");
			OR_FROM_OTHERS.setWorthwhileToTranslate(false);
			OR_FROM_OTHERS.setUnimportant(ClassesSets.elementOf(A, BOOLEAN_SET), ClassesSets.elementOf(B, BOOLEAN_SET));
			OR_FROM_OTHERS.setMain(Logic.equals(Arrays.asList(or(A, B), nandWithBrackets(nand(A, A), nand(B, B)))));
			// a or b = (a nand a) nand (b nand b)
			lemma(OR_FROM_OTHERS, "OR by other operators", null, "Darstellung von OR mittels anderer Operatoren", null);

			MODUS_PONENS = SINGLETON.naive("MODUS_PONENS");
			MODUS_PONENS.setUnimportant(ClassesSets.elementOf(A, BOOLEAN_SET), ClassesSets.elementOf(B, BOOLEAN_SET));
			MODUS_PONENS.setMain(implies(Logic.bracket(and(Logic.bracket(implies(A, B)), A)), B));
			MODUS_PONENS.setReferences(References.wikiDe("Modus_ponens"), References.wikiEn("Modus_ponens"));
			// ((a->b) and a) -> b
			lemma(MODUS_PONENS, "Modus ponens", null, "Modus Ponens", null);

			MODUS_TOLLENS = SINGLETON.naive("MODUS_TOLLENS");
			MODUS_TOLLENS.setUnimportant(ClassesSets.elementOf(A, BOOLEAN_SET), ClassesSets.elementOf(B, BOOLEAN_SET));
			MODUS_TOLLENS.setMain(implies(Logic.bracket(and(Logic.bracket(implies(A, B)), not(B))), not(A)));
			MODUS_TOLLENS.setReferences(References.wikiDe("Modus_tollens"), References.wikiEn("Modus_tollens"));
			// ((a->b) and not b) -> not a
			lemma(MODUS_TOLLENS, "Modus tollens", null, "Modus Tollens", null);

			CONTRAPOSITION = SINGLETON.naive("CONTRAPOSITION");
			CONTRAPOSITION.setUnimportant(ClassesSets.elementOf(A, BOOLEAN_SET), ClassesSets.elementOf(B, BOOLEAN_SET));
			CONTRAPOSITION.setMain(iff(Logic.bracket(implies(A, B)), Logic.bracket(implies(not(B), not(A)))));
			CONTRAPOSITION.setReferences(References.wikiDe("Kontraposition"), References.wikiEn("Contraposition"));
			// (a->b) <-> (not b -> not a)
			lemma(CONTRAPOSITION, "contraposition", null, "Kontraposition", null);

			DISJUNCTIVE_SYLLOGISM = SINGLETON.naive("DISJUNCTIVE_SYLLOGISM");
			DISJUNCTIVE_SYLLOGISM.setUnimportant(ClassesSets.elementOf(A, BOOLEAN_SET), ClassesSets.elementOf(B, BOOLEAN_SET));
			DISJUNCTIVE_SYLLOGISM.setMain(implies(Logic.bracket(and(Logic.bracket(or(A, B)), not(B))), A));
			DISJUNCTIVE_SYLLOGISM.setReferences(References.wikiDe("Modus_tollendo_ponens"), References.wikiEn("Disjunctive_syllogism"));
			// ((a or b) and not b) -> a
			lemma(DISJUNCTIVE_SYLLOGISM, "Disjunctive syllogism", null, "Disjunktiver Syllogismus", null);

			DOUBLE_NEGATION = SINGLETON.naive("DOUBLE_NEGATION");
			DOUBLE_NEGATION.setUnimportant(ClassesSets.elementOf(A, BOOLEAN_SET));
			DOUBLE_NEGATION.setMain(Logic.equals(Arrays.asList(A, notWithBrackets(not(A)))));
			DOUBLE_NEGATION.setReferences(References.wikiEn("Double_negation"));
			// a <-> not not a
			lemma(DOUBLE_NEGATION, "Double negation", null, "Doppelte Verneinung",
					"Die doppelte Verneinung verwendet man bei einem Beweis durch Widerspruch (indirekter Beweis). Anstelle ", A, " zu beweisen, nimmt man ",
					not(A),
					" an und beweist daraus das Falsum ", FALSUM, " als Widerspruch. Also gilt ", Logic.equals(not(A), FALSUM), " und es ist ",
					Logic.equals(A, not(not(A)), not(FALSUM), VERUM));

			DE_MORGAN_1 = SINGLETON.naive("DE_MORGAN_1");
			DE_MORGAN_1.setWorthwhileToTranslate(false);
			DE_MORGAN_1.setUnimportant(ClassesSets.elementOf(A, BOOLEAN_SET), ClassesSets.elementOf(B, BOOLEAN_SET));
			DE_MORGAN_1.setMain(Logic.equals(Arrays.asList(notWithBrackets(or(A, B)), andWithBrackets(not(A), not(B)))));
			// not (a or b) = (not a) and (not b)
			lemma(DE_MORGAN_1, "De Morgan (rule 1)", null, "De Morgan (Regel 1)", null);

			DE_MORGAN_2 = SINGLETON.naive("DE_MORGAN_2");
			DE_MORGAN_2.setWorthwhileToTranslate(false);
			DE_MORGAN_2.setUnimportant(ClassesSets.elementOf(A, BOOLEAN_SET), ClassesSets.elementOf(B, BOOLEAN_SET));
			DE_MORGAN_2.setMain(Logic.equals(Arrays.asList(notWithBrackets(and(A, B)), orWithBrackets(not(A), not(B)))));
			DE_MORGAN_2.setReferences(References.wikiDe("De_Morgansche_Gesetze"), References.wikiEn("De_Morgan's_laws"));
			// not (a and b) = (not a) or (not b)
			lemma(DE_MORGAN_2, "De Morgan (rule 2)", null, "De Morgan (Regel 2)", null);


			RINGSCHLUSS = SINGLETON.naive("RINGSCHLUSS");
			RINGSCHLUSS.setWorthwhileToTranslate(false);
			RINGSCHLUSS.setUnimportant(ClassesSets.elementOf(A, BOOLEAN_SET), ClassesSets.elementOf(B, BOOLEAN_SET), ClassesSets.elementOf(C, BOOLEAN_SET));
			Statement ringschlussLeft = Logic.bracket(and(andWithBrackets(implies(A, B), implies(B, C)), Logic.bracket(implies(C, A))));
			RINGSCHLUSS.setMain(implies(ringschlussLeft, Logic.bracket(iff(A, C))));
			lemma(RINGSCHLUSS, null, null, "Ringschluss", null);
			RINGSCHLUSS.setProofs(iffByIf(//
					implies(ringschlussLeft, Logic.bracket(implies(A, C))), //
					Basics.proofline(implies(andWithBrackets(implies(A, B), implies(B, C)), Logic.bracket(implies(A, C))),
							Basics.defNames(TRANSITIVE_CONDITIONAL)), //
					implies(ringschlussLeft, Logic.bracket(impliedBy(A, C))), //
					Basics.TRIVIAL));

			DISTRIBUTIVE_1 = SINGLETON.naive("DISTRIBUTIVE_1");
			DISTRIBUTIVE_1.setWorthwhileToTranslate(false);
			DISTRIBUTIVE_1.setUnimportant(ClassesSets.elementOf(A, BOOLEAN_SET), ClassesSets.elementOf(B, BOOLEAN_SET), ClassesSets.elementOf(C, BOOLEAN_SET));
			DISTRIBUTIVE_1.setMain(Logic.equals(Arrays.asList(and(A, Logic.bracket(or(B, C))), orWithBrackets(and(A, B), and(A, C)))));
			// a and(b or c) = (a or b) and (a or c)
			lemma(DISTRIBUTIVE_1, "Distributive law (1)", null, "Distributivgesetz UND-ODER", null);

			DISTRIBUTIVE_2 = SINGLETON.naive("DISTRIBUTIVE_2");
			DISTRIBUTIVE_2.setWorthwhileToTranslate(false);
			DISTRIBUTIVE_2.setUnimportant(ClassesSets.elementOf(A, BOOLEAN_SET), ClassesSets.elementOf(B, BOOLEAN_SET), ClassesSets.elementOf(C, BOOLEAN_SET));
			DISTRIBUTIVE_2.setMain(Logic.equals(Arrays.asList(or(A, Logic.bracket(and(B, C))), andWithBrackets(or(A, B), or(A, C)))));
			// a or(b and c) = (a and b) or (a and c)
			lemma(DISTRIBUTIVE_2, "Distributive law (2)", null, "Distributivgesetz ODER-UND", null);
			
			
			IDEMPOTENT_AND = SINGLETON.naive("IDEMPOTENT_AND");
			IDEMPOTENT_AND.setWorthwhileToTranslate(false);
			IDEMPOTENT_AND.setUnimportant(ClassesSets.elementOf(A, BOOLEAN_SET));
			IDEMPOTENT_AND.setMain(Logic.equals(Arrays.asList(A, and(A, A))));
			// a = a and a
			lemma(IDEMPOTENT_AND, "Idempotent (AND)", null, "Idempotenz (AND)", null);

			IDEMPOTENT_OR = SINGLETON.naive("IDEMPOTENT_OR");
			IDEMPOTENT_OR.setWorthwhileToTranslate(false);
			IDEMPOTENT_OR.setUnimportant(ClassesSets.elementOf(A, BOOLEAN_SET));
			IDEMPOTENT_OR.setMain(Logic.equals(Arrays.asList(A, or(A, A))));
			// a = a or a
			lemma(IDEMPOTENT_OR, "Idempotent (OR)", null, "Idempotenz (OR)", null);

			TERTIUM_NON_DATUR = SINGLETON.naive("TERTIUM_NON_DATUR");
			TERTIUM_NON_DATUR.setWorthwhileToTranslate(false);
			TERTIUM_NON_DATUR.setUnimportant(ClassesSets.elementOf(A, BOOLEAN_SET));
			TERTIUM_NON_DATUR.setMain(Logic.equals(Arrays.asList(or(A, not(A)), VERUM)));
			TERTIUM_NON_DATUR.setReferences(References.wikiDe("Satz_vom_ausgeschlossenen_Dritten"), References.wikiEn("Law_of_excluded_middle"));
			// a or not a = true
			lemma(TERTIUM_NON_DATUR, "Law of excluded middle", null, "Satz vom ausgeschlossenen Dritten (tertium non datur)", null);

			NONCONTRADICTION = SINGLETON.naive("NONCONTRADICTION");
			NONCONTRADICTION.setWorthwhileToTranslate(false);
			NONCONTRADICTION.setUnimportant(ClassesSets.elementOf(A, BOOLEAN_SET));
			NONCONTRADICTION.setMain(Logic.equals(Arrays.asList(and(A, not(A)), FALSUM)));
			NONCONTRADICTION.setReferences(References.wikiDe("Satz_vom_Widerspruch"), References.wikiEn("Law_of_noncontradiction"));
			// a and not a = false
			lemma(NONCONTRADICTION, "Law of noncontradiction", null, "Satz vom Widerspruch", null);

			AND_MULTIPLE = and();
			MathMLGenerator.displayRule(AND_MULTIPLE, WRITE_OPERATION_AND);

			// TODO define correctly with associative and induction

			SPECIALIZATION = SINGLETON.naive("SPECIALIZATION");
			SPECIALIZATION.setUnimportant(ClassesSets.elementOf(A, BOOLEAN_SET), ClassesSets.elementOf(B, BOOLEAN_SET));
			SPECIALIZATION.setMain(implies(and(A, B), A));
			lemma(SPECIALIZATION, "specialization", null, "Spezialisierung", null);

			GENERALIZATION = SINGLETON.naive("GENERALIZATION");
			GENERALIZATION.setUnimportant(ClassesSets.elementOf(A, BOOLEAN_SET), ClassesSets.elementOf(B, BOOLEAN_SET));
			GENERALIZATION.setMain(implies(A, or(A, B)));
			lemma(GENERALIZATION, "generalization", null, "Generalisierung", null);

			BOOL_XOR = TuplesFactory.pair(BOOLEAN_SET, XOR_RELATION);
			BOOL_AND = TuplesFactory.pair(BOOLEAN_SET, AND_RELATION);

		} catch (Exception ex) {
			throw ThrowableBoostUtils.toRuntimeException(ex);
		}

	}


	public static StudyUnit createStudyUnit1() {

		return new StudyUnit(SINGLETON, 1,
				Arrays.asList(VERUM, FALSUM, BOOLEAN_SET, NOT_RELATION, //
						AND_RELATION, AND_COMMUTATIVE, AND_IDEMPOTENT, AND_ASSOCIATIVE, //
						OR_RELATION, OR_COMMUTATIVE, OR_IDEMPOTENT, OR_ASSOCIATIVE, //
						CONDITIONAL_RELATION, TRANSITIVE_CONDITIONAL, //
						BICONDITIONAL_RELATION, XOR_RELATION, NAND_RELATION, NOR_RELATION, NOT_FROM_OTHERS, BICONDITIONAL_FROM_OTHERS, XOR_FROM_OTHERS,
						CONDITIONAL_FROM_OTHERS, NAND_FROM_OTHERS, AND_FROM_OTHERS, OR_FROM_OTHERS, MODUS_PONENS, MODUS_TOLLENS, CONTRAPOSITION,
						DISJUNCTIVE_SYLLOGISM,
						DOUBLE_NEGATION, DE_MORGAN_1, DE_MORGAN_2,
						RINGSCHLUSS, DISTRIBUTIVE_1, DISTRIBUTIVE_2, IDEMPOTENT_AND, IDEMPOTENT_OR, TERTIUM_NON_DATUR, NONCONTRADICTION, SPECIALIZATION,
						GENERALIZATION),
				Arrays.asList(BICONDITONAL_PROOF_BY_IMPLICATIONS, AND_MULTIPLE, IMPLIED_BY_NAIVE, DEFINED_AS_EQUAL_TO, CONDITIONAL_MULTILINE, CONDITIONAL_MULTI,
						BICONDITIONAL_MULTILINE, AND_MULTILINE, DEFINED_AS_EQUAL_TO_MULTILINE));
	}

	private Bool() {
	}


	public static void addProofs() throws Exception {
		// Statement varA = Logic.var("A");
		// Statement varB = Logic.var("B");
		// Statement varC = Logic.var("C");

		Statement notA = not(A);
		Statement notB = not(B);

		Statement aAndB = and(A, B);
		Statement aAndNotB = and(A, notB);
		Statement bAndC = and(B, C);

		Statement aBicondidtionalB = iff(A, B);

		Statement aImpliesB = implies(A, B);
		Statement aImpliesC = implies(A, C);
		Statement bImpliesA = implies(B, A);
		Statement bImpliesC = implies(B, C);

		Statement aImpliesB_AND_bImpliesA = andWithBrackets(aImpliesB, bImpliesA);
		Statement aImpliesB_AND_bImpliesC = andWithBrackets(aImpliesB, bImpliesC);

		Statement aNandA = nand(A, A);
		Statement aNandB = nand(A, B);

		Statement not_AandB = not(Logic.bracket(aAndB));
		Statement notAandNotB = and(notA, notB);

		Statement aOrB = or(A, B);
		Statement bOrC = or(B, C);

		Statement aXorB = xor(A, B);

		Statement bNandB = nand(B, B);

		Statement transitiveImplication = impliesWithBrackets(aImpliesB_AND_bImpliesC, aImpliesC);


		AND_ASSOCIATIVE.setProofs(new PrologProofGenerator().displayValueTable(MathStudiesProofs.getStatementsUpTo(Bool.AND_ASSOCIATIVE), new Statement[] {},
				A, B, C, aAndB, and(Logic.bracket(aAndB), C), bAndC, and(A, Logic.bracket(bAndC))));

		OR_ASSOCIATIVE.setProofs(new PrologProofGenerator().displayValueTable(MathStudiesProofs.getStatementsUpTo(Bool.OR_ASSOCIATIVE), new Statement[] {},
				A, B, C, aOrB, or(Logic.bracket(aOrB), C), bOrC, or(A, Logic.bracket(bOrC))));

		TRANSITIVE_CONDITIONAL.setProofs(new PrologProofGenerator().displayValueTable(MathStudiesProofs.getStatementsUpTo(Bool.TRANSITIVE_CONDITIONAL),
				new Statement[] {}, A, B, C, aImpliesB, bImpliesC, aImpliesB_AND_bImpliesC, aImpliesC, transitiveImplication));

		NOT_FROM_OTHERS.setProofs(new PrologProofGenerator().displayValueTable(MathStudiesProofs.getStatementsUpTo(Bool.NOT_FROM_OTHERS), new Statement[] {},
				A, A, notA, aNandA));

		BICONDITIONAL_FROM_OTHERS.setProofs(new PrologProofGenerator().displayValueTable(MathStudiesProofs.getStatementsUpTo(Bool.BICONDITIONAL_FROM_OTHERS),
				new Statement[] {}, A, B, aBicondidtionalB, aImpliesB, bImpliesA, aImpliesB_AND_bImpliesA, aXorB, notAandNotB, aAndB));

		XOR_FROM_OTHERS.setProofs(new PrologProofGenerator().displayValueTable(MathStudiesProofs.getStatementsUpTo(Bool.XOR_FROM_OTHERS), new Statement[] {},
				A, B, aXorB, aOrB, not_AandB, aBicondidtionalB));

		CONDITIONAL_FROM_OTHERS.setProofs(new PrologProofGenerator().displayValueTable(MathStudiesProofs.getStatementsUpTo(Bool.CONDITIONAL_FROM_OTHERS),
				new Statement[] {}, A, B, aImpliesB, notA, B, A, bNandB, aAndNotB, notB, notA));

		NAND_FROM_OTHERS.setProofs(new PrologProofGenerator().displayValueTable(MathStudiesProofs.getStatementsUpTo(Bool.NAND_FROM_OTHERS),
				new Statement[] {}, A, B, aNandB, aAndB, notA, notB));

		AND_FROM_OTHERS.setProofs(new PrologProofGenerator().displayValueTable(MathStudiesProofs.getStatementsUpTo(Bool.NAND_FROM_OTHERS),
				new Statement[] {}, A, B, aAndB, aNandB, aNandB));

		OR_FROM_OTHERS.setProofs(new PrologProofGenerator().displayValueTable(MathStudiesProofs.getStatementsUpTo(Bool.NAND_FROM_OTHERS), new Statement[] {},
				A, B, aOrB, aNandA, bNandB));

		MODUS_PONENS.setProofs(new PrologProofGenerator().displayValueTable(MathStudiesProofs.getStatementsUpTo(Bool.MODUS_PONENS), new Statement[] {}, A, B,
				aImpliesB, Bool.and(Logic.bracket(aImpliesB), A), Bool.implies(Bool.andWithBrackets(Logic.bracket(aImpliesB), A), B)));

		MODUS_TOLLENS.setProofs(new PrologProofGenerator().displayValueTable(MathStudiesProofs.getStatementsUpTo(Bool.MODUS_TOLLENS), new Statement[] {}, A,
				B, notB, aImpliesB, Bool.and(Logic.bracket(aImpliesB), notB),
				Bool.implies(Bool.andWithBrackets(Logic.bracket(aImpliesB), notB), notA)));

		CONTRAPOSITION.setProofs(new PrologProofGenerator().displayValueTable(MathStudiesProofs.getStatementsUpTo(Bool.CONTRAPOSITION), new Statement[] {}, A,
				B, aImpliesB, notB, notA, Bool.implies(notB, notA)));

		DISJUNCTIVE_SYLLOGISM
				.setProofs(new PrologProofGenerator().displayValueTable(MathStudiesProofs.getStatementsUpTo(Bool.DISJUNCTIVE_SYLLOGISM), new Statement[] {},
						A, B, notB, aOrB, Bool.and(Logic.bracket(aOrB), notB), Bool.implies(Bool.andWithBrackets(Logic.bracket(aOrB), notB), A)));

		DOUBLE_NEGATION.setProofs(new PrologProofGenerator().displayValueTable(MathStudiesProofs.getStatementsUpTo(Bool.DOUBLE_NEGATION), new Statement[] {},
				A, notA, not(notA)));

		DE_MORGAN_1.setProofs(new PrologProofGenerator().displayValueTable(MathStudiesProofs.getStatementsUpTo(Bool.DE_MORGAN_1), new Statement[] {}, A, B,
				notA, notB, and(notA, notB), aOrB));

		DE_MORGAN_2.setProofs(new PrologProofGenerator().displayValueTable(MathStudiesProofs.getStatementsUpTo(Bool.DE_MORGAN_2), new Statement[] {}, A, B,
				notA, notB, or(notA, notB), aAndB));

		DISTRIBUTIVE_1.setProofs(new PrologProofGenerator().displayValueTable(MathStudiesProofs.getStatementsUpTo(Bool.DISTRIBUTIVE_1), new Statement[] {},
				A, B, C, or(B, C), and(A, or(B, C)), and(A, B), and(A, C), orWithBrackets(and(A, B), and(A, C))));

		DISTRIBUTIVE_2.setProofs(new PrologProofGenerator().displayValueTable(MathStudiesProofs.getStatementsUpTo(Bool.DISTRIBUTIVE_2), new Statement[] {},
				A, B, C, and(B, C), or(A, and(B, C)), or(A, B), or(A, C), andWithBrackets(or(A, B), or(A, C))));

		IDEMPOTENT_AND.setProofs(
				new PrologProofGenerator().displayValueTable(MathStudiesProofs.getStatementsUpTo(Bool.IDEMPOTENT_AND), new Statement[] {}, A, A, and(A, A)));

		IDEMPOTENT_OR.setProofs(
				new PrologProofGenerator().displayValueTable(MathStudiesProofs.getStatementsUpTo(Bool.IDEMPOTENT_OR), new Statement[] {}, A, A, or(A, A)));

		TERTIUM_NON_DATUR.setProofs(new PrologProofGenerator().displayValueTable(MathStudiesProofs.getStatementsUpTo(Bool.TERTIUM_NON_DATUR),
				new Statement[] {}, A, notA, or(A, notA)));

		NONCONTRADICTION.setProofs(new PrologProofGenerator().displayValueTable(MathStudiesProofs.getStatementsUpTo(Bool.NONCONTRADICTION),
				new Statement[] {}, A, notA, and(A, notA)));

		SPECIALIZATION.setProofs(new PrologProofGenerator().displayValueTable(MathStudiesProofs.getStatementsUpTo(Bool.SPECIALIZATION), new Statement[] {},
				A, B, and(A, B), implies(and(A, B), A)));

		GENERALIZATION.setProofs(new PrologProofGenerator().displayValueTable(MathStudiesProofs.getStatementsUpTo(Bool.SPECIALIZATION), new Statement[] {},
				A, B, or(A, B), implies(A, or(A, B))));
	}



}
